/* XmpListener.java

   COPYRIGHT 2013 KRUPCZAK.ORG, LLC.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
   USA
 
   For more information, visit:
   http://www.krupczak.org/
*/

package org.krupczak.xmp;

import java.net.*;
import java.io.*;
import java.util.Date;
import java.io.UnsupportedEncodingException;
import javax.net.*;
import java.security.*;
import java.security.cert.*;
import javax.net.ssl.*;
import java.util.Scanner;

/**
 * XmpListener is used to establish an Xmp protocol listener
 * for use with Xmp clients and servers (agents).  Instantiating
 * an object causes an Xmp listener to be created waiting to accept
 * Xmp protocol connections.  Use this object for receiving Traps or for
 * writing mock agents for testing purposes.
 * @author Bobby Krupczak, rdk@krupczak.org
 * @version $Id$
 * @see Xmp
 * @see XmpVar
 * @see XmpMessage
 **/

public class XmpListener {

  // default server port
  public static final int defaultServerPort = Xmp.XMP_PORT;
  public static final int STATE_CLOSED = 0;
  public static final int STATE_LISTEN = 1;

  // instance variables *****************************************
  public String serverAddr = "*";
  public int serverPort = 0;
  SocketOpts sockopts;
  SSLServerSocket listenSocket;
  int state;
  int totalConnections;
  int totalErrors;

  /* constructors */

  // Constructor takes default addr and XMP port

  // default constructor called by others
  public XmpListener(SocketOpts sockopts)
  {
      this.sockopts = sockopts;

      serverPort = defaultServerPort;
      state = STATE_CLOSED;
      totalConnections = 0;
      totalErrors = 0;
  }

  // user decided addr and port to bind to and listen on
  public XmpListener(SocketOpts sockopts, String addr, int port)
  {
      this(sockopts);

      serverAddr = addr;
      serverPort = port;

      try {
          // for now, we're ignoring addr XXX
          ServerSocket ss;

          ss = sockopts.sslServerSocketFactory.createServerSocket(serverPort);

          listenSocket = (SSLServerSocket)ss;

          listenSocket.setReuseAddress(sockopts.getReuseAddr());
          listenSocket.setEnableSessionCreation(true);
          state = STATE_LISTEN;

      } catch (IOException e) {
          state = STATE_CLOSED;
          System.out.println("XmpListener: failed "+e.getMessage());
      }
  }

  // pass an IP address/port to bind to and listen on 
  public XmpListener(SocketOpts sockopts, InetAddress addr, int port)
  {
      this(sockopts,addr.toString(),port);
  }

  /* private methods **************************** */


  /* public methods ***************************** */

  // get the total number connections this listener has 
  // accepted
  int getNumberConnections() { return totalConnections; }
  int getNumberErrors() { return totalErrors; }

  // handle the SSL handshake?
  // any reading/writing to/from the socket will initiate
  // and complete SSL handshake; consequently, do we need
  // to get involved in this?

  // also, because our trust store has our CA cert in it,
  // we believe the handshake will validate the client cert; 
  // we would like to confirm this XXX

  // listen for a connection, accept it, and then return
  // the new XmpSession; caller is responsible for
  // threading model -- that is, they hand session to 
  // thread or they handle the connection/session sequentially

  public XmpSession listenForConnection()
  {
      SSLSocket sessionSocket;
      XmpSession aSession;
      InetSocketAddress remoteSocketAddress;

      if (listenSocket == null)
	 return null;

      try {
 
	  if ((sessionSocket = (SSLSocket)listenSocket.accept()) == null) {
              totalErrors++;
	      return null;             
          }
      }
      catch (IOException e) {
	  totalErrors++;
          System.out.println("IOException during accept "+e.getMessage());
          return null;
      }

      totalConnections++;

      // before we start handshake, see who is connecting to us
      if ((remoteSocketAddress = (InetSocketAddress)sessionSocket.getRemoteSocketAddress()) == null) {
          System.out.println("listenForConnection: unable to get remote address");
          try {
            sessionSocket.close();
	  }
	  catch (IOException e) { totalErrors++; }
          totalErrors++;
          return null;
      }

      // start the synchronous handshake; when call returns
      // handshake is complete and we should check results
      try { 

          sessionSocket.startHandshake(); 

          // handshake complete; verify client/cert XXX

      } catch (IOException e) {

           totalErrors++;
           System.out.println("IOException during handshake from "+
                               remoteSocketAddress+" "
                               +e.getMessage());
           e.printStackTrace();
           try { 
              sessionSocket.close(); 
           } 
           catch (IOException e1) { totalErrors++; }
           return null;
      }

      //System.out.println("listenForConnection: returning connected socket");
      //System.out.println("listenForConnection: socket "+sessionSocket);

      aSession = new XmpSession(sockopts,sessionSocket,
                                remoteSocketAddress.getAddress(),
                                remoteSocketAddress.getPort());

      return aSession;
  }

  // get ip version of listening socket
  public int getIpVersion()
  {
      InetAddress localAddress;

      // given socket, get local address and determine
      // if its ipv4 or ipv6 (we know its always an InetAddress 
      if ((localAddress = listenSocket.getInetAddress()) == null)
	  return 0;

      if (localAddress instanceof Inet4Address)
	  return 4;
      if (localAddress instanceof Inet6Address)
	  return 6;
      return 0;
  }

  public boolean getWantClientAuth() 
  { 
       if (listenSocket != null)
          return listenSocket.getWantClientAuth(); 
       else 
	  return false;
  }
  public void setWantClientAuth(boolean want) 
  {
      if (listenSocket != null)
          listenSocket.setWantClientAuth(want);
  }

  public boolean getNeedClientAuth()
  {
       if (listenSocket != null)
          return listenSocket.getNeedClientAuth(); 
       else 
	  return false;
  }

  public void setNeedClientAuth(boolean need) 
  {
      if (listenSocket != null)
          listenSocket.setNeedClientAuth(need);
  }

} /* class XmpListener */
