/*******************************************************************************
 * This file is part of OpenNMS(R).
 *
 * Copyright (C) 2006-2012 The OpenNMS Group, Inc.
 * OpenNMS(R) is Copyright (C) 1999-2012 The OpenNMS Group, Inc.
 *
 * OpenNMS(R) is a registered trademark of The OpenNMS Group, Inc.
 *
 * OpenNMS(R) is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation, either version 3 of the License,
 * or (at your option) any later version.
 *
 * OpenNMS(R) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenNMS(R).  If not, see:
 *      http://www.gnu.org/licenses/
 *
 * For more information contact:
 *     OpenNMS(R) Licensing <license@opennms.org>
 *     http://www.opennms.org/
 *     http://www.opennms.com/
 *******************************************************************************/

package org.opennms.netmgt.capsd;

import static org.opennms.core.utils.InetAddressUtils.addr;
import static org.opennms.core.utils.InetAddressUtils.str;

import java.net.InetAddress;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.opennms.core.db.DataSourceFactory;
import org.opennms.core.utils.ConfigFileConstants;
import org.opennms.core.utils.DBUtils;
import org.opennms.netmgt.EventConstants;
import org.opennms.netmgt.capsd.IfCollector.SupportedProtocol;
import org.opennms.netmgt.capsd.snmp.IfTable;
import org.opennms.netmgt.capsd.snmp.IfTableEntry;
import org.opennms.netmgt.capsd.snmp.IfXTableEntry;
import org.opennms.netmgt.capsd.snmp.IpAddrTable;
import org.opennms.netmgt.capsd.snmp.SystemGroup;
import org.opennms.netmgt.config.CapsdConfig;
import org.opennms.netmgt.config.CapsdConfigFactory;
import org.opennms.netmgt.config.PollerConfig;
import org.opennms.netmgt.config.PollerConfigFactory;
import org.opennms.netmgt.eventd.EventIpcManagerFactory;
import org.opennms.netmgt.filter.FilterDaoFactory;
import org.opennms.netmgt.model.OnmsNode.NodeLabelSource;
import org.opennms.netmgt.model.OnmsNode.NodeType;
import org.opennms.netmgt.model.events.EventBuilder;
import org.opennms.netmgt.xml.event.Event;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class is designed to rescan all the managed interfaces for a specified
 * node, update the database based on the information collected, and generate
 * events necessary to notify the other OpenNMS services. The constructor takes
 * an integer which is the node identifier of the node to be rescanned. .
 *
 * @author <a href="mailto:brozow@opennms.org">Matt Brozowski</a>
 * @author <a href="mailto:jamesz@opennms.org">James Zuo </a>
 * @author <a href="mailto:mike@opennms.org">Mike Davidson </a>
 * @author <a href="mailto:weave@oculan.com">Brian Weaver </a>
 * @author <a href="http://www.opennms.org/">OpenNMS </a>
 */
public final class RescanProcessor implements Runnable {

    private static final Logger LOG = LoggerFactory.getLogger(RescanProcessor.class);

    private static final String s_ZERO_ZERO_ZERO_ZERO = "0.0.0.0";
    private static final InetAddress ZERO_ZERO_ZERO_ZERO = addr(s_ZERO_ZERO_ZERO_ZERO);

    /**
     * SQL statement for retrieving the 'nodetype' field of the node table for
     * the specified nodeid. Used to determine if the node is active ('A') or
     * been marked as deleted ('D').
     */
    static final String SQL_DB_RETRIEVE_NODE_TYPE = "SELECT nodetype FROM node WHERE nodeID=?";

    /**
     * SQL statement used to retrieve other nodeIds that have the same
     * ipinterface as the updating node.
     */
    static final String SQL_DB_RETRIEVE_OTHER_NODES = "SELECT ipinterface.nodeid FROM ipinterface, node WHERE ipinterface.ismanaged != 'D' AND ipinterface.ipaddr = ? AND ipinterface.nodeid = node.nodeid AND node.foreignsource IS NULL AND ipinterface.nodeid !=? ";

    /**
     * SQL statements used to reparent an interface and its associated services
     * under a new parent nodeid
     */
    static final String SQL_DB_REPARENT_IP_INTERFACE_LOOKUP = "SELECT ipaddr, ifindex FROM ipinterface " + "WHERE nodeID=? AND ipaddr=? AND isManaged!='D'";

    static final String SQL_DB_REPARENT_IP_INTERFACE_DELETE = "DELETE FROM ipinterface " + "WHERE nodeID=? AND ipaddr=?";

    static final String SQL_DB_REPARENT_IP_INTERFACE = "UPDATE ipinterface SET nodeID=? WHERE nodeID=? AND ipaddr=? AND isManaged!='D'";

    static final String SQL_DB_REPARENT_SNMP_IF_LOOKUP = "SELECT id FROM snmpinterface WHERE nodeID=? AND snmpifindex=?";

    static final String SQL_DB_REPARENT_SNMP_IF_DELETE = "DELETE FROM snmpinterface " + "WHERE nodeID=? AND snmpifindex=?";

    static final String SQL_DB_REPARENT_SNMP_INTERFACE = "UPDATE snmpinterface SET nodeID=? WHERE nodeID=? AND snmpifindex=?";

    static final String SQL_DB_REPARENT_IF_SERVICES_LOOKUP = "SELECT serviceid FROM ifservices " + "WHERE nodeID=? AND ipaddr=? AND ifindex = ? " + "AND status!='D'";

    static final String SQL_DB_REPARENT_IF_SERVICES_DELETE = "DELETE FROM ifservices WHERE nodeID=? AND ipaddr=? ";

    static final String SQL_DB_REPARENT_IF_SERVICES = "UPDATE ifservices SET nodeID=? WHERE nodeID=? AND ipaddr=? AND status!='D'";

    /**
     * SQL statements used to clear up ipinterface table, ifservices table and
     * snmpinterface table when delete a duplicate node.
     */
    static final String SQL_DB_DELETE_DUP_INTERFACE = "DELETE FROM ipinterface WHERE nodeID=?";

    static final String SQL_DB_DELETE_DUP_SERVICES = "DELETE FROM ifservices WHERE nodeid=?";

    static final String SQL_DB_DELETE_DUP_SNMPINTERFACE = "DELETE FROM snmpinterface WHERE nodeid =?";

    /**
     * SQL statement used to retrieve service IDs so that service name can be
     * determined from ID, and map of service names
     */
    private static final String SQL_RETRIEVE_SERVICE_IDS = "SELECT serviceid,servicename  FROM service";

    /**
     * SQL statement used to update ipinterface.ismanaged
     * when rescan discovers new interface
     */
    private static final String SQL_DB_UPDATE_ISMANAGED = "UPDATE ipinterface SET ismanaged=? WHERE nodeID=? AND ipaddr=? AND isManaged!='D'";

    /**
     * Indicates if the rescan is in response to a forceRescan event.
     */
    private boolean m_forceRescan;

    /**
     * Event list...during the rescan significant database changes cause events
     * (interfaceReparented, nodeGainedService, and others) to be created and
     * added to the event list. The last thing the rescan process does is send
     * the events out.
     */
    private final List<Event> m_eventList = new ArrayList<Event>();

    /**
     * Set during the rescan to true if any of the ifIndex values associated
     * with a node's interface's were modified as a result of the scan.
     */
    private boolean m_ifIndexOnNodeChangedFlag;

    /**
     * Set during the rescan to true if a new interface is added to the
     * snmpInterfaces table or if any key fields of the node's snmpIntefaces
     * table were modified (such as ifIndex or ifType)
     */
    private boolean m_snmpIfTableChangedFlag;

    private CapsdDbSyncer m_capsdDbSyncer;

    private PluginManager m_pluginManager;

    private int m_nodeId;

    private static Set<Integer> s_queuedRescanTracker;

    /**
     * Constructor.
     * 
     * @param nodeInfo
     *            Scheduler.NodeInfo object containing the nodeid of the node to
     *            be rescanned.
     * @param forceRescan
     *            True if a forced rescan is to be performed (all interfaces not
     *            just managed interfaces scanned), false otherwise.
     */
    RescanProcessor(final Scheduler.NodeInfo nodeInfo, final boolean forceRescan, final CapsdDbSyncer capsdDbSyncer, final PluginManager pluginManager) {
        this(nodeInfo.getNodeId(), forceRescan, capsdDbSyncer, pluginManager);
    }

    /**
     * <p>Constructor for RescanProcessor.</p>
     *
     * @param nodeId a int.
     * @param forceRescan a boolean.
     * @param capsdDbSyncer a {@link org.opennms.netmgt.capsd.CapsdDbSyncer} object.
     * @param pluginManager a {@link org.opennms.netmgt.capsd.PluginManager} object.
     */
    public RescanProcessor(final int nodeId, final boolean forceRescan, final CapsdDbSyncer capsdDbSyncer, final PluginManager pluginManager) {
        m_nodeId = nodeId;
        m_forceRescan = forceRescan;
        m_capsdDbSyncer = capsdDbSyncer;
        m_pluginManager = pluginManager;

        // Add the node ID of the node to be rescanned to the Set that tracks
        // rescan requests
        synchronized (s_queuedRescanTracker) {
            s_queuedRescanTracker.add(nodeId);
        }
    }

    /**
     * This method is responsible for updating the node table using the most
     * recent data collected from the node's managed interfaces.
     * 
     * @param dbc
     *            Database connection
     * @param now
     *            Date object representing the time of the rescan.
     * @param dbNodeEntry
     *            DbNodeEntry object representing existing values in the
     *            database
     * @param currPrimarySnmpIf
     *            Primary SNMP interface address based on latest collection
     * @param dbIpInterfaces
     *            Array of DbIpInterfaceEntry objects representing all the
     *            interfaces retrieved from the database for this node.
     * @param collectorMap
     *            Map of IfCollector objects...one per managed interface.
     * 
     * @return DbNodeEntry object representing the updated values from the node
     *         table in the database.
     * 
     * @throws SQLException
     *             if there is a problem updating the node table.
     */
    private DbNodeEntry updateNode(final Connection dbc, final Date now, final DbNodeEntry dbNodeEntry, final InetAddress currPrimarySnmpIf, final DbIpInterfaceEntry[] dbIpInterfaces, final Map<String, IfCollector> collectorMap) throws SQLException {
        LOG.debug("updateNode: updating node id {}", dbNodeEntry.getNodeId());

        /*
         * Clone the existing dbNodeEntry so we have all the original
         * values of the 'node' table fields in case we need to generate
         * 'node***Changed' events following the update.
         */
        final DbNodeEntry originalDbNodeEntry = DbNodeEntry.clone(dbNodeEntry);

        /*
         * Create node which represents the most recently retrieved
         * information in the collector for this node
         */
        final DbNodeEntry currNodeEntry = DbNodeEntry.create();
        currNodeEntry.setNodeType(NodeType.ACTIVE);

        // Set node label and SMB info based on latest collection
        setNodeLabelAndSmbInfo(collectorMap, dbNodeEntry, currNodeEntry, currPrimarySnmpIf);

        // Set SNMP info
        if (currPrimarySnmpIf != null) {
            /*
             * We prefer to use the collector for the primary SNMP interface
             * to update SNMP data in the node table. However a collector
             * for the primary SNMP interface may not exist in the map if
             * a node has only recently had SNMP support enabled or if the
             * new primary SNMP interface was only recently added to the
             * node. At any rate if it exists use it, if not use the
             * first collector which supports SNMP.
             */
            final String addr = str(currPrimarySnmpIf);
            IfCollector primaryIfc = addr == null? null : collectorMap.get(addr);
            if (primaryIfc == null) {
                for (IfCollector tmp : collectorMap.values()) {
                    if (tmp.getSnmpCollector() != null) {
                        primaryIfc = tmp;
                        break;
                    }
                }
            }

            /*
             * Sanity check...should always have a primary interface
             * collector at this point
             */
            if (primaryIfc == null) {
                LOG.error("updateNode: failed to determine primary interface collector for node {}", dbNodeEntry.getNodeId());
                throw new RuntimeException("Update node failed for node " + dbNodeEntry.getNodeId() + ", unable to determine primary interface collector.");
            }

            final IfSnmpCollector snmpc = primaryIfc.getSnmpCollector();
            if (snmpc != null && snmpc.hasSystemGroup()) {
                SystemGroup sysgrp = snmpc.getSystemGroup();

                // sysObjectId
                currNodeEntry.setSystemOID(sysgrp.getSysObjectID());

                // sysName
                final String sysName = sysgrp.getSysName();
                if (sysName != null && sysName.length() > 0) {
                    currNodeEntry.setSystemName(sysName);
                }

                // sysDescription
                final String sysDescr = sysgrp.getSysDescr();
                if (sysDescr != null && sysDescr.length() > 0) {
                    currNodeEntry.setSystemDescription(sysDescr);
                }

                // sysLocation
                final String sysLocation = sysgrp.getSysLocation();
                if (sysLocation != null && sysLocation.length() > 0) {
                    currNodeEntry.setSystemLocation(sysLocation);
                }

                // sysContact
                final String sysContact = sysgrp.getSysContact();
                if (sysContact != null && sysContact.length() > 0) {
                    currNodeEntry.setSystemContact(sysContact);
                }
            }
        }

        /*
         * Currently, we do not use the ParentId except in mapping.
         * Unfortunately, this is never
         * set in the currNodeEntry so it gets reset here. As a workaround,
         * setting it to the old value.
         */

        currNodeEntry.updateParentId(dbNodeEntry.getParentId());

        // Update any fields which have changed
        LOG.debug("updateNode: -------dumping old node-------: {}", dbNodeEntry);
        LOG.debug("updateNode: -------dumping new node-------: {}", currNodeEntry);
        dbNodeEntry.updateParentId(currNodeEntry.getParentId());
        dbNodeEntry.updateNodeType(currNodeEntry.getNodeType());
        dbNodeEntry.updateSystemOID(currNodeEntry.getSystemOID());
        dbNodeEntry.updateSystemName(currNodeEntry.getSystemName());
        dbNodeEntry.updateSystemDescription(currNodeEntry.getSystemDescription());
        dbNodeEntry.updateSystemLocation(currNodeEntry.getSystemLocation());
        dbNodeEntry.updateSystemContact(currNodeEntry.getSystemContact());
        dbNodeEntry.updateNetBIOSName(currNodeEntry.getNetBIOSName());
        dbNodeEntry.updateDomainName(currNodeEntry.getDomainName());
        dbNodeEntry.updateOS(currNodeEntry.getOS());
        dbNodeEntry.setLastPoll(now);

        /*
         * Only update node label/source if original node entry is
         * not set to user-defined.
         */
        if (dbNodeEntry.getLabelSource() != NodeLabelSource.USER) {
            dbNodeEntry.updateLabel(currNodeEntry.getLabel());
            dbNodeEntry.updateLabelSource(currNodeEntry.getLabelSource());
        }

        // Set event flags
        final boolean nodeLabelChangedFlag = dbNodeEntry.hasLabelChanged() || dbNodeEntry.hasLabelSourceChanged();
        final boolean nodeInfoChangedFlag = dbNodeEntry.hasSystemOIDChanged() || dbNodeEntry.hasSystemNameChanged() || dbNodeEntry.hasSystemDescriptionChanged() || dbNodeEntry.hasSystemLocationChanged() || dbNodeEntry.hasSystemContactChanged() || dbNodeEntry.hasNetBIOSNameChanged() || dbNodeEntry.hasDomainNameChanged() || dbNodeEntry.hasOSChanged();

        // Call store to update the database
        dbNodeEntry.store(dbc);

        // Create nodeLabelChanged event if necessary
        if (nodeLabelChangedFlag) {
            m_eventList.add(createNodeLabelChangedEvent(dbNodeEntry, originalDbNodeEntry));
        }

        // Create nodeInfoChangedEvent if necessary
        if (nodeInfoChangedFlag) {
            m_eventList.add(createNodeInfoChangedEvent(dbNodeEntry, originalDbNodeEntry));
        }

        return dbNodeEntry;
    }

    /**
     * This method is responsible for updating all of the interface's associated
     * with a node.
     * 
     * @param dbc
     *            Database connection.
     * @param now
     *            Date/time to be associated with the update.
     * @param node
     *            Node entry for the node being rescanned
     * @param collectorMap
     *            Map of IfCollector objects associated with the node.
     * @param doesSnmp
     *            Indicates that the interface does support SNMP
     * 
     * @throws SQLException
     *             if there is a problem updating the ipInterface table.
     */
    private void updateInterfaces(final Connection dbc, final Date now, final DbNodeEntry node, final Map<String, IfCollector> collectorMap, final boolean doesSnmp) throws SQLException {
        /*
         * make sure we have a current PackageIpListMap
         * this was getting done once for each managed ip
         * interface in updateInterfaceInfo. Seems more
         * efficient to just do it once here, and then later
         * for new interfaces (which aren't in the DB yet at
         * this point) in updateInterfaceInfo.
         */
        LOG.debug("updateInterfaces: Rebuilding PackageIpListMap");
        final PollerConfig pollerCfgFactory = PollerConfigFactory.getInstance();
        pollerCfgFactory.rebuildPackageIpListMap();

        /*
         * List of update interfaces
         * This list is maintained so that for nodes with multiple
         * interfaces which support SNMP, interfaces are not updated
         * more than once.
         */
        final List<InetAddress> updatedIfList = new ArrayList<InetAddress>();

        IfSnmpCollector snmpCollector = null;

        if (doesSnmp) {
            /*
             * Reset modification flags. These flags are set by
             * the updateInterface() method when changes have been
             * detected which warrant further action (such as
             * generating an event).
             */
            m_ifIndexOnNodeChangedFlag = false;
            m_snmpIfTableChangedFlag = false;

            /*
             * Determine if any of the interface collector objects have
             * an SNMP collector associated with them. If so, use the first
             * interface with SNMP data collected to update all SNMP-found
             * interfaces.
             */
            IfCollector collectorWithSnmp = null;
            for (final IfCollector tmp : collectorMap.values()) {
                if (tmp.getSnmpCollector() != null) {
                    collectorWithSnmp = tmp;
                    break;
                }
            }

            if (collectorWithSnmp != null) {
                snmpCollector = collectorWithSnmp.getSnmpCollector();

                updateInterface(dbc, now, node, collectorWithSnmp.getTarget(), collectorWithSnmp.getTarget(), collectorWithSnmp.getSupportedProtocols(), snmpCollector, doesSnmp);
                updatedIfList.add(collectorWithSnmp.getTarget());

                // Update subtargets
                if (collectorWithSnmp.hasAdditionalTargets()) {
                    final Map<InetAddress, List<SupportedProtocol>> subTargets = collectorWithSnmp.getAdditionalTargets();
                    for (final Map.Entry<InetAddress,List<SupportedProtocol>> entry : subTargets.entrySet()) {
                        updateInterface(dbc, now, node, collectorWithSnmp.getTarget(), entry.getKey(), entry.getValue(), snmpCollector, doesSnmp);
                        updatedIfList.add(entry.getKey());
                    }
                }

                // Add any new non-IP interfaces
                if (collectorWithSnmp.hasNonIpInterfaces()) {
                    for(final Integer ifIndex : collectorWithSnmp.getNonIpInterfaces()) {
                        updateNonIpInterface(dbc, now, node, ifIndex.intValue(), snmpCollector);
                    }
                }
            }
        }

        /*
         * Majority of interfaces should have been updated by this
         * point (provided the node supports SNMP). Only non-SNMP
         * interfaces and those associated with the node via
         * SMB (NetBIOS name) should remain. Loop through collector
         * map and update any remaining interfaces. Use the
         * updatedIfList object to filter out any interfaces which
         * have already been updated
         */

        // Iterate over interfaces from collection map
        for (final IfCollector ifc : collectorMap.values()) {
            // Update target
            final InetAddress ifaddr = ifc.getTarget();
            if (!updatedIfList.contains(ifaddr)) {
                updateInterface(dbc, now, node, ifc.getTarget(), ifaddr, ifc.getSupportedProtocols(), snmpCollector, doesSnmp);
                updatedIfList.add(ifaddr);
            }

            // Update subtargets
            if (ifc.hasAdditionalTargets()) {
                final Map<InetAddress, List<SupportedProtocol>> subTargets = ifc.getAdditionalTargets();
                for (final Map.Entry<InetAddress,List<SupportedProtocol>> entry : subTargets.entrySet()) {
                    final InetAddress subIf = entry.getKey();
                    if (!updatedIfList.contains(subIf)) {
                        updateInterface(dbc, now, node, ifc.getTarget(), subIf, entry.getValue(), snmpCollector, doesSnmp);
                        updatedIfList.add(subIf);
                    }
                }
            }
        }
    }

    /**
     * This method is responsible for updating the ipInterface table entry for a
     * specific interface.
     * 
     * @param dbc
     *            Database Connection
     * @param now
     *            Date/time to be associated with the update.
     * @param node
     *            Node entry for the node being rescanned
     * @param ifIndex
     *            Interface index of non-IP interface to update
     * @param snmpc
     *            SNMP collector or null if SNMP not supported.
     * 
     * @throws SQLException
     *             if there is a problem updating the ipInterface table.
     */
    private void updateNonIpInterface(final Connection dbc, final Date now, final DbNodeEntry node, final int ifIndex, final IfSnmpCollector snmpc) throws SQLException {
        LOG.debug("updateNonIpInterface: node= {} ifIndex= {}", node.getNodeId(), ifIndex);

        // Sanity Check
        if (snmpc == null || snmpc.failed()) {
            return;
        }

        // Construct InetAddress object for "0.0.0.0" address
        final InetAddress ifAddr = ZERO_ZERO_ZERO_ZERO;
        if (ifAddr == null) return;

        updateSnmpInfoForNonIpInterface(dbc, node, ifIndex, snmpc, ifAddr);
    }

    /**
     * SnmpInterface table updates for non-IP interface.
     */
    private void updateSnmpInfoForNonIpInterface(final Connection dbc, final DbNodeEntry node, final int ifIndex, final IfSnmpCollector snmpc, final InetAddress ifAddr) throws SQLException {
        LOG.debug("updateNonIpInterface: updating non-IP SNMP interface with nodeId={} and ifIndex={}", node.getNodeId(), ifIndex);

        // Create and load SNMP Interface entry from the database
        boolean newSnmpIfTableEntry = false;
        DbSnmpInterfaceEntry dbSnmpIfEntry = DbSnmpInterfaceEntry.get(dbc,node.getNodeId(), ifIndex);
        if (dbSnmpIfEntry == null) {
            /*
             * SNMP Interface not found with this nodeId & ifIndex, create new
             * interface
             */
            LOG.debug("updateNonIpInterface: non-IP SNMP interface with ifIndex {} not in database, creating new snmpInterface object.", ifIndex);
            dbSnmpIfEntry = DbSnmpInterfaceEntry.create(node.getNodeId(), ifIndex);
            newSnmpIfTableEntry = true;
        }

        final IfTableEntry ifte = findEntryByIfIndex(ifIndex, snmpc);
        final IfXTableEntry ifxte = findXEntryByIfIndex(ifIndex, snmpc);

        /*
         * Make sure we have a valid IfTableEntry object and update
         * any values which have changed
         */
        if (ifte != null) {
            // index
            // dbSnmpIfEntry.updateIfIndex(ifIndex);

            /*
             * netmask
             *
             * NOTE: non-IP interfaces don't have netmasks so skip
             */

            updateType(ifte, dbSnmpIfEntry);
            updateDescription(ifIndex, ifte, dbSnmpIfEntry);
            updatePhysicalAddress(ifIndex, ifte, dbSnmpIfEntry);
            updateSpeed(ifIndex, ifte, ifxte, dbSnmpIfEntry);
            updateAdminStatus(ifte, dbSnmpIfEntry);
            updateOperationalStatus(ifte, dbSnmpIfEntry);
            updateName(ifIndex, snmpc, dbSnmpIfEntry);
            updateAlias(ifIndex, snmpc, dbSnmpIfEntry);
        } // end if valid ifTable entry

        /*
         * If this is a new interface or if any of the following
         * key fields have changed set the m_snmpIfTableChangedFlag
         * variable to TRUE. This will potentially trigger an event
         * which will cause the poller to reinitialize the primary
         * SNMP interface for the node.
         */
        // dbSnmpIfEntry.hasIfIndexChanged() ||
        if (!m_snmpIfTableChangedFlag
                && newSnmpIfTableEntry
                || dbSnmpIfEntry.hasTypeChanged()
                || dbSnmpIfEntry.hasNameChanged()
                || dbSnmpIfEntry.hasDescriptionChanged()
                || dbSnmpIfEntry.hasPhysicalAddressChanged()
                || dbSnmpIfEntry.hasAliasChanged()) {
            m_snmpIfTableChangedFlag = true;
        }

        // Update the database
        dbSnmpIfEntry.store(dbc);
    }

    /**
     * Find the ifTable entry for this interface.
     */
    private static IfTableEntry findEntryByIfIndex(final int ifIndex, final IfSnmpCollector snmpc) {
        if (snmpc.hasIfTable()) {
            return snmpc.getIfTable().getEntry(ifIndex);
        }

        return null;
    }

    /**
     * Find the ifXTable entry for this interface.
     */
    private static IfXTableEntry findXEntryByIfIndex(final int ifIndex, final IfSnmpCollector snmpc) {
        if (snmpc.hasIfXTable()) {
            return snmpc.getIfXTable().getEntry(ifIndex);
        }

        return null;
    }

    private static void updateAlias(final int ifIndex, final IfSnmpCollector snmpc, final DbSnmpInterfaceEntry dbSnmpIfEntry) {
        // alias (from interface extensions table)
        final String ifAlias = snmpc.getIfAlias(ifIndex);
        dbSnmpIfEntry.updateAlias(ifAlias == null? "" : ifAlias);
    }

    private static void updateName(final int ifIndex, final IfSnmpCollector snmpc, final DbSnmpInterfaceEntry dbSnmpIfEntry) {
        // name (from interface extensions table)
        final String ifName = snmpc.getIfName(ifIndex);
        if (ifName != null && ifName.length() > 0) {
            dbSnmpIfEntry.updateName(ifName);
        }
    }

    private static void updateOperationalStatus(final IfTableEntry ifte, final DbSnmpInterfaceEntry dbSnmpIfEntry) {
        final Integer sint = ifte.getIfOperStatus();
        dbSnmpIfEntry.updateOperationalStatus(sint == null? 0 : sint.intValue());
    }

    private static void updateAdminStatus(final IfTableEntry ifte, final DbSnmpInterfaceEntry dbSnmpIfEntry) {
        final Integer sint = ifte.getIfAdminStatus();
        dbSnmpIfEntry.updateAdminStatus(sint == null? 0 : sint.intValue());
    }

    private static void updateType(final IfTableEntry ifte, final DbSnmpInterfaceEntry dbSnmpIfEntry) {
        final Integer sint = ifte.getIfType();
        dbSnmpIfEntry.updateType(sint == null? 0 : sint.intValue());
    }

    private static void updateDescription(final int ifIndex, final IfTableEntry ifte, final DbSnmpInterfaceEntry dbSnmpIfEntry) {
        final String str = ifte.getIfDescr();
        LOG.debug("updateNonIpInterface: ifIndex: {} has ifDescription: {}", ifIndex, str);
        if (str != null && str.length() > 0) {
            dbSnmpIfEntry.updateDescription(str);
        }
    }

    private static void updatePhysicalAddress(final int ifIndex, final IfTableEntry ifte, final DbSnmpInterfaceEntry dbSnmpIfEntry) {
        final String physAddr = ifte.getPhysAddr();

        LOG.debug("updateNonIpInterface: ifIndex: {} has physical address '{}'", ifIndex, physAddr);

        if (physAddr != null && physAddr.length() == 12) {
            dbSnmpIfEntry.updatePhysicalAddress(physAddr);
        }
    }

    static void updateSpeed(final int ifIndex, final IfTableEntry ifte, final IfXTableEntry ifxte, final DbSnmpInterfaceEntry dbSnmpIfEntry) {
        try {
            dbSnmpIfEntry.updateSpeed(getInterfaceSpeed(ifte, ifxte));
        } catch (final Throwable t) {
            LOG.warn("updateNonIpInterface: ifSpeed '{}' for ifIndex {} is invalid, inserting 0", ifte.getDisplayString(IfTableEntry.IF_SPEED), ifIndex, t);
            dbSnmpIfEntry.updateSpeed(0);
        }
    }

    private static long getInterfaceSpeed(final IfTableEntry ifte, final IfXTableEntry ifxte) {
        if (ifxte != null && ifxte.getIfHighSpeed() != null && ifxte.getIfHighSpeed() > 4294) {
            return ifxte.getIfHighSpeed() * 1000000L; 
        }

        if (ifte != null && ifte.getIfSpeed() != null) {
            return ifte.getIfSpeed();
        }

        return 0;
    }

    /**
     * This method is responsible for updating the ipInterface table entry for a
     * specific interface.
     * 
     * @param dbc
     *            Database Connection
     * @param now
     *            Date/time to be associated with the update.
     * @param node
     *            Node entry for the node being rescanned
     * @param target
     *            Target interface (from IfCollector.getTarget())
     * @param ifaddr
     *            Interface being updated.
     * @param protocols
     *            Protocols supported by the interface.
     * @param snmpc
     *            SNMP collector or null if SNMP not supported.
     * @param doesSnmp
     *            Indicates that the interface supports SNMP
     * 
     * @throws SQLException
     *             if there is a problem updating the ipInterface table.
     */
    private void updateInterface(final Connection dbc, final Date now, final DbNodeEntry node, final InetAddress target, final InetAddress ifaddr, final List<SupportedProtocol> protocols, final IfSnmpCollector snmpc, final boolean doesSnmp) throws SQLException {
        /*
         * Reparenting
         *
         * This sub-interface was not previously associated with this node. If
         * the sub-interface is already associated with another node we must do
         * one of the following:
         *
         * 1. If the target interface (the one being rescanned) appears to be an
         * interface alias all of the interfaces under the sub-interface's node
         * will be reparented under the nodeid of the target interface.
         *
         * 2. If however the interface is not an alias, only the sub-interface
         * will be reparented under the nodeid of the interface being rescanned.
         *
         * In the reparenting process, the database ipinterface, snmpinterface
         * and ifservices table entries associated with the reparented interface
         * will be "updated" to reflect the new nodeid. If the old node has
         * no remaining interfaces following the reparenting it will be marked
         * as deleted.
         */

        /*
         * Special case: Need to skip interface reparenting for '0.0.0.0'
         * interfaces as well as loopback interfaces ('127.*.*.*').
         */
        final String ifaddrString = str(ifaddr);
        LOG.debug("updateInterface: updating interface {} (targetIf={})", ifaddrString, str(target));
        if (doesSnmp) {
            LOG.debug("updateInterface: the SNMP collection passed in is collected via {}", (snmpc ==  null ? "No SnmpCollection passed in (snmpc == null)" : str(snmpc.getCollectorTargetAddress())));
        }

        boolean reparentFlag = false;
        boolean newIpIfEntry = false;
        int ifIndex = -1;

        DbIpInterfaceEntry dbIpIfEntry = DbIpInterfaceEntry.get(dbc,node.getNodeId(), ifaddr);

        if (doesSnmp && snmpc != null && snmpc.hasIpAddrTable()) {
            // Attempt to load IP Interface entry from the database
            ifIndex = snmpc.getIfIndex(ifaddr);
            LOG.debug("updateInterface: interface = {} ifIndex = {}. Checking for this address on other nodes.", ifaddrString, ifIndex);

            /*
             * the updating interface may have already existed in the
             * ipinterface table with different
             * nodeIds. If it exist in a different node, verify if all the
             * interfaces on that node
             * are contained in the snmpc of the updating interface. If they
             * are, reparent all
             * the interfaces on that node to the node of the updating
             * interface, otherwise, just add
             * the interface to the updating node.
             */
            // Verify that SNMP collection contains ipAddrTable entries
            final IpAddrTable ipAddrTable = snmpc.getIpAddrTable();

            if (ipAddrTable == null) {
                LOG.error("updateInterface: null ipAddrTable in the SNMP collection");
            } else {
                if (s_ZERO_ZERO_ZERO_ZERO.equals(ifaddrString) || ifaddr.isLoopbackAddress()) {
                    LOG.debug("updateInterface: Skipping address from snmpc ipAddrTable {}", ifaddrString);
                } else {
                    LOG.debug("updateInterface: Checking address from snmpc ipAddrTable {}", ifaddrString);

                    final DBUtils d = new DBUtils(RescanProcessor.class);
                    try {
                        final PreparedStatement stmt = dbc.prepareStatement(SQL_DB_RETRIEVE_OTHER_NODES);
                        d.watch(stmt);
                        stmt.setString(1, ifaddrString);
                        stmt.setInt(2, node.getNodeId());

                        final ResultSet rs = stmt.executeQuery();
                        d.watch(rs);
                        while (rs.next()) {
                            final int existingNodeId = rs.getInt(1);
                            LOG.debug("updateInterface: ckecking for {} on existing nodeid {}", ifaddrString, existingNodeId);

                            final DbNodeEntry suspectNodeEntry = DbNodeEntry.get(dbc, existingNodeId);
                            if (suspectNodeEntry == null) {
                                // This can happen if a node has been deleted.
                                continue;
                            }

                            /*
                             * Retrieve list of interfaces associated with the
                             * old node
                             */
                            final DbIpInterfaceEntry[] tmpIfArray = suspectNodeEntry.getInterfaces(dbc);

                            /*
                             * Verify if the suspectNodeEntry is a duplicate
                             * node
                             */
                            if (areDbInterfacesInSnmpCollection(tmpIfArray, snmpc)) {
                                /*
                                 * Reparent each interface under the targets'
                                 * nodeid
                                 */
                                for (int i = 0; i < tmpIfArray.length; i++) {
                                    final InetAddress addr = tmpIfArray[i].getIfAddress();
                                    final int index = snmpc.getIfIndex(addr);

                                    // Skip non-IP or loopback interfaces
                                    final String addrString = str(addr);
                                    if (addrString == null || s_ZERO_ZERO_ZERO_ZERO.equals(addrString) || addr.isLoopbackAddress()) {
                                        continue;
                                    }

                                    LOG.debug("updateInterface: reparenting interface {} under node: {} from existing node: {}", addrString, node.getNodeId(), existingNodeId);

                                    reparentInterface(dbc, addr, index, node.getNodeId(), existingNodeId);

                                    // Create interfaceReparented event
                                    createInterfaceReparentedEvent(node, existingNodeId, addr);
                                }

                                LOG.debug("updateInterface: interface {} is added to node: {} by reparenting from existing node: {}", ifaddrString, node.getNodeId(), existingNodeId);
                                dbIpIfEntry = DbIpInterfaceEntry.get(dbc, node.getNodeId(), ifaddr);
                                reparentFlag = true;

                                // delete duplicate node after reparenting.
                                deleteDuplicateNode(dbc, suspectNodeEntry);
                                createDuplicateNodeDeletedEvent(suspectNodeEntry);
                            }
                        }
                    }

                    catch (final SQLException e) {
                        LOG.error("SQLException while updating interface: {} on nodeid: {}", ifaddrString, node.getNodeId());
                        throw e;
                    } finally {
                        d.cleanUp();
                    }
                }
            }
        }

        /*
         * if no reparenting occurred on the updating interface, add it to the
         * updating node.
         */
        if (dbIpIfEntry == null) {
            /*
             * Interface not found with this nodeId so create new interface
             * entry
             */
            LOG.debug("updateInterface: interface {} ifIndex {} not in database under nodeid {}, creating new interface object.", ifaddr, ifIndex, node.getNodeId());

            /*
             * If doesSnmp is set to true, the dbIpIfEntry must not be stored
             * to the database until the corresponding DbSnmpInterfaceEntry is
             * stored.
             */
            if (ifIndex == -1 && !doesSnmp) {
                dbIpIfEntry = DbIpInterfaceEntry.create(node.getNodeId(), ifaddr);
            } else {
                dbIpIfEntry = DbIpInterfaceEntry.create(node.getNodeId(), ifaddr, ifIndex);
                /*
                 * XXX uh, what????? - dj@opennms.org
                 * This wasn't getting done for some reason, so do it explicitly
                 */
                dbIpIfEntry.setIfIndex(ifIndex);
            }

            if (isDuplicateInterface(dbc, ifaddr, node.getNodeId())) {
                m_eventList.add(createDuplicateIpAddressEvent(dbIpIfEntry));
            }
            newIpIfEntry = true;
        }

        final DbIpInterfaceEntry currIpIfEntry = getNewDbIpInterfaceEntry(node, snmpc, doesSnmp, ifaddr);

        /*
         * XXX Note that updateSnmpInfo only gets called if doesSnmp is
         * true, but a new dbIpIfEntry with an ifIndex might have been
         * create()ed above if ifIndex != -1 || doesSnmp.  This might be
         * a problem if doesSnmp is false but ifIndex != -1, as the ipInterface
         * entry will point an snmpInterface entry that might not exist.
         */
        if (doesSnmp && snmpc != null) {
            // update SNMP info if available
            updateSnmpInfo(dbc, node, snmpc, currIpIfEntry.getIfAddress(), currIpIfEntry.getIfIndex());
        }

        // update ipinterface for the updating interface
        updateInterfaceInfo(dbc, now, node, dbIpIfEntry, currIpIfEntry, newIpIfEntry, reparentFlag);

        // update IfServices for the updating interface
        updateServiceInfo(dbc, node, dbIpIfEntry, newIpIfEntry, protocols);

    }

    /**
     * This method is responsible to delete any interface associated with the
     * duplicate node, delete any entry left in ifservices table and
     * snmpinterface table for the duplicate node, and make the node as
     * 'deleted'.
     * 
     * @param dbc
     *            Database Connection
     * @param duplicateNode
     *            Duplicate node to delete.
     * 
     */
    private static void deleteDuplicateNode(final Connection dbc, final DbNodeEntry duplicateNode) throws SQLException {
        final DBUtils d = new DBUtils(RescanProcessor.class);
        try {
            final PreparedStatement ifStmt = dbc.prepareStatement(SQL_DB_DELETE_DUP_INTERFACE);
            d.watch(ifStmt);
            final PreparedStatement svcStmt = dbc.prepareStatement(SQL_DB_DELETE_DUP_SERVICES);
            d.watch(svcStmt);
            final PreparedStatement snmpStmt = dbc.prepareStatement(SQL_DB_DELETE_DUP_SNMPINTERFACE);
            d.watch(snmpStmt);
            ifStmt.setInt(1, duplicateNode.getNodeId());
            svcStmt.setInt(1, duplicateNode.getNodeId());
            snmpStmt.setInt(1, duplicateNode.getNodeId());

            ifStmt.executeUpdate();
            svcStmt.executeUpdate();
            snmpStmt.executeUpdate();

            duplicateNode.setNodeType(NodeType.DELETED);
            duplicateNode.store(dbc);
        } catch (final SQLException sqlE) {
            LOG.error("deleteDuplicateNode  SQLException while deleting duplicate node: {}", duplicateNode.getNodeId());
            throw sqlE;
        } finally {
            d.cleanUp();
        }

    }

    /**
     * This method verify if an ipaddress is existing in other node except in
     * the updating node.
     * 
     * @param dbc
     *            Database Connection
     * @param ifaddr
     *            Ip address being verified.
     * @param nodeId
     *            Node Id of the node being rescanned
     * 
     */
    private static boolean isDuplicateInterface(final Connection dbc, final InetAddress ifaddr, final int nodeId) throws SQLException {
        final DBUtils d = new DBUtils(RescanProcessor.class);
        try {
            final PreparedStatement stmt = dbc.prepareStatement(SQL_DB_RETRIEVE_OTHER_NODES);
            d.watch(stmt);
            stmt.setString(1, str(ifaddr));
            stmt.setInt(2, nodeId);

            final ResultSet rs = stmt.executeQuery();
            d.watch(rs);
            while (rs.next()) {
                return true;
            }
        } catch (final SQLException sqlE) {
            LOG.error("isDuplicateInterface: SQLException while updating interface: {} on nodeid: {}", str(ifaddr), nodeId);
            throw sqlE;
        } finally {
            d.cleanUp();
        }
        return false;
    }

    /**
     * This method is responsible for updating the ipinterface table entry for a
     * specific interface.
     * 
     * @param dbc
     *            Database Connection.
     * @param now
     *            Date/time to be associated with the update.
     * @param node
     *            Node entry for the node being rescanned.
     * @param dbIpIfEntry
     *            interface entry of the updating interface.
     * @param snmpc
     *            SNMP collector or null if SNMP not supported.
     * @param isNewIpEntry
     *            if dbIpIfEntry is a new entry.
     * @param isReparented
     *            if dbIpIfEntry is reparented.
     * @param doesSnmp
     *            if node supports SNMP.
     * 
     * @throws SQLException
     *             if there is a problem updating the ipinterface table.
     */
    private void updateInterfaceInfo(final Connection dbc, final Date now, final DbNodeEntry node, final DbIpInterfaceEntry dbIpIfEntry, final DbIpInterfaceEntry currIpIfEntry, final boolean isNewIpEntry, final boolean isReparented) throws SQLException {
        final PollerConfig pollerCfgFactory = PollerConfigFactory.getInstance();

        final InetAddress ifaddr = dbIpIfEntry.getIfAddress();

        /*
         * Clone the existing database entry so we have access to the values
         * of the database fields associated with the interface in the event
         * that something has changed.
         */
        final DbIpInterfaceEntry originalIpIfEntry = DbIpInterfaceEntry.clone(dbIpIfEntry);

        // Update any fields which have changed
        dbIpIfEntry.setLastPoll(now);
        dbIpIfEntry.updateHostname(currIpIfEntry.getHostname());
        dbIpIfEntry.updateManagedState(currIpIfEntry.getManagedState());
        dbIpIfEntry.updateStatus(currIpIfEntry.getStatus());
        dbIpIfEntry.updatePrimaryState(currIpIfEntry.getPrimaryState());

        /*
         * XXX Note: the ifIndex will not be updated if updateIfIndex(-1)
         * is called.  In other words, an ifIndex of a value other than -1
         * (non-null in the database) will never change to -1 (which is null
         * in the database) by calling updateIfIndex.  setIfIndex does work,
         * however if m_useIfIndexAsKey is set in the DbIpInterfaceEntry,
         * no entries (or at least not the right entry) will be updated
         * because the WHERE clause for the UPDATE will be referring to the
         * *new* ifIndex.
         */
        dbIpIfEntry.updateIfIndex(currIpIfEntry.getIfIndex());

        /*
         * Set event flags
         * NOTE: Must set these flags prior to call to
         * DbIpInterfaceEntry.store()
         * method which will cause the change map to be cleared.
         */
        final boolean ifIndexChangedFlag = dbIpIfEntry.hasIfIndexChanged();
        final boolean ipHostnameChangedFlag = dbIpIfEntry.hasHostnameChanged();

        // Update the database
        dbIpIfEntry.store(dbc);

        /*
         * If the interface was not already in the database under
         * the node being rescanned or some other node send a
         * nodeGainedInterface event.
         */
        if (isNewIpEntry && !isReparented) {
            createNodeGainedInterfaceEvent(dbIpIfEntry);
        }

        // InterfaceIndexChanged event
        LOG.debug("updateInterfaceInfo: ifIndex changed: {}", ifIndexChangedFlag);
        if (ifIndexChangedFlag) {
            m_eventList.add(createInterfaceIndexChangedEvent(dbIpIfEntry, originalIpIfEntry));
            m_ifIndexOnNodeChangedFlag = true;
        }

        // IPHostNameChanged event
        LOG.debug("updateInterfaceInfo: hostname changed: {}", ipHostnameChangedFlag);
        if (ipHostnameChangedFlag) {
            m_eventList.add(createIpHostNameChangedEvent(dbIpIfEntry, originalIpIfEntry));
        }

        if (isNewIpEntry) {
            /*
             * If it's new, the packageIpListMap needs to be rebuilt,
             * polling status rechecked, and ismanaged updated if necessary
             */
            final String ifaddrString = str(ifaddr);
            LOG.debug("updateInterfaceInfo: rebuilding PackageIpListMap for new interface {}", ifaddrString);
            PollerConfigFactory.getInstance().rebuildPackageIpListMap();
            org.opennms.netmgt.config.poller.Package ipPkg = ifaddrString == null? null : pollerCfgFactory.getFirstPackageMatch(ifaddrString);
            
            final boolean ipToBePolled = (ipPkg != null);
            LOG.debug("updateInterfaceInfo: interface {} to be polled: {}", ifaddrString, ipToBePolled);
            if (ipToBePolled) {
                final DBUtils d = new DBUtils(RescanProcessor.class);
                try {
                    final PreparedStatement stmt = dbc.prepareStatement(SQL_DB_UPDATE_ISMANAGED);
                    d.watch(stmt);
                    stmt.setString(1, new String(new char[] { DbIpInterfaceEntry.STATE_MANAGED }));
                    stmt.setLong(2, dbIpIfEntry.getNodeId());
                    stmt.setString(3, ifaddrString);
                    stmt.executeUpdate();
                    LOG.debug("updateInterfaceInfo: updated managed state for new interface {} on node {} to managed", ifaddrString, dbIpIfEntry.getNodeId());
                } finally {
                    d.cleanUp();
                }
            }
        }
    }

    /**
     * Create IP interface entry representing latest information
     * retrieved for the interface via the collector.  If doesSnmp is set to
     * <i>true</i>, this entry must <b>not</b> be stored to the database until
     * the corresponding DbSnmpInterfaceEntry is stored.
     */
    private static DbIpInterfaceEntry getNewDbIpInterfaceEntry(final DbNodeEntry node, final IfSnmpCollector snmpc, final boolean doesSnmp, final InetAddress ifaddr) {
        final CapsdConfig cFactory = CapsdConfigFactory.getInstance();
        final PollerConfig pollerCfgFactory = PollerConfigFactory.getInstance();

        int ifIndex = -1;

        final DbIpInterfaceEntry currIpIfEntry;
        final String ifaddrString = str(ifaddr);
        if (doesSnmp) {
            if (snmpc != null && snmpc.hasIpAddrTable()) {
                ifIndex = snmpc.getIfIndex(ifaddr);
            }
            if (ifIndex == -1) {
                LOG.debug("updateInterfaceInfo: interface {} has no valid ifIndex. Assuming this is a lame SNMP host with no ipAddrTable", ifaddrString);
                ifIndex = CapsdConfig.LAME_SNMP_HOST_IFINDEX;
            }
            currIpIfEntry = DbIpInterfaceEntry.create(node.getNodeId(), ifaddr, ifIndex);
        } else {
            currIpIfEntry = DbIpInterfaceEntry.create(node.getNodeId(), ifaddr);
        }

        // Hostname
        currIpIfEntry.setHostname(ifaddr.getHostName());

        /*
         * Managed state
         * NOTE: (reference internal bug# 201)
         * If the ip is 'managed', it might still be 'not polled' based
         * on the poller configuration.
         *
         * Try to avoid re-evaluating the ip against filters for
         * each service, try to get the first package here and use
         * that for service evaluation
         *
         * At this point IF the ip is already in the database, package filter
         * evaluation should go through OK. New interfaces will be dealt with
         * later
         */
        org.opennms.netmgt.config.poller.Package ipPkg = null;

        if (cFactory.isAddressUnmanaged(ifaddr)) {
            currIpIfEntry.setManagedState(DbIpInterfaceEntry.STATE_UNMANAGED);
        } else {
            ipPkg = ifaddrString == null? null : pollerCfgFactory.getFirstPackageMatch(ifaddrString);
            final boolean ipToBePolled = (ipPkg != null);

            if (ipToBePolled) {
                currIpIfEntry.setManagedState(DbIpInterfaceEntry.STATE_MANAGED);
            } else {
                currIpIfEntry.setManagedState(DbIpInterfaceEntry.STATE_NOT_POLLED);
            }

            LOG.debug("updateInterfaceInfo: interface {} to be polled = {}", ifaddrString, ipToBePolled);
        }

        /*
         * If SNMP data collection is available set SNMP Primary state
         * as well as ifIndex and ifStatus.
         *
         * For all interfaces simply set 'isSnmpPrimary' field to
         * not eligible for now. Following the interface updates
         * the primary and secondary SNMP interfaces will be
         * determined and the value of the 'isSnmpPrimary' field
         * set accordingly for each interface. The old primary
         * interface should have already been saved for future
         * reference.
         */
        if (doesSnmp && snmpc != null && snmpc.hasIpAddrTable()) {
            if (ifIndex != -1) {
                if (snmpc.hasIfTable()) {
                    currIpIfEntry.setStatus(snmpc.getAdminStatus(ifIndex));
                }
            } else {
                // No ifIndex found
                LOG.debug("updateInterfaceInfo:  No ifIndex found for {}. Not eligible for primary SNMP interface.", ifaddrString);
            }
            currIpIfEntry.setPrimaryState(DbIpInterfaceEntry.SNMP_NOT_ELIGIBLE);
        } else if (doesSnmp) {
            currIpIfEntry.setPrimaryState(DbIpInterfaceEntry.SNMP_NOT_ELIGIBLE);
        }
        return currIpIfEntry;
    }

    /**
     * This method is responsible for updating the ifservices table entry for a
     * specific interface.
     * 
     * @param dbc
     *            Database Connection.
     * @param node
     *            Node entry for the node being rescanned.
     * @param dbIpIfEntry
     *            interface entry of the updating interface.
     * @param isNewIpEntry
     *            if the dbIpIfEntry is a new entry.
     * @param protocols
     *            Protocols supported by the interface.
     * 
     * @throws SQLException
     *             if there is a problem updating the ifservices table.
     */
    private void updateServiceInfo(final Connection dbc, final DbNodeEntry node, final DbIpInterfaceEntry dbIpIfEntry, final boolean isNewIpEntry, final List<SupportedProtocol> protocols) throws SQLException {
        final CapsdConfig cFactory = CapsdConfigFactory.getInstance();
        final PollerConfig pollerCfgFactory = PollerConfigFactory.getInstance();
        org.opennms.netmgt.config.poller.Package ipPkg = null;

        final InetAddress ifaddr = dbIpIfEntry.getIfAddress();

        // Retrieve from the database the interface's service list
        DbIfServiceEntry[] dbSupportedServices = dbIpIfEntry.getServices(dbc);

        final int ifIndex = dbIpIfEntry.getIfIndex();

        if (LOG.isDebugEnabled()) {
            if (ifIndex == -1) {
                LOG.debug("updateServiceInfo: Retrieving interface's service list from database for host {}", dbIpIfEntry.getHostname());
            } else {
                LOG.debug("updateServiceInfo: Retrieving interface's service list from database for host {} ifindex {}", dbIpIfEntry.getHostname(), ifIndex);
            }
        }

        /*
         * add newly supported protocols
         *		
         * NOTE!!!!!: (reference internal bug# 201)
         * If the ip is 'managed', the service can still be 'not polled'
         * based on the poller configuration - at this point the ip is already
         * in the database, so package filter evaluation should go through OK
         */
        if (LOG.isDebugEnabled()) {
            LOG.debug("updateServiceInfo: Checking for new services on host {}", dbIpIfEntry.getHostname());
        }

        final Iterator<SupportedProtocol> iproto = protocols.iterator();
        while (iproto.hasNext()) {
            final SupportedProtocol p = iproto.next();
            final Number sid = m_capsdDbSyncer.getServiceId(p.getProtocolName());

            /*
             * Only adding newly supported services so check against the service
             * list retrieved from the database
             */
            boolean found = false;
            for (int i = 0; i < dbSupportedServices.length && !found; i++) {
                if (dbSupportedServices[i].getServiceId() == sid.intValue()) {
                    found = true;
                    break;
                }
            }

            if (!found) {
                final DbIfServiceEntry ifSvcEntry = DbIfServiceEntry.create(node.getNodeId(), ifaddr, sid.intValue());

                // now fill in the entry
                final String ifaddrString = str(ifaddr);
                if (cFactory.isAddressUnmanaged(ifaddr)) {
                    ifSvcEntry.setStatus(DbIfServiceEntry.STATUS_UNMANAGED);
                } else {
                    ipPkg = ifaddrString == null? null : pollerCfgFactory.getFirstPackageMatch(ifaddrString);
                    if (ipPkg == null) {
                        ifSvcEntry.setStatus(DbIfServiceEntry.STATUS_NOT_POLLED);
                    } else if (isServicePolledLocally(ifaddrString, p.getProtocolName(), ipPkg)) {
                        ifSvcEntry.setStatus(DbIfServiceEntry.STATUS_ACTIVE);
                    } else if (isServicePolled(ifaddrString, p.getProtocolName(), ipPkg)) {
                        ifSvcEntry.setStatus(DbIpInterfaceEntry.STATE_REMOTE);
                    } else {
                        ifSvcEntry.setStatus(DbIfServiceEntry.STATUS_NOT_POLLED);
                    }
                }

                /*
                 * Set qualifier if available. Currently the qualifier field
                 * is used to store the port at which the protocol was found.
                 */
                if (p.getQualifiers() != null && p.getQualifiers().get("port") != null) {
                    try {
                        final Integer port = (Integer) p.getQualifiers().get("port");
                        LOG.debug("updateServiceInfo: got a port qualifier: {} for service: {}", port, p.getProtocolName());
                        ifSvcEntry.setQualifier(port.toString());
                    } catch (final ClassCastException ccE) {
                        // Do nothing
                    }
                }

                ifSvcEntry.setSource(DbIfServiceEntry.SOURCE_PLUGIN);
                ifSvcEntry.setNotify(DbIfServiceEntry.NOTIFY_ON);

                if (ifIndex != -1) {
                    ifSvcEntry.setIfIndex(ifIndex);
                }

                ifSvcEntry.store();

                LOG.debug("updateIfServices: update service: {} for interface:{} on node:{}", p.getProtocolName(), ifaddrString, node.getNodeId());

                // Generate nodeGainedService event
                m_eventList.add(createNodeGainedServiceEvent(node, dbIpIfEntry, p.getProtocolName()));

                /*
                 * If this interface already existed in the database and SNMP
                 * service has been gained then create interfaceSupportsSNMP
                 * event
                 */
                if (!isNewIpEntry && p.getProtocolName().equalsIgnoreCase("SNMP")) {
                    m_eventList.add(createInterfaceSupportsSNMPEvent(dbIpIfEntry));
                }
            }
            // Update the supported services list
            dbSupportedServices = dbIpIfEntry.getServices(dbc);
        } // end while(more protocols)

        if (m_forceRescan) {
            updateServicesOnForcedRescan(node, dbIpIfEntry, dbSupportedServices);
        }
    }

    private static boolean isServicePolled(final String ifAddr, final String svcName, final org.opennms.netmgt.config.poller.Package ipPkg) {
        boolean svcToBePolled = false;
        if (ipPkg != null) {
            if (PollerConfigFactory.getInstance().isPolled(svcName, ipPkg)) {
                return true;
            }
            if (PollerConfigFactory.getInstance().isPolled(ifAddr, svcName)) {
                return true;
            }
        }
        return false;
    }

    private static boolean isServicePolledLocally(final String ifAddr, final String svcName, final org.opennms.netmgt.config.poller.Package ipPkg) {
        if (ipPkg != null && !ipPkg.getRemote()) {
            if (PollerConfigFactory.getInstance().isPolled(svcName, ipPkg)) {
                return true;
            }
            if (PollerConfigFactory.getInstance().isPolledLocally(ifAddr, svcName)) {
                return true;
            }
        }
        return false;
    }

    /**
     * This method is responsible for updating the status of services for an
     * interface during a forced rescan
     * 
     * @param node
     *            Node entry for the node being rescanned
     * @param dbIpIfEntry
     *            interface entry of the updating interface
     * @param dbSupportedServices
     *            services on the updating interface
     * 
     * @throws SQLException
     *             if there is a problem updating the snmpInterface table.
     */
    private void updateServicesOnForcedRescan(final DbNodeEntry node, final DbIpInterfaceEntry dbIpIfEntry, final DbIfServiceEntry[] dbSupportedServices) throws SQLException {
        /*
         * Now process previously existing protocols to update polling status.
         * Additional checks on forced rescan for existing services go here.
         * Specifically, has service been forced managed/unmanaged or has
         * polling status changed?
         */

        final PollerConfig pollerCfgFactory = PollerConfigFactory.getInstance();
        final CapsdConfig cFactory = CapsdConfigFactory.getInstance();
        final InetAddress ifaddr = dbIpIfEntry.getIfAddress();
        final String ifaddrString = str(ifaddr);

        org.opennms.netmgt.config.poller.Package ipPkg = ifaddrString == null? null : pollerCfgFactory.getFirstPackageMatch(ifaddrString);

        boolean ipToBePolled = (ipPkg != null);

        LOG.debug("updateServicesOnForcedRescan: Checking status of existing services on host {}", ifaddr);

        // Get service names from database
        final Map<Integer, String> serviceNames = new HashMap<Integer, String>();
        final DBUtils d = new DBUtils(RescanProcessor.class);
        try {
            final Connection ctest = DataSourceFactory.getInstance().getConnection();
            d.watch(ctest);
            final PreparedStatement loadStmt = ctest.prepareStatement(SQL_RETRIEVE_SERVICE_IDS);
            d.watch(loadStmt);

            // go ahead and load the service table
            final ResultSet rs = loadStmt.executeQuery();
            d.watch(rs);
            while (rs.next()) {
                final Integer id = Integer.valueOf(rs.getInt(1));
                final String name = rs.getString(2);
                serviceNames.put(id, name);
            }
        } catch (final Throwable t) {
            LOG.error("Error reading services table", t);
        } finally {
            d.cleanUp();
        }

        for (int i = 0; i < dbSupportedServices.length; i++) {
            final Integer id = dbSupportedServices[i].getServiceId();
            final String sn = (serviceNames.get(id)).toString();

            final DbIfServiceEntry ifSvcEntry = DbIfServiceEntry.get(node.getNodeId(), ifaddr, dbSupportedServices[i].getServiceId());
            LOG.debug("updateServicesOnForcedRescan: old status for nodeId {}, ifaddr {}, serviceId {} = {}", node.getNodeId(), ifaddr, dbSupportedServices[i].getServiceId(), ifSvcEntry.getStatus());

            // now fill in the entry

            boolean svcChangeToActive = false;
            boolean svcChangeToNotPolled = false;
            boolean svcChangeToForced = false;
            if (!cFactory.isAddressUnmanaged(ifaddr)) {
                boolean svcToBePolled = false;
                if (ipToBePolled) {
                    if (ipPkg == null) {
                        ipPkg = pollerCfgFactory.getFirstPackageMatch(ifaddrString);
                    }
                    if (ipPkg != null) {
                        LOG.debug("updateServicesOnForcedRescan: Is service to be polled for package = {}, service = {}", ipPkg.getName(), sn);
                        svcToBePolled = pollerCfgFactory.isPolled(sn, ipPkg);
                        if (!svcToBePolled) {
                            LOG.debug("updateServicesOnForcedRescan: Is service to be polled for ifaddr = {}, service = {}", ifaddrString, sn);
                            svcToBePolled = pollerCfgFactory.isPolled(ifaddrString, sn);
                        }
                        if (!svcToBePolled) {
                            LOG.debug("updateServicesOnForcedRescan: Service not to be polled");
                        }
                    } else {
                        LOG.debug("updateServicesOnForcedRescan: No poller package found");
                    }
                } else {
                    LOG.debug("updateServicesOnForcedRescan: Service not polled because interface is not polled");

                }

                if (ifSvcEntry.getStatus() == DbIfServiceEntry.STATUS_FORCED) {
                    if (svcToBePolled) {
                        // Do nothing
                        LOG.debug("updateServicesOnForcedRescan: status = FORCED. No action taken.");
                    } else {
                        // change the status to "N"
                        ifSvcEntry.updateStatus(DbIfServiceEntry.STATUS_NOT_POLLED);
                        svcChangeToNotPolled = true;
                        LOG.debug("updateServicesOnForcedRescan: status = FORCED. Changed to NOT_POLLED");
                    }
                } else if (ifSvcEntry.getStatus() == DbIfServiceEntry.STATUS_SUSPEND) {
                    if (svcToBePolled) {
                        // change the status to "F"
                        ifSvcEntry.updateStatus(DbIfServiceEntry.STATUS_FORCED);
                        svcChangeToForced = true;
                        LOG.debug("updateServicesOnForcedRescan: status = SUSPEND. Changed to FORCED");
                    } else {
                        // change the status to "N"
                        ifSvcEntry.updateStatus(DbIfServiceEntry.STATUS_NOT_POLLED);
                        svcChangeToNotPolled = true;
                        LOG.debug("updateServicesOnForcedRescan: status = SUSPEND. Changed to NOT_POLLED");
                    }
                } else if (ifSvcEntry.getStatus() == DbIfServiceEntry.STATUS_RESUME) {
                    if (svcToBePolled) {
                        // change the status to "A"
                        ifSvcEntry.updateStatus(DbIfServiceEntry.STATUS_ACTIVE);
                        svcChangeToActive = true;
                        LOG.debug("updateServicesOnForcedRescan: status = RESUME. Changed to ACTIVE");
                    } else {
                        // change the status to "N"
                        ifSvcEntry.updateStatus(DbIfServiceEntry.STATUS_NOT_POLLED);
                        svcChangeToNotPolled = true;
                        LOG.debug("updateServicesOnForcedRescan: status = RESUME. Changed to NOT_POLLED");
                    }
                } else if (svcToBePolled && ifSvcEntry.getStatus() != DbIfServiceEntry.STATUS_ACTIVE) {
                    // set the status to "A"
                    ifSvcEntry.updateStatus(DbIfServiceEntry.STATUS_ACTIVE);
                    svcChangeToActive = true;
                    LOG.debug("updateServicesOnForcedRescan: New status = ACTIVE");
                } else if (!svcToBePolled && ifSvcEntry.getStatus() == DbIfServiceEntry.STATUS_ACTIVE) {
                    // set the status to "N"
                    ifSvcEntry.updateStatus(DbIfServiceEntry.STATUS_NOT_POLLED);
                    svcChangeToNotPolled = true;
                    LOG.debug("updateServicesOnForcedRescan: New status = NOT_POLLED");
                } else {
                    LOG.debug("updateServicesOnForcedRescan: Status Unchanged");
                }
            }

            if (svcChangeToActive) {
                ifSvcEntry.store();
                m_eventList.add(createResumePollingServiceEvent(node, dbIpIfEntry, sn));
            } else if (svcChangeToNotPolled || svcChangeToForced) {
                ifSvcEntry.store();
                m_eventList.add(createSuspendPollingServiceEvent(node, dbIpIfEntry, sn));
            }
        }
    }

    /**
     * This method is responsible for updating the snmpInterface table entry for
     * a specific interface.
     * 
     * @param dbc
     *            Database Connection
     * @param node
     *            Node entry for the node being rescanned
     * @param dbIpIfEntry
     *            interface entry of the updating interface
     * @param snmpc
     *            SNMP collector or null if SNMP not supported.
     * 
     * @throws SQLException
     *             if there is a problem updating the snmpInterface table.
     */
    private void updateSnmpInfo(final Connection dbc, final DbNodeEntry node, final IfSnmpCollector snmpc, final InetAddress ifaddr, final int ifIndex) throws SQLException {
        /*
         * If SNMP info is available update the snmpInterface table entry with
         * anything that has changed.
         */		
        if (snmpc != null && !snmpc.failed() && ifIndex != -1) {
            LOG.debug("updateSnmpInfo: updating SNMP interface for nodeId/ifIndex={}/{}", node.getNodeId(), ifIndex);

            // Create and load SNMP Interface entry from the database
            boolean newSnmpIfTableEntry = false;
            DbSnmpInterfaceEntry dbSnmpIfEntry =  DbSnmpInterfaceEntry.get(dbc, node.getNodeId(), ifIndex);
            if (dbSnmpIfEntry == null) {
                /*
                 * SNMP Interface not found with this nodeId, create new
                 * interface
                 */
                LOG.debug("updateSnmpInfo: SNMP interface index {} not in database, creating new interface object.", ifIndex);
                dbSnmpIfEntry = DbSnmpInterfaceEntry.create(node.getNodeId(), ifIndex);
                newSnmpIfTableEntry = true;
            }

            /*
             * Create SNMP interface entry representing latest information
             * retrieved for the interface via the collector
             */
            final DbSnmpInterfaceEntry currSnmpIfEntry = DbSnmpInterfaceEntry.create(node.getNodeId(), ifIndex);

            // Find the ifTable entry for this interface
            final IfTable ift = snmpc.getIfTable();
            IfTableEntry ifte = null;
            for (final IfTableEntry current : ift) {
                // index
                final Integer sint = current.getIfIndex();
                if (sint != null) {
                    if (ifIndex == sint.intValue()) {
                        ifte = current;
                        break;
                    }
                }
            }

            // Make sure we have a valid IfTableEntry object
            if (ifte == null && ifIndex == CapsdConfig.LAME_SNMP_HOST_IFINDEX) {
                LOG.debug("updateSnmpInfo: interface {} appears to be a lame SNMP host", str(snmpc.getCollectorTargetAddress()));
            } else if (ifte != null) {
                /*
                 * IP address and netmask
                 *
                 * WARNING: IfSnmpCollector.getIfAddressAndMask() ONLY returns
                 * the FIRST IP address and mask for a given interface as
                 * specified in the ipAddrTable.
                 */
                InetAddress[] aaddrs = snmpc.getIfAddressAndMask(ifIndex);


                if (aaddrs == null) {
                    // disable collection on interface with no ip address by default
                    currSnmpIfEntry.setCollect("N");
                } else {
                    // mark the interface is collection enable
                    currSnmpIfEntry.setCollect("C");

                    // netmask
                    if (aaddrs[1] != null) {
                        LOG.debug("updateSnmpInfo: interface {} has netmask: {}", str(aaddrs[0]), str(aaddrs[1]));
                        currSnmpIfEntry.setNetmask(aaddrs[1]);
                    }

                } 

                // type
                Integer sint = ifte.getIfType();
                currSnmpIfEntry.setType(sint.intValue());

                // description
                String str = ifte.getIfDescr();
                LOG.debug("updateSnmpInfo: {} has ifDescription: {}", ifaddr, str);
                if (str != null && str.length() > 0) {
                    currSnmpIfEntry.setDescription(str);
                }

                String physAddr = ifte.getPhysAddr();

                LOG.debug("updateSnmpInfo: {} has phys address: -{}-", ifaddr, physAddr);

                if (physAddr != null && physAddr.length() == 12) {
                    currSnmpIfEntry.setPhysicalAddress(physAddr);
                }

                // speed
                Long speed = snmpc.getInterfaceSpeed(ifIndex);

                //set the default speed to 10MB if not retrievable.
                currSnmpIfEntry.setSpeed((speed == null
                        ? 10000000L : speed.longValue())); 

                // admin status
                sint = ifte.getIfAdminStatus();
                currSnmpIfEntry.setAdminStatus(sint == null ? 0 : sint.intValue());

                // oper status
                sint = ifte.getIfOperStatus();
                currSnmpIfEntry.setOperationalStatus(sint == null ? 0 : sint.intValue());

                // name (from interface extensions table)
                String ifName = snmpc.getIfName(ifIndex);
                if (ifName != null && ifName.length() > 0) {
                    currSnmpIfEntry.setName(ifName);
                }

                // alias (from interface extensions table)
                String ifAlias = snmpc.getIfAlias(ifIndex);
                if (ifAlias != null) {
                    currSnmpIfEntry.setAlias(ifAlias);
                } else {
                    currSnmpIfEntry.setAlias("");
                }		    

            } // end if valid ifTable entry

            // Update any fields which have changed
            // dbSnmpIfEntry.updateIfIndex(currSnmpIfEntry.getIfIndex());
            dbSnmpIfEntry.updateNetmask(currSnmpIfEntry.getNetmask());
            dbSnmpIfEntry.updatePhysicalAddress(currSnmpIfEntry.getPhysicalAddress());
            dbSnmpIfEntry.updateDescription(currSnmpIfEntry.getDescription());
            dbSnmpIfEntry.updateName(currSnmpIfEntry.getName());
            dbSnmpIfEntry.updateType(currSnmpIfEntry.getType());
            dbSnmpIfEntry.updateSpeed(currSnmpIfEntry.getSpeed());
            dbSnmpIfEntry.updateAdminStatus(currSnmpIfEntry.getAdminStatus());
            dbSnmpIfEntry.updateOperationalStatus(currSnmpIfEntry.getOperationalStatus());
            dbSnmpIfEntry.updateAlias(currSnmpIfEntry.getAlias());
            dbSnmpIfEntry.updateCollect(currSnmpIfEntry.getCollect());

            /*
             * If this is a new interface or if any of the following
             * key fields have changed set the m_snmpIfTableChangedFlag
             * variable to TRUE. This will potentially trigger an event
             * which will cause the poller to reinitialize the primary
             * SNMP interface for the node.
             */
            // dbSnmpIfEntry.hasIfIndexChanged() ||
            if (!m_snmpIfTableChangedFlag && newSnmpIfTableEntry
                    || dbSnmpIfEntry.hasTypeChanged()
                    || dbSnmpIfEntry.hasNameChanged()
                    || dbSnmpIfEntry.hasDescriptionChanged()
                    || dbSnmpIfEntry.hasPhysicalAddressChanged()
                    || dbSnmpIfEntry.hasAliasChanged()) {
                m_snmpIfTableChangedFlag = true;
            }

            // Update the database
            dbSnmpIfEntry.store(dbc);

            // end if complete snmp info available
        } else if (snmpc != null && snmpc.hasIpAddrTable() && ifIndex != -1) {
            LOG.debug("updateSnmpInfo: updating SNMP interface for nodeId={}/ifIndex={}/ipAddr={} based on ipAddrTable only - No ifTable available", node.getNodeId(), ifIndex, ifaddr);

            // Create and load SNMP Interface entry from the database
            DbSnmpInterfaceEntry dbSnmpIfEntry =
                    DbSnmpInterfaceEntry.get(dbc, node.getNodeId(), ifIndex);
            if (dbSnmpIfEntry == null) {
                /*
                 * SNMP Interface not found with this nodeId, create new
                 * interface
                 */
                LOG.debug("updateSnmpInfo: SNMP interface index {} not in database, creating new interface object.", ifIndex);
                dbSnmpIfEntry = DbSnmpInterfaceEntry.create(node.getNodeId(),
                                                            ifIndex);
            }

            /*
             * Create SNMP interface entry representing latest information
             * retrieved for the interface via the collector
             */
            DbSnmpInterfaceEntry.create(node.getNodeId(), ifIndex);

            // Update the database
            dbSnmpIfEntry.store(dbc);
            // end if partial snmp info available
        } else if (snmpc != null) {
            LOG.debug("updateSnmpInfo: updating SNMP interface for nodeId={}/ipAddr={} based on ip address only - No ipAddrTable available", node.getNodeId(), ifaddr);

            // Create and load SNMP Interface entry from the database

            DbSnmpInterfaceEntry dbSnmpIfEntry = DbSnmpInterfaceEntry.get(dbc, node.getNodeId(), CapsdConfig.LAME_SNMP_HOST_IFINDEX);
            if (dbSnmpIfEntry == null) {
                /*
                 * SNMP Interface not found with this nodeId, create new
                 * interface
                 */
                LOG.debug("updateSnmpInfo: SNMP interface index {} not in database, creating new interface object.", CapsdConfig.LAME_SNMP_HOST_IFINDEX);
                dbSnmpIfEntry = DbSnmpInterfaceEntry.create(node.getNodeId(), CapsdConfig.LAME_SNMP_HOST_IFINDEX);

            }

            /*
             * Create SNMP interface entry representing latest information
             * retrieved for the interface via the collector
             */
            DbSnmpInterfaceEntry.create(node.getNodeId(), ifIndex);

            // Update the database
            dbSnmpIfEntry.store(dbc);
        }
    }

    /**
     * This method is responsible for reparenting an interface's database table
     * entries under its new node identifier. The following tables are updated:
     * 
     * ipInterface snmpInterface ifServices
     * 
     * @param dbc
     *            Database connection
     * @param ifAddr
     *            Interface to be reparented.
     * @param newNodeId
     *            Interface's new node identifier
     * @param oldNodeId
     *            Interfaces' old node identifier
     * 
     * @throws SQLException
     *             if a database error occurs during reparenting.
     */
    private static void reparentInterface(final Connection dbc, final InetAddress ifAddr, final int ifIndex, final int newNodeId, final int oldNodeId) throws SQLException {
        String ipaddr = str(ifAddr);
        final DBUtils d = new DBUtils(RescanProcessor.class);

        try {
            final PreparedStatement ifLookupStmt = dbc.prepareStatement(SQL_DB_REPARENT_IP_INTERFACE_LOOKUP);
            d.watch(ifLookupStmt);
            final PreparedStatement ifDeleteStmt = dbc.prepareStatement(SQL_DB_REPARENT_IP_INTERFACE_DELETE);
            d.watch(ifDeleteStmt);
            final PreparedStatement ipInterfaceStmt = dbc.prepareStatement(SQL_DB_REPARENT_IP_INTERFACE);
            d.watch(ipInterfaceStmt);
            final PreparedStatement snmpIfLookupStmt = dbc.prepareStatement(SQL_DB_REPARENT_SNMP_IF_LOOKUP);
            d.watch(snmpIfLookupStmt);
            final PreparedStatement snmpIfDeleteStmt = dbc.prepareStatement(SQL_DB_REPARENT_SNMP_IF_DELETE);
            d.watch(snmpIfDeleteStmt);
            final PreparedStatement snmpInterfaceStmt = dbc.prepareStatement(SQL_DB_REPARENT_SNMP_INTERFACE);
            d.watch(snmpInterfaceStmt);
            final PreparedStatement ifServicesLookupStmt = dbc.prepareStatement(SQL_DB_REPARENT_IF_SERVICES_LOOKUP);
            d.watch(ifServicesLookupStmt);
            final PreparedStatement ifServicesDeleteStmt = dbc.prepareStatement(SQL_DB_REPARENT_IF_SERVICES_DELETE);
            d.watch(ifServicesDeleteStmt);
            final PreparedStatement ifServicesStmt = dbc.prepareStatement(SQL_DB_REPARENT_IF_SERVICES);
            d.watch(ifServicesStmt);

            LOG.debug("reparentInterface: reparenting address/ifIndex/nodeID: {}/{}/{}", ipaddr, ifIndex, newNodeId);

            /*
             * SNMP interface
             *
             * NOTE: Only reparent SNMP interfaces if we have valid ifIndex
             */
            if (ifIndex < 1) {
                LOG.debug("reparentInterface: don't have a valid ifIndex, skipping snmpInterface table reparenting.");
            } else {
                /*
                 * NOTE: Now that the snmpInterface table is uniquely keyed
                 * by nodeId and ifIndex we must only reparent the
                 * old entry if there isn't already an entry with
                 * the same nodeid/ifindex pairing. If it can't
                 * be reparented it will be deleted.
                 */

                /*
                 * Look for matching nodeid/ifindex for the entry to be
                 * reparented
                 */
                snmpIfLookupStmt.setInt(1, newNodeId);
                snmpIfLookupStmt.setInt(2, ifIndex);
                final ResultSet rs = snmpIfLookupStmt.executeQuery();
                d.watch(rs);
                if (rs.next()) {
                    /*
                     * Looks like we got a match so just delete
                     * the entry from the old node
                     */
                    LOG.debug("reparentInterface: interface with ifindex {} already exists under new node {} in snmpinterface table, deleting from under old node {}", ifIndex, newNodeId, oldNodeId);

                    snmpIfDeleteStmt.setInt(1, oldNodeId);
                    snmpIfDeleteStmt.setInt(2, ifIndex);

                    snmpIfDeleteStmt.executeUpdate();
                } else {
                    /*
                     * Update the 'snmpinterface' table entry so that this
                     * interface's nodeID is set to the value of reparentNodeID
                     */
                    LOG.debug("reparentInterface: interface with ifindex {} does not yet exist under new node {} in snmpinterface table, reparenting.", ifIndex, newNodeId);

                    snmpInterfaceStmt.setInt(1, newNodeId);
                    snmpInterfaceStmt.setInt(2, oldNodeId);
                    snmpInterfaceStmt.setInt(3, ifIndex);

                    // execute and log
                    snmpInterfaceStmt.executeUpdate();
                }
            }

            // Look for matching nodeid/ifindex for the entry to be reparented
            ifLookupStmt.setInt(1, newNodeId);
            ifLookupStmt.setString(2, ipaddr);
            final ResultSet rs = ifLookupStmt.executeQuery();
            d.watch(rs);
            if (rs.next()) {
                /*
                 * Looks like we got a match so just delete
                 * the entry from the old node
                 */
                LOG.debug("reparentInterface: interface with ifindex {} already exists under new node {} in ipinterface table, deleting from under old node {}", ifIndex, newNodeId, oldNodeId);

                ifDeleteStmt.setInt(1, oldNodeId);
                ifDeleteStmt.setString(2, ipaddr);

                ifDeleteStmt.executeUpdate();
            } else {
                /*
                 * Update the 'ipinterface' table entry so that this
                 * interface's nodeID is set to the value of reparentNodeID
                 */
                LOG.debug("reparentInterface: interface with ifindex {} does not yet exist under new node {} in ipinterface table, reparenting.", ifIndex, newNodeId);

                ipInterfaceStmt.setInt(1, newNodeId);
                ipInterfaceStmt.setInt(2, oldNodeId);
                ipInterfaceStmt.setString(3, ipaddr);

                // execute and log
                ipInterfaceStmt.executeUpdate();
            }

            // Look for matching nodeid/ifindex for the entry to be reparented
            ifServicesLookupStmt.setInt(1, newNodeId);
            ifServicesLookupStmt.setString(2, ipaddr);
            ifServicesLookupStmt.setInt(3, ifIndex);
            final ResultSet rsServicesLookup = ifServicesLookupStmt.executeQuery();
            d.watch(rsServicesLookup);
            if (rsServicesLookup.next()) {
                /*
                 * Looks like we got a match so just delete
                 * the entry from the old node
                 */
                LOG.debug("reparentInterface: interface with ifindex {} already exists under new node {} in ifservices table, deleting from under old node {}", ifIndex, newNodeId, oldNodeId);

                ifServicesDeleteStmt.setInt(1, oldNodeId);
                ifServicesDeleteStmt.setString(2, ipaddr);

                ifServicesDeleteStmt.executeUpdate();
            } else {
                /*
                 * Update the 'snmpinterface' table entry so that this
                 * interface's nodeID is set to the value of reparentNodeID
                 */
                LOG.debug("reparentInterface: interface with ifindex {} does not yet exist under new node {} in ifservices table, reparenting.", ifIndex, newNodeId);

                /*
                 * Update the 'nodeID' field of all 'ifservices' table entries
                 * for the reparented interfaces.
                 */
                ifServicesStmt.setInt(1, newNodeId);
                ifServicesStmt.setInt(2, oldNodeId);
                ifServicesStmt.setString(3, ipaddr);

                // execute and log
                ifServicesStmt.executeUpdate();
            }

            LOG.debug("reparentInterface: reparented {} : ifIndex: {} : oldNodeID: {} newNodeID: {}", ipaddr, ifIndex, oldNodeId, newNodeId);
        } catch (final SQLException sqlE) {
            LOG.error("SQLException while reparenting addr/ifindex/nodeid {}/{}/{}", ipaddr, ifIndex, oldNodeId);
            throw sqlE;
        } finally {
            d.cleanUp();
        }
    }

    /**
     * Builds a list of InetAddress objects representing each of the interfaces
     * from the collector map object which support SNMP and have a valid ifIndex
     * and have an IfType of loopback.
     * 
     * This is part of a feature to choose a non 127.*.*.* loopback address as
     * the primary SNMP interface.
     * 
     * @param collectorMap
     *            Map of IfCollector objects containing data collected from all
     *            of the node's interfaces.
     * @param snmpc
     *            Reference to SNMP collection object
     * 
     * @return List of InetAddress objects.
     */
    private static List<InetAddress> buildLBSnmpAddressList(final Map<String, IfCollector> collectorMap, final IfSnmpCollector snmpc) {
        final List<InetAddress> addresses = new ArrayList<InetAddress>();

        // Verify that we have SNMP info
        if (snmpc == null) {
            LOG.debug("buildLBSnmpAddressList: no SNMP info available...");
            return addresses;
        }
        if (!snmpc.hasIfTable()) {
            LOG.debug("buildLBSnmpAddressList: no SNMP ifTable available...");
            return addresses;
        }

        /*
         * To be eligible to be the primary SNMP interface for a node:
         * 
         * 1. The interface must support SNMP
         * 2. The interface must have a valid ifIndex.
         */
        for (final IfCollector ifc : collectorMap.values()) {
            // Add eligible target.
            final InetAddress ifaddr = ifc.getTarget();

            if (addresses.contains(ifaddr) == false) {
                if (SuspectEventProcessor.supportsSnmp(ifc.getSupportedProtocols()) && SuspectEventProcessor.hasIfIndex(ifaddr, snmpc) && SuspectEventProcessor.getIfType(ifaddr, snmpc) == 24) {
                    LOG.debug("buildLBSnmpAddressList: adding target interface {} temporarily marked as primary!", str(ifaddr));
                    addresses.add(ifaddr);
                }
            }

            // Now go through list of sub-targets
            if (ifc.hasAdditionalTargets()) {
                final Map<InetAddress, List<SupportedProtocol>> subTargets = ifc.getAdditionalTargets();
                for (final Map.Entry<InetAddress, List<SupportedProtocol>> entry : subTargets.entrySet()) {
                    final InetAddress xifaddr = entry.getKey();
                    final List<SupportedProtocol> protocols = entry.getValue();
                    if (addresses.contains(xifaddr) == false) {
                        if (SuspectEventProcessor.supportsSnmp(protocols) && SuspectEventProcessor.hasIfIndex(xifaddr, snmpc) && SuspectEventProcessor.getIfType(xifaddr, snmpc) == 24) {
                            LOG.debug("buildLBSnmpAddressList: adding subtarget interface {} temporarily marked as primary!", str(xifaddr));
                            addresses.add(xifaddr);
                        }
                    }
                }
            }
        }

        return addresses;
    }

    /**
     * Builds a list of InetAddress objects representing each of the interfaces
     * from the collector map object which support SNMP and have a valid
     * ifIndex.
     * 
     * @param collectorMap
     *            Map of IfCollector objects containing data collected from all
     *            of the node's interfaces.
     * @param snmpc
     *            Reference to SNMP collection object
     * 
     * @return List of InetAddress objects.
     */
    private static List<InetAddress> buildSnmpAddressList(final Map<String, IfCollector> collectorMap, final IfSnmpCollector snmpc) {
        final List<InetAddress> addresses = new ArrayList<InetAddress>();

        // Verify that we have SNMP info
        if (snmpc == null) {
            LOG.debug("buildSnmpAddressList: no SNMP info available...");
            return addresses;
        }

        /*
         * To be eligible to be the primary SNMP interface for a node:
         * 
         * 1. The interface must support SNMP
         * 2. The interface must have a valid ifIndex.
         */
        for (final IfCollector ifc : collectorMap.values()) {
            // Add eligible target.
            final InetAddress ifaddr = ifc.getTarget();

            if (addresses.contains(ifaddr) == false) {
                if (SuspectEventProcessor.supportsSnmp(ifc.getSupportedProtocols()) && SuspectEventProcessor.hasIfIndex(ifaddr, snmpc)) {
                    LOG.debug("buildSnmpAddressList: adding target interface {} temporarily marked as primary!", str(ifaddr));
                    addresses.add(ifaddr);
                }
            }

            // Now go through list of sub-targets
            if (ifc.hasAdditionalTargets()) {
                final Map<InetAddress, List<SupportedProtocol>> subTargets = ifc.getAdditionalTargets();
                for (final Map.Entry<InetAddress,List<SupportedProtocol>> entry : subTargets.entrySet()) {
                    final InetAddress xifaddr = entry.getKey();
                    final List<SupportedProtocol> protocols = entry.getValue();
                    // Add eligible subtargets.
                    if (addresses.contains(xifaddr) == false) {
                        if (SuspectEventProcessor.supportsSnmp(protocols) && SuspectEventProcessor.hasIfIndex(xifaddr, snmpc)) {
                            LOG.debug("buildSnmpAddressList: adding subtarget interface {} temporarily marked as primary!", str(xifaddr));
                            addresses.add(xifaddr);
                        }
                    }
                }
            }
        }

        return addresses;
    }

    /**
     * This method is responsible for determining the primary IP interface for
     * the node being rescanned.
     * 
     * @param collectorMap
     *            Map of IfCollector objects containing data collected from all
     *            of the node's interfaces.
     * 
     * @return InetAddress The primary IP interface for the node or null if a
     *         primary interface for the node could not be determined.
     */
    private static InetAddress determinePrimaryIpInterface(final Map<String, IfCollector> collectorMap) {
        InetAddress primaryIf = null;

        for (final IfCollector ifc : collectorMap.values()) {
            InetAddress currIf = ifc.getTarget();

            if (primaryIf == null) {
                primaryIf = currIf;
                continue;
            } else {
                // Test the target interface of the collector first.
                primaryIf = SuspectEventProcessor.compareAndSelectPrimary(currIf, primaryIf);

                // Now test each of the collected subtargets
                if (ifc.hasAdditionalTargets()) {
                    final Map<InetAddress, List<SupportedProtocol>> subTargets = ifc.getAdditionalTargets();
                    for (final InetAddress subIf : subTargets.keySet()) {
                        currIf = subIf;
                        primaryIf = SuspectEventProcessor.compareAndSelectPrimary(currIf, primaryIf);
                    }
                }
            }
        }

        if (LOG.isDebugEnabled()) {
            if (primaryIf != null) {
                LOG.debug("determinePrimaryIpInterface: selected primary interface: {}", str(primaryIf));
            } else {
                LOG.debug("determinePrimaryIpInterface: no primary interface found");
            }
        }
        return primaryIf;
    }

    /**
     * Primarily, this method is responsible for assigning the node's nodeLabel
     * value using information collected from the node's various interfaces.
     * Additionally, if the node talks NetBIOS/SMB, then the node's NetBIOS name
     * and operating system fields are assigned.
     * 
     * @param collectorMap
     *            Map of IfCollector objects, one per interface.
     * @param dbNodeEntry
     *            Node entry, as it exists in the database.
     * @param currNodeEntry
     *            Current node entry, as collected during the current rescan.
     * @param currPrimarySnmpIf
     *            Primary SNMP interface, as determined from the collection
     *            retrieved during the current rescan.
     */
    private static void setNodeLabelAndSmbInfo(final Map<String, IfCollector> collectorMap, final DbNodeEntry dbNodeEntry, final DbNodeEntry currNodeEntry, final InetAddress currPrimarySnmpIf) {
        boolean labelSet = false;

        /*
         * We are going to change the order in which labels are assigned.
         * First, we check DNS - the hostname of the primary interface.
         * Then we check SMB - next SNMP sysName - and finally IP address
         * This is different then in 1.0 - when SMB came first.
         */

        InetAddress primaryIf = null;

        if (!labelSet) {
            /*
             * If no label is set, attempt to get the hostname for the primary
             * SNMP interface.
             * Note: this was wrong prior to 1.0.1 - the method
             * determinePrimaryIpInterface
             * would return the lowest numbered interface, not necessarily the
             * primary SNMP interface.
             */
            if (currPrimarySnmpIf != null) {
                primaryIf = currPrimarySnmpIf;
            } else {
                primaryIf = determinePrimaryIpInterface(collectorMap);
            }
            if (primaryIf == null) {
                LOG.error("setNodeLabelAndSmbInfo: failed to find primary interface...");
            } else {
                String hostName = primaryIf.getHostName();
                if (!hostName.equals(str(primaryIf))) {
                    labelSet = true;

                    currNodeEntry.setLabel(hostName);
                    currNodeEntry.setLabelSource(NodeLabelSource.HOSTNAME);
                }
            }
        }

        IfSmbCollector savedSmbcRef = null;

        // Does the node entry in database have a NetBIOS name?
        if (dbNodeEntry.getNetBIOSName() != null) {
            /*
             * Yes it does, so search through collected info for all
             * interfaces and see if any have a NetBIOS name
             * which matches the existing one in the database
             */
            final Collection<IfCollector> values = collectorMap.values();
            final Iterator<IfCollector> iter = values.iterator();
            while (iter.hasNext() && !labelSet) {
                final IfCollector ifc = iter.next();
                final IfSmbCollector smbc = ifc.getSmbCollector();
                if (smbc != null) {
                    if (smbc.getNbtName() != null) {
                        /*
                         * Save reference to first IfSmbCollector object
                         * for future use.
                         */
                        savedSmbcRef = smbc;

                        final String netbiosName = smbc.getNbtName().toUpperCase();
                        if (netbiosName.equals(dbNodeEntry.getNetBIOSName())) {
                            // Found a match.
                            labelSet = true;

                            currNodeEntry.setLabel(netbiosName);
                            currNodeEntry.setLabelSource(NodeLabelSource.NETBIOS);
                            currNodeEntry.setNetBIOSName(netbiosName);

                            if (smbc.getDomainName() != null) {
                                currNodeEntry.setDomainName(smbc.getDomainName());
                            }
                        }
                    }
                }
            }
        } else {
            /*
             * No it does not, attempt to find an interface
             * collector that does have a NetBIOS name and
             * save a reference to that collector
             */
            for (final IfCollector ifc : collectorMap.values()) {
                final IfSmbCollector smbc = ifc.getSmbCollector();
                if (smbc != null && smbc.getNbtName() != null) {
                    savedSmbcRef = smbc;
                }
            }
        }

        /*
         * If node label has not yet been set and SMB info is available
         * use that info to set the node label and NetBIOS name
         */
        if (!labelSet && savedSmbcRef != null) {
            labelSet = true;

            currNodeEntry.setLabel(savedSmbcRef.getNbtName());
            currNodeEntry.setLabelSource(NodeLabelSource.NETBIOS);
            currNodeEntry.setNetBIOSName(currNodeEntry.getLabel());

            if (savedSmbcRef.getDomainName() != null) {
                currNodeEntry.setDomainName(savedSmbcRef.getDomainName());
            }
        }

        /*
         * If we get this far no IP hostname or SMB info was available. Next we
         * want to use MIB-II sysName for the node label. The primary SNMP
         * interface has already been determined so use it if available.
         */
        if (!labelSet && currPrimarySnmpIf != null) {
            /*
             * We prefer to use the collector for the primary SNMP interface
             * however a collector for the primary SNMP interface may not exist
             * in the map if a node has only recently had SNMP support enabled
             * or if the new primary SNMP interface was only recently added to
             * the node. At any rate if it exists use it, if not use the
             * first collector which supports SNMP.
             */
            final String currPrimarySnmpAddress = str(currPrimarySnmpIf);
            IfCollector ifc = currPrimarySnmpAddress == null? null : collectorMap.get(currPrimarySnmpAddress);
            if (ifc == null) {
                final Collection<IfCollector> collectors = collectorMap.values();
                final Iterator<IfCollector> iter = collectors.iterator();
                while (iter.hasNext()) {
                    ifc = iter.next();
                    if (ifc.getSnmpCollector() != null) {
                        break;
                    }
                }
            }

            // Sanity check
            if (ifc == null || ifc.getSnmpCollector() == null) {
                LOG.warn("setNodeLabelAndSmbInfo: primary SNMP interface set to {} but no SNMP collector found.", currPrimarySnmpAddress);
            } else {
                final IfSnmpCollector snmpc = ifc.getSnmpCollector();
                final SystemGroup sysgrp = snmpc.getSystemGroup();

                final String str = sysgrp.getSysName();
                if (str != null && str.length() > 0) {
                    labelSet = true;
                    currNodeEntry.setLabel(str);
                    currNodeEntry.setLabelSource(NodeLabelSource.SYSNAME);
                }
            }
        }

        if (!labelSet) {
            /*
             * If we get this far no SNMP info was available so we will default
             * to the IP address of the primary interface.
             */
            if (primaryIf != null) {
                currNodeEntry.setLabel(str(primaryIf));
                currNodeEntry.setLabelSource(NodeLabelSource.ADDRESS);
            } else {
                /*
                 * If all else fails, just use the current values from
                 * the database.
                 */
                currNodeEntry.setLabel(dbNodeEntry.getLabel());
                currNodeEntry.setLabelSource(dbNodeEntry.getLabelSource());
            }
        }
    }

    /**
     * Utility method used to determine if the specified node has been marked as
     * deleted in the node table.
     * 
     * @param dbc
     *            Database connection.
     * @param nodeId
     *            Node identifier to check
     * 
     * @return TRUE if node has been marked as deleted, FALSE otherwise.
     */
    private static boolean isNodeDeleted(final Connection dbc, final int nodeId) throws SQLException {
        boolean nodeDeleted = false;

        /*
         * Prepare & execute the SQL statement to retrieve the 'nodetype' field
         * from the node table for the specified nodeid.
         */
        final DBUtils d = new DBUtils(RescanProcessor.class);
        try {
            final PreparedStatement stmt = dbc.prepareStatement(SQL_DB_RETRIEVE_NODE_TYPE);
            d.watch(stmt);
            stmt.setInt(1, nodeId);
            final ResultSet rs = stmt.executeQuery();
            d.watch(rs);
            rs.next();
            final String nodeTypeStr = rs.getString(1);
            if (!rs.wasNull()) {
                if (NodeType.DELETED.toString().equals(nodeTypeStr.charAt(0))) {
                    nodeDeleted = true;
                }
            }
        } finally {
            d.cleanUp();
        }

        return nodeDeleted;
    }

    /**
     * This method is used to verify if each interface on a node stored in the
     * database is in the specified SNMP data collection.
     * 
     * @param dbInterfaces
     *            the ipInterfaces on a node stored in the database
     * @param snmpc
     *            IfSnmpCollector object containing SNMP collected ipAddrTable
     *            information.
     * 
     * @return True if each ipInterface is contained in the ipAddrTable of the
     *         specified SNMP collection.
     * 
     */
    private static boolean areDbInterfacesInSnmpCollection(final DbIpInterfaceEntry[] dbInterfaces, final IfSnmpCollector snmpc) {
        // Sanity check...null parms?
        if (dbInterfaces == null || snmpc == null) {
            LOG.error("areDbInterfacesInSnmpCollection: empty dbInterfaces or IfSnmpCollector.");
            return false;
        }

        // SNMP collection successful?
        if (!snmpc.hasIpAddrTable()) {
            LOG.error("areDbInterfacesInSnmpCollection: SNMP Collector failed.");
            return false;
        }

        // Verify that SNMP collection contains ipAddrTable entries
        IpAddrTable ipAddrTable = null;

        if (snmpc.hasIpAddrTable()) {
            ipAddrTable = snmpc.getIpAddrTable();
        }

        if (ipAddrTable == null) {
            LOG.error("areDbInterfacesInSnmpCollection: null ipAddrTable in the SNMP collection");
            return false;
        }

        final List<InetAddress> ipAddrList = ipAddrTable.getIpAddresses();

        /*
         * Loop through the interface table entries until there are no more
         * entries or we've found a match
         */
        for (final DbIpInterfaceEntry dbInterface : dbInterfaces) {
            final InetAddress ipaddr = dbInterface.getIfAddress();

            // Skip non-IP or loopback interfaces
            final String ipaddrString = str(ipaddr);
            if (ipaddrString == null || s_ZERO_ZERO_ZERO_ZERO.equals(ipaddrString) || ipaddr.isLoopbackAddress()) {
                continue;
            }

            boolean found = false;
            for (final InetAddress addr : ipAddrList) {
                // Skip non-IP or loopback interfaces
                final String addrString = str(addr);
                if (addrString == null || s_ZERO_ZERO_ZERO_ZERO.equals(addrString) || addr.isLoopbackAddress()) {
                    continue;
                }

                if (ipaddrString.equals(addrString)) {
                    LOG.debug("areDbInterfacesInSnmpCollection: found match for ipaddress: {}", ipaddrString);
                    return true;
                }
            }

            if (!found) {
                LOG.debug("areDbInterfacesInSnmpCollection: ipaddress : {} not in the SNMP collection. SNMP collection may not be usable.", ipaddrString);
                return false;
            }
        }

        return true;
    }

    /**
     * This is where all the work of the class is done.
     */
    @Override
    public void run() {
        // perform rescan of the node
        final DbNodeEntry dbNodeEntry = getNode();

        if (dbNodeEntry == null) {
            return;
        }

        if (dbNodeEntry.getForeignSource() != null) {
            LOG.info("Skipping rescan of node {} since it was imported with foreign source {}", getNodeId(), dbNodeEntry.getForeignSource());
            return;
        }

        LOG.debug("start rescanning node: {}", getNodeId());

        final DbIpInterfaceEntry[] dbInterfaces = getInterfaces(dbNodeEntry);

        if (dbInterfaces == null) {
            LOG.debug("no interfaces found in the database to rescan for node: {}", getNodeId());
            return;
        }

        // this indicates whether or not we found an iface the responds to snmp
        boolean doesSnmp = true;

        IpAddrTable ipAddTable = null;
        List<InetAddress> prevAddrList = null;
        boolean snmpcAgree = false;
        boolean gotSnmpc = false;
        Map<String, IfCollector> collectorMap = new HashMap<String, IfCollector>();
        final Map<String, IfCollector> nonSnmpCollectorMap = new HashMap<String, IfCollector>();
        final Set<InetAddress> probedAddrs = new HashSet<InetAddress>();

        boolean gotSnmpCollection = false;
        final DbIpInterfaceEntry oldPrimarySnmpInterface = DbNodeEntry.getPrimarySnmpInterface(dbInterfaces);
        if (oldPrimarySnmpInterface != null) {
            gotSnmpCollection = scanPrimarySnmpInterface(oldPrimarySnmpInterface, collectorMap, probedAddrs);
        }

        if (!gotSnmpCollection) {
            /*
             * Run collector for each retrieved interface and add result
             * to a collector map.
             */
            for (int i = 0; i < dbInterfaces.length; i++) {
                LOG.info("run: Running collector for interface {} of {}", i, dbInterfaces.length);
                final InetAddress ifaddr = dbInterfaces[i].getIfAddress();
                final String ifaddrString = str(ifaddr);

                /*
                 * collect the information from the interface.
                 * NOTE: skip '127.*.*.*' and '0.0.0.0' addresses.
                 */
                if (!scannableInterface(dbInterfaces, ifaddr)) {
                    LOG.debug("run: skipping scan of address: {}", ifaddrString);
                    continue;
                }

                if (ifaddrString == null) {
                    LOG.debug("run: unable to scan inet address: {}", ifaddr);
                    continue;
                }

                LOG.debug("running collection for {}", ifaddrString);

                final IfCollector collector = new IfCollector(m_pluginManager, ifaddr, true, probedAddrs);
                collector.run();

                final IfSnmpCollector snmpc = collector.getSnmpCollector();
                if (snmpc != null) {
                    gotSnmpc = true;
                }
                if (snmpc != null && snmpc.hasIpAddrTable() && snmpc.getIfIndex(snmpc.getCollectorTargetAddress()) != -1) {
                    if (areDbInterfacesInSnmpCollection(dbInterfaces, snmpc)) {
                        collectorMap.put(ifaddrString, collector);
                        gotSnmpCollection = true;
                        LOG.debug("SNMP data collected via {}", ifaddrString);
                        LOG.debug("Adding {} to collectorMap for node: {}", ifaddrString, getNodeId());
                        snmpcAgree = false;
                        break;
                    } else if (ipAddTable == null) {
                        snmpcAgree = true;
                        collectorMap.put(ifaddrString, collector);
                        ipAddTable = snmpc.getIpAddrTable();
                        prevAddrList = ipAddTable.getIpAddresses();

                        if (LOG.isDebugEnabled()) {
                            LOG.debug("SNMP data collected via {} does not agree with database.  Tentatively adding to the collectorMap and continuing", ifaddrString);
                            for(final InetAddress a : prevAddrList) {
                                LOG.debug("IP address in list = {}", a);
                            }
                        }
                    } else if (ipAddTable != null && snmpcAgree == true) {
                        ipAddTable = snmpc.getIpAddrTable();
                        final List<InetAddress> addrList = ipAddTable.getIpAddresses();

                        boolean listMatch = addrList.size() == prevAddrList.size() && addrList.containsAll(prevAddrList);
                        if (listMatch) {
                            LOG.debug("Current and previous address lists match");
                        } else {
                            LOG.debug("Current and previous address lists DO NOT match");
                            snmpcAgree = false;
                        }
                        collector.deleteSnmpCollector();
                    }
                    if (snmpcAgree == false) {
                        LOG.debug("SNMP data collected via {} does not agree with database or with other interface(s) on this node.", ifaddrString);
                    }
                } else {
                    /*
                     * Build a non-SNMP collectorMap, skipping 127.*.*.*
                     * and 0.0.0.0
                     */
                    nonSnmpCollectorMap.put(ifaddrString, collector);
                    LOG.debug("Adding {} to nonSnmpCollectorMap for node: {}", ifaddrString, getNodeId());
                }
            }
        }

        if (!gotSnmpCollection && snmpcAgree == false) {
            /*
             * We didn't get a collection from a primary snmp interface,
             * and we didn't get a collection that agrees with the db, and
             * multiple interface collections don't agree with each other.
             * First check for lame SNMP host, otherwise use the
             * nonSnmpCollectorMap and set doesSnmp = false
             */
            collectorMap = nonSnmpCollectorMap;
            if (nonSnmpCollectorMap.size() == 1 && gotSnmpc) {
                doesSnmp = true;
                LOG.debug("node {} appears to be a lame SNMP host... Proceeding", getNodeId());
            } else {
                doesSnmp = false;
                if (LOG.isDebugEnabled()) {
                    if (gotSnmpc == false) {
                        LOG.debug("Could not collect SNMP data for node: {}", getNodeId());
                    } else {
                        LOG.debug("Not using SNMP data for node: {}.  Collection does not agree with database.", getNodeId());
                    }
                }
            }
        } else if (snmpcAgree == true) {
            /*
             * We didn't get a collection from a primary snmp interface,
             * and we didn't get a collection that agrees with the db, but
             * all collections we DID get agree with each other.
             * May want to create an event here
             */
            LOG.debug("SNMP collection for node: {} does not agree with database, but there is no conflict among the interfaces on this node which respond to SNMP. Proceeding...", getNodeId());
            m_eventList.add(createSnmpConflictsWithDbEvent(dbNodeEntry));
        }

        final DBUtils d = new DBUtils();

        // Update the database
        boolean updateCompleted = false;
        try {
            /*
             * Synchronize on the Capsd sync lock so we can check if
             * the interface is already in the database and perform
             * the necessary inserts in one atomic operation
             *	
             * The SuspectEventProcessor class is also synchronizing on this
             * lock prior to performing database inserts or updates.
             */
            LOG.debug("Waiting for capsd dbLock to process {}", getNodeId());
            synchronized (Capsd.getDbSyncLock()) {
                LOG.debug("Got capsd dbLock. processing {}", getNodeId());
                // Get database connection
                final Connection dbc = DataSourceFactory.getInstance().getConnection();
                d.watch(dbc);

                /*
                 * There is a slight possibility that the node being rescanned
                 * has been deleted (due to reparenting) by another thread
                 * between the time this rescan was started and the database
                 * sync lock was grabbed. Verify that the current nodeid is
                 * still valid (ie, not deleted) before continuing.
                 */
                if (!isNodeDeleted(dbc, getNodeId())) {
                    // Update interface information
                    final Date now = new Date();
                    updateInterfaces(dbc, now, dbNodeEntry, collectorMap, doesSnmp);

                    if (doesSnmp) {
                        InetAddress oldPriIf = null;
                        if (oldPrimarySnmpInterface != null) {
                            oldPriIf = oldPrimarySnmpInterface.getIfAddress();
                        }
                        final InetAddress newSnmpPrimaryIf = updatePrimarySnmpInterface(dbc, dbNodeEntry, collectorMap, oldPriIf);
                        updateNode(dbc, now, dbNodeEntry, newSnmpPrimaryIf, dbInterfaces, collectorMap);
                    }
                    updateCompleted = true;
                    m_eventList.add(createRescanCompletedEvent(dbNodeEntry));
                }
            }
        } catch (final Throwable t) {
            LOG.error("Error updating records for node ID {}", getNodeId(), t);
        } finally {
            // Finished with the database connection, close it.
            d.cleanUp();

            // Remove the node we just scanned from the tracker set
            synchronized (s_queuedRescanTracker) {
                s_queuedRescanTracker.remove(getNodeId());
            }
        }

        // Send events associcatd with the rescan
        if (updateCompleted) {
            // Send all events created during rescan process to eventd
            for (final Event event : m_eventList) {
                try {
                    EventIpcManagerFactory.getIpcManager().sendNow(event);
                } catch (final Throwable t) {
                    LOG.warn("run: unexpected throwable exception caught while sending event", t);
                }
            }
        }

        LOG.debug("{}escan for node w/ nodeid {} completed.", (m_forceRescan ? "Forced r" : "R"), getNodeId());
    }

    /**
     * <p>scannableInterface</p>
     *
     * @param dbInterfaces an array of {@link org.opennms.netmgt.capsd.DbIpInterfaceEntry} objects.
     * @param ifaddr a {@link java.net.InetAddress} object.
     * @return a boolean.
     */
    protected static boolean scannableInterface(final DbIpInterfaceEntry[] dbInterfaces, final InetAddress ifaddr) {
        final String ifaddrString = str(ifaddr);
        if (ifaddrString == null) return false;

        final boolean localHostAddress = (ifaddr.isLoopbackAddress() && dbInterfaces.length > 1);
        final boolean nonIpAddress = s_ZERO_ZERO_ZERO_ZERO.equals(ifaddrString);
        final boolean scannable = !localHostAddress && !nonIpAddress;
        return scannable;
    }

    private int getNodeId() {
        return m_nodeId;
    }

    private boolean scanPrimarySnmpInterface(final DbIpInterfaceEntry oldPrimarySnmpInterface, final Map<String, IfCollector> collectorMap, final Set<InetAddress> probedAddrs) {
        /*
         * Run collector for DB primary snmp interface and add result
         * to a collector map.
         */
        final InetAddress ifaddr = oldPrimarySnmpInterface.getIfAddress();
        final String ifaddrString = str(ifaddr);

        if (ifaddrString == null) {
            LOG.info("old primary SNMP interface has an invalid address: {}", ifaddr);
            return false;
        }

        LOG.debug("running collection for DB primary SNMP interface {}", ifaddrString);
        final IfCollector collector = new IfCollector(m_pluginManager, ifaddr, true, probedAddrs);
        collector.run();

        final IfSnmpCollector snmpc = collector.getSnmpCollector();
        if (snmpc == null) {
            LOG.debug("SNMP Collector from DB primary SNMP interface is null");
            return false;
        }

        collectorMap.put(ifaddrString, collector);
        LOG.debug("SNMP data collected from DB primary SNMP interface {}", ifaddrString);
        if (!snmpc.hasIfTable()) {
            LOG.debug("SNMP Collector has no IfTable");
        }
        if (!snmpc.hasIpAddrTable() || snmpc.getIfIndex(snmpc.getCollectorTargetAddress()) == -1) {
            LOG.debug("SNMP Collector has no IpAddrTable. Assume its a lame SNMP host.");
        }

        return true;
    }

    private InetAddress updatePrimarySnmpInterface(final Connection dbc, final DbNodeEntry dbNodeEntry, final Map<String, IfCollector> collectorMap, final InetAddress oldPriIf) throws SQLException {
        /*
         * Now that all interfaces have been added to the
         * database we can update the 'primarySnmpInterface'
         * field of the ipInterface table. Necessary because
         * the IP address must already be in the database
         * to evaluate against a filter rule.
         *
         * First create a list of eligible loopback interfaces
         * and a list of all eligible interfaces. Test in the
         * following order:
         * 
         * 1) strict = true (interface must be part of a Collectd
         * package) and loopback.
         * 
         * 2) strict = true and all eligible interfaces.
         * strict = false and loopback.
         * 
         * 4) strict = false and all eligible interfaces.
         */
        FilterDaoFactory.getInstance().flushActiveIpAddressListCache();
        final IfSnmpCollector snmpc = findSnmpCollector(collectorMap);
        final List<InetAddress> snmpLBAddresses = buildLBSnmpAddressList(collectorMap, snmpc);
        final List<InetAddress> snmpAddresses = buildSnmpAddressList(collectorMap, snmpc);

        // first set the value of issnmpprimary for secondaries
        for (final InetAddress addr : snmpAddresses) {
            final String addrString = str(addr);
            if (m_capsdDbSyncer.isServiceCollectionEnabled(addrString, "SNMP")) {
                final DBUtils d = new DBUtils(RescanProcessor.class);
                try {
                    final PreparedStatement stmt = dbc.prepareStatement("UPDATE ipInterface SET isSnmpPrimary='S' WHERE nodeId=? AND ipAddr=? AND isManaged!='D'");
                    d.watch(stmt);
                    stmt.setInt(1, dbNodeEntry.getNodeId());
                    stmt.setString(2, addrString);
                    stmt.executeUpdate();
                    LOG.debug("updatePrimarySnmpInterface: updated {} to secondary.", addrString);
                } finally {
                    d.cleanUp();
                }
            }
        }

        InetAddress newSnmpPrimaryIf = m_capsdDbSyncer.determinePrimarySnmpInterface(snmpLBAddresses, true);
        String psiType = ConfigFileConstants.getFileName(ConfigFileConstants.COLLECTD_CONFIG_FILE_NAME) + " loopback addresses";

        if (newSnmpPrimaryIf == null) {
            newSnmpPrimaryIf = m_capsdDbSyncer.determinePrimarySnmpInterface(snmpAddresses, true);
            psiType = ConfigFileConstants.getFileName(ConfigFileConstants.COLLECTD_CONFIG_FILE_NAME) + " addresses";
        }

        if (newSnmpPrimaryIf == null) {
            newSnmpPrimaryIf = m_capsdDbSyncer.determinePrimarySnmpInterface(snmpLBAddresses, false);
            psiType = "DB loopback addresses";
        }

        if (newSnmpPrimaryIf == null) {
            newSnmpPrimaryIf = m_capsdDbSyncer.determinePrimarySnmpInterface(snmpAddresses, false);
            psiType = "DB addresses";
        }

        if (newSnmpPrimaryIf == null) {
            newSnmpPrimaryIf = snmpc.getCollectorTargetAddress();
            psiType = "SNMP collector target address";
        }

        if (newSnmpPrimaryIf != null) {
            LOG.debug("updatePrimarySnmpInterface: primary SNMP interface is: {}, selected from {}", newSnmpPrimaryIf, psiType);
            SuspectEventProcessor.setPrimarySnmpInterface(dbc, dbNodeEntry, newSnmpPrimaryIf, oldPriIf);
        } else {
            LOG.debug("SuspectEventProcessor: Unable to determine a primary SNMP interface");
        }   

        /*
         * Now that we've identified the new primary SNMP
         * interface we can determine if it is necessary to
         * generate certain SNMP data collection related
         * events
         */
        generateSnmpDataCollectionEvents(dbNodeEntry, oldPriIf, newSnmpPrimaryIf);
        return newSnmpPrimaryIf;
    }

    /**
     * @param collectorMap
     * @return
     */
    private static IfSnmpCollector findSnmpCollector(final Map<String, IfCollector> collectorMap) {
        for (final IfCollector collector : collectorMap.values()) {
            if (collector.hasSnmpCollection()) {
                return collector.getSnmpCollector();
            }
        }
        return null;
    }

    private DbIpInterfaceEntry[] getInterfaces(final DbNodeEntry dbNodeEntry) {
        /*
         * If this is a forced rescan then retrieve all the interfaces
         * associated with this node and perform collections against them.
         * Otherwise, this is a regularly scheduled rescan, only the
         * node's managed interfaces are to be retrieved and collected.
         */ 

        /*
         * Retrieve list of interfaces associated with this nodeID
         * from the database
         */
        LOG.debug("retrieving managed interfaces for node: {}", getNodeId());

        try {
            return (m_forceRescan ? dbNodeEntry.getInterfaces() : dbNodeEntry.getManagedInterfaces());
        } catch (final NullPointerException npE) {
            LOG.error("RescanProcessor: Null pointer when retrieving {} interfaces for node {}", (m_forceRescan ? "" : "managed"), getNodeId(), npE);
            LOG.error("Rescan failed for node w/ nodeid {}", getNodeId());
        } catch (final SQLException sqlE) {
            LOG.error("RescanProcessor: unable to load interface info for nodeId {} from the database.", getNodeId(), sqlE);
            LOG.error("Rescan failed for node w/ nodeid {}", getNodeId());
        }
        return null;
    }

    private DbNodeEntry getNode() {
        /*
         * Get DbNodeEntry object which represents this node and
         * load it from the database
         */
        try {
            return  DbNodeEntry.get(getNodeId());
        } catch (SQLException e) {
            LOG.error("RescanProcessor: unable to load node info for nodeId {} from the database.", getNodeId(), e);
            LOG.error("Rescan failed for node w/ nodeid {}", getNodeId());
        }
        return null;
    }

    /**
     * Determines if any SNMP data collection related events need to be
     * generated based upon the results of the current rescan. If necessary will
     * generate one of the following events: 'reinitializePrimarySnmpInterface'
     * 'primarySnmpInterfaceChanged'
     * 
     * @param nodeEntry
     *            DbNodeEntry object of the node being rescanned.
     * @param oldPriIf
     *            Previous primary SNMP interface (from the DB).
     * @param primarySnmpIf
     *            Primary SNMP interface as determined by the current rescan.
     */
    private void generateSnmpDataCollectionEvents(final DbNodeEntry nodeEntry, final InetAddress oldPriIf, final InetAddress primarySnmpIf) {
        /*
         * NOTE: If SNMP service was not previously supported on this node
         * then oldPriIf will be null. If this is the case
         * then no need to generate primarySnmpInterfaceChanged event,
         * the nodeGainedService event generated due to the addition of
         * SNMP is sufficient.
         */
        if (oldPriIf == null && primarySnmpIf != null) {
            LOG.debug("generateSnmpDataCollectionEvents: Either SNMP support was recently enabled on this node, or node doesn't support ipAddrTable MIB.");
            m_eventList.add(createPrimarySnmpInterfaceChangedEvent(nodeEntry.getNodeId(), primarySnmpIf, null));
            return;
        } else {
            /*
             * A PrimarySnmpInterfaceChanged event is generated if the scan
             * found a different primary SNMP interface than what is stored
             * in the database.
             */
            if (primarySnmpIf != null && !oldPriIf.equals(primarySnmpIf)) {
                LOG.debug("generateSnmpDataCollectionEvents: primary SNMP interface has changed.  Was: {} Is: {}", str(oldPriIf), str(primarySnmpIf));
                m_eventList.add(createPrimarySnmpInterfaceChangedEvent(nodeEntry.getNodeId(), primarySnmpIf, oldPriIf));
                return;
            }
        }

        /*
         * An interface map is built by the SNMP poller when the primary
         * SNMP interface is initialized by the service monitor. This map
         * is used to associate each interface on the node with its
         * ifIndex and ifLabel for purposes of performing data collection
         * and storage. If an ifIndex has changed for one or more
         * interfaces or if a new interface was added to the node then
         * the primary SNMP interface must be reinitialized so that this
         * interface map can be rebuilt with the new information.
         */
        if (m_ifIndexOnNodeChangedFlag || m_snmpIfTableChangedFlag) {
            LOG.debug("generateSnmpDataCollectionEvents: Generating reinitializeSnmpInterface event for interface {}", str(primarySnmpIf));
            m_eventList.add(createReinitializePrimarySnmpInterfaceEvent(nodeEntry.getNodeId(), primarySnmpIf));
        }
    }

    private static EventBuilder eventBuilder(final String uei) {
        return new EventBuilder(uei, "OpenNMS.Capsd").setHost(Capsd.getLocalHostAddress());
    }

    private static EventBuilder nodeEventBuilder(final String uei, final long nodeId) {
        return eventBuilder(uei).setNodeid(nodeId);
    }

    private static EventBuilder interfaceEventBuilder(final String uei, final long nodeId, final String ipAddr) {
        return eventBuilder(uei).setNodeid(nodeId).setInterface(addr(ipAddr));
    }

    private static EventBuilder serviceEventBuilder(final String uei, final long nodeId, final String ipAddr, final String svc) {
        return eventBuilder(uei).setNodeid(nodeId).setInterface(addr(ipAddr)).setService(svc);
    }

    /**
     * This method is responsible for generating a nodeLabelChanged event and
     * adding it to the event list.
     * 
     * @param updatedEntry
     *            Updated node entry object
     * @param originalEntry
     *            Original node entry object
     */
    private static Event createNodeLabelChangedEvent(final DbNodeEntry updatedEntry, final DbNodeEntry originalEntry) {
        LOG.debug("createNodeLabelChangedEvent: nodeId: {} oldLabel: '{}' oldSource: '{}' newLabel: '{}' newLabelSource: '{}'", updatedEntry.getNodeId(), originalEntry.getLabel(), originalEntry.getLabelSource(), updatedEntry.getLabel(), updatedEntry.getLabelSource());

        final EventBuilder bldr = nodeEventBuilder(EventConstants.NODE_LABEL_CHANGED_EVENT_UEI, updatedEntry.getNodeId());

        if (originalEntry.getLabel() != null) {
            bldr.addParam(EventConstants.PARM_OLD_NODE_LABEL, originalEntry.getLabel());
            if (originalEntry.getLabelSource() != null) {
                bldr.addParam(EventConstants.PARM_OLD_NODE_LABEL_SOURCE, originalEntry.getLabelSource().toString());
            }
        }

        if (updatedEntry.getLabel() != null) {
            bldr.addParam(EventConstants.PARM_NEW_NODE_LABEL, updatedEntry.getLabel());
            if (updatedEntry.getLabelSource() != null) {
                bldr.addParam(EventConstants.PARM_NEW_NODE_LABEL_SOURCE, updatedEntry.getLabelSource().toString());
            }
        }

        LOG.debug("createNodeLabelChangedEvent: successfully created nodeLabelChanged event for nodeid: {}", updatedEntry.getNodeId());

        // Add event to the list of events to be sent out.
        return bldr.getEvent();
    }

    /**
     * This method is responsible for generating a nodeInfoChanged event and
     * adding it to the event list.
     * 
     * @param updatedEntry
     *            Updated node entry object
     * @param originalEntry
     *            Original node entry object
     */
    private static Event createNodeInfoChangedEvent(final DbNodeEntry updatedEntry, final DbNodeEntry originalEntry) {
        LOG.debug("createNodeInfoChangedEvent: nodeId: {}", updatedEntry.getNodeId());

        final EventBuilder bldr = nodeEventBuilder(EventConstants.NODE_INFO_CHANGED_EVENT_UEI, updatedEntry.getNodeId());

        // SysOID
        if (updatedEntry.getSystemOID() != null) {
            bldr.addParam(EventConstants.PARM_NODE_SYSOID, updatedEntry.getSystemOID());
        }

        // SysName
        if (updatedEntry.getSystemName() != null) {
            bldr.addParam(EventConstants.PARM_NODE_SYSNAME, updatedEntry.getSystemName());
        }

        // SysDescription
        if (updatedEntry.getSystemDescription() != null) {
            bldr.addParam(EventConstants.PARM_NODE_SYSDESCRIPTION, updatedEntry.getSystemDescription());
        }

        // SysLocation
        if (updatedEntry.getSystemLocation() != null) {
            bldr.addParam(EventConstants.PARM_NODE_SYSLOCATION, updatedEntry.getSystemLocation());
        }

        // SysContact
        if (updatedEntry.getSystemContact() != null) {
            bldr.addParam(EventConstants.PARM_NODE_SYSCONTACT, updatedEntry.getSystemContact());
        }

        // NetBIOS name
        if (updatedEntry.getNetBIOSName() != null) {
            bldr.addParam(EventConstants.PARM_NODE_NETBIOS_NAME, updatedEntry.getNetBIOSName());
        }

        // Domain name
        if (updatedEntry.getDomainName() != null) {
            bldr.addParam(EventConstants.PARM_NODE_DOMAIN_NAME, updatedEntry.getDomainName());
        }

        // Operating System
        if (updatedEntry.getOS() != null) {
            bldr.addParam(EventConstants.PARM_NODE_OPERATING_SYSTEM, updatedEntry.getOS());
        }

        LOG.debug("createNodeInfoChangedEvent: successfully created nodeInfoChanged event for nodeid: {}", updatedEntry.getNodeId());

        // Add event to the list of events to be sent out.
        return bldr.getEvent();
    }

    /**
     * This method is responsible for generating a primarySnmpInterfaceChanged
     * event and adding it to the event list.
     * 
     * @param nodeId
     *            Nodeid of node being rescanned.
     * @param newPrimaryIf
     *            new primary SNMP interface address
     * @param oldPrimaryIf
     *            old primary SNMP interface address
     * @return 
     */
    private static Event createPrimarySnmpInterfaceChangedEvent(final int nodeId, final InetAddress newPrimaryIf, final InetAddress oldPrimaryIf) {
        final String oldPrimaryAddr = str(oldPrimaryIf);
        final String newPrimaryAddr = str(newPrimaryIf);

        LOG.debug("createPrimarySnmpInterfaceChangedEvent: nodeId: {} oldPrimarySnmpIf: '{}' newPrimarySnmpIf: '{}'", nodeId, oldPrimaryAddr, newPrimaryAddr);

        final EventBuilder bldr = serviceEventBuilder(EventConstants.PRIMARY_SNMP_INTERFACE_CHANGED_EVENT_UEI, nodeId, newPrimaryAddr, "SNMP");

        if (oldPrimaryAddr != null) {
            bldr.addParam(EventConstants.PARM_OLD_PRIMARY_SNMP_ADDRESS, oldPrimaryAddr);
        }

        if (newPrimaryAddr != null) {
            bldr.addParam(EventConstants.PARM_NEW_PRIMARY_SNMP_ADDRESS, newPrimaryAddr);
        }

        LOG.debug("createPrimarySnmpInterfaceChangedEvent: successfully created primarySnmpInterfaceChanged event for nodeid: {}", nodeId);

        // Add event to the list of events to be sent out.
        return bldr.getEvent();
    }

    /**
     * This method is responsible for generating a interfaceIndexChanged event
     * and adding it to the event list.
     * 
     * @param updatedEntry
     *            updated IP interface database entry
     * @param originalEntry
     *            original IP interface database entry
     * @return 
     */
    private static Event createInterfaceIndexChangedEvent(final DbIpInterfaceEntry updatedEntry, final DbIpInterfaceEntry originalEntry) {
        LOG.debug("createInterfaceIndexChangedEvent: nodeId: {} oldIfIndex: {} newIfIndex: {}", updatedEntry.getNodeId(), originalEntry.getIfIndex(), updatedEntry.getIfIndex());

        final EventBuilder bldr = interfaceEventBuilder(EventConstants.INTERFACE_INDEX_CHANGED_EVENT_UEI, updatedEntry.getNodeId(), str(updatedEntry.getIfAddress()));

        bldr.addParam(EventConstants.PARM_OLD_IFINDEX, originalEntry.getIfIndex());
        bldr.addParam(EventConstants.PARM_NEW_IFINDEX, updatedEntry.getIfIndex());

        LOG.debug("createInterfaceIndexChangedEvent: successfully created interfaceIndexChanged event for nodeid: {}", updatedEntry.getNodeId());

        // Add event to the list of events to be sent out.
        return bldr.getEvent();
    }

    /**
     * This method is responsible for generating an ipHostNameChanged event and
     * adding it to the event list.
     * 
     * @param updatedEntry
     *            updated IP interface database entry
     * @param originalEntry
     *            original IP interface database entry
     * @return 
     */
    private static Event createIpHostNameChangedEvent(final DbIpInterfaceEntry updatedEntry, final DbIpInterfaceEntry originalEntry) {
        LOG.debug("createIpHostNameChangedEvent: nodeId: {} oldHostName: {} newHostName: {}", updatedEntry.getNodeId(), originalEntry.getHostname(), updatedEntry.getHostname());

        final EventBuilder bldr = interfaceEventBuilder(EventConstants.INTERFACE_IP_HOSTNAME_CHANGED_EVENT_UEI, updatedEntry.getNodeId(), str(updatedEntry.getIfAddress()));

        // Add old IP Hostname
        if (originalEntry.getHostname() != null) {
            bldr.addParam(EventConstants.PARM_OLD_IP_HOSTNAME, originalEntry.getHostname());
        }

        // Add new IP Hostname
        if (updatedEntry.getHostname() != null) {
            bldr.addParam(EventConstants.PARM_IP_HOSTNAME, updatedEntry.getHostname());
        }

        LOG.debug("createIpHostNameChangedEvent: successfully created ipHostNameChanged event for nodeid: {}", updatedEntry.getNodeId());

        // Add event to the list of events to be sent out.
        return bldr.getEvent();
    }

    /**
     * This method is responsible for generating a interfaceReparented event and
     * adding it to the event list.
     * 
     * @param newNode
     *            Entry of node under which the interface was added.
     * @param oldNodeId
     *            Node identifier of node from which the interface was removed.
     * @param reparentedIf
     *            Reparented interface
     * @return 
     */
    private static Event createInterfaceReparentedEvent(final DbNodeEntry newNode, final int oldNodeId, final InetAddress reparentedIf) {
        final String reparentedAddress = str(reparentedIf);
        LOG.debug("createInterfaceReparentedEvent: ifAddr: {} oldNodeId: {} newNodeId: {}", reparentedAddress, oldNodeId, newNode.getNodeId());

        final EventBuilder bldr = interfaceEventBuilder(EventConstants.INTERFACE_REPARENTED_EVENT_UEI, newNode.getNodeId(), reparentedAddress);

        // Add old node id
        bldr.addParam(EventConstants.PARM_OLD_NODEID, oldNodeId);

        // Add new node id
        bldr.addParam(EventConstants.PARM_NEW_NODEID, newNode.getNodeId());

        // Add ip host name
        bldr.addParam(EventConstants.PARM_IP_HOSTNAME, reparentedIf.getHostName());

        // Add node label and node label source
        if (newNode.getLabel() != null) {
            bldr.addParam(EventConstants.PARM_NODE_LABEL, newNode.getLabel());
            if (newNode.getLabelSource() != null) {
                bldr.addParam(EventConstants.PARM_NODE_LABEL_SOURCE, newNode.getLabelSource().toString());
            }
        }

        // Add nodeSysName
        if (newNode.getSystemName() != null) {
            bldr.addParam(EventConstants.PARM_NODE_SYSNAME, newNode.getSystemName());
        }

        // Add nodeSysDescription
        if (newNode.getSystemDescription() != null) {
            bldr.addParam(EventConstants.PARM_NODE_SYSDESCRIPTION, newNode.getSystemDescription());
        }

        LOG.debug("createInterfaceReparentedEvent: successfully created interfaceReparented event for nodeid/interface: {}/{}", newNode.getNodeId(), reparentedAddress);

        // Add event to the list of events to be sent out.
        return bldr.getEvent();
    }

    /**
     * This method is responsible for generating a duplicateNodeDeleted event
     * and adding it to the event list.
     * 
     * @param deletedNode
     *            Entry of duplciate node which was deleted.
     * @return 
     */
    private static Event createDuplicateNodeDeletedEvent(final DbNodeEntry deletedNode) {
        LOG.debug("createDuplicateNodeDeletedEvent: delete nodeid: {}", deletedNode.getNodeId());

        final EventBuilder bldr = nodeEventBuilder(EventConstants.DUP_NODE_DELETED_EVENT_UEI, deletedNode.getNodeId());

        LOG.debug("createDuplicateNodeDeletedEvent: successfully created duplicateNodeDeleted event for nodeid: {}", deletedNode.getNodeId());

        // Add event to the list of events to be sent out.
        return bldr.getEvent();
    }

    /**
     * This method is responsible for generating a nodeGainedInterface event and
     * adding it to the event list.
     * 
     * @param ifEntry
     *            Entry of new interface.
     * @return 
     */
    private static Event createNodeGainedInterfaceEvent(final DbIpInterfaceEntry ifEntry) {
        final String ifAddress = str(ifEntry.getIfAddress());
        LOG.debug("createNodeGainedInterfaceEvent: nodeId: {} interface: {}", ifEntry.getNodeId(), ifAddress);

        final EventBuilder bldr = interfaceEventBuilder(EventConstants.NODE_GAINED_INTERFACE_EVENT_UEI, ifEntry.getNodeId(), ifAddress);

        // Add ip host name
        bldr.addParam(EventConstants.PARM_IP_HOSTNAME, ifEntry.getHostname() == null ? "" : ifEntry.getHostname());

        // Add discovery method
        bldr.addParam(EventConstants.PARM_METHOD, "icmp");

        LOG.debug("createNodeGainedInterfaceEvent: successfully created nodeGainedInterface event for nodeid: {}", ifEntry.getNodeId());

        // Add event to the list of events to be sent out.
        return bldr.getEvent();
    }

    /**
     * This method is responsible for generating a duplicateIpAddress event and
     * adding it to the event list.
     * 
     * @param ifEntry
     *            Entry of new interface.
     * @return 
     */
    private static Event createDuplicateIpAddressEvent(final DbIpInterfaceEntry ifEntry) {
        final String ifAddress = str(ifEntry.getIfAddress());

        LOG.debug("createDuplicateIpAddressEvent: nodeId: {} interface: {}", ifEntry.getNodeId(), ifAddress);

        EventBuilder bldr = interfaceEventBuilder(EventConstants.DUPLICATE_IPINTERFACE_EVENT_UEI, ifEntry.getNodeId(), ifAddress);

        bldr.addParam(EventConstants.PARM_IP_HOSTNAME, ifEntry.getHostname() == null ? "" : ifEntry.getHostname());

        // Add discovery method
        bldr.addParam(EventConstants.PARM_METHOD, "icmp");

        LOG.debug("createDuplicateIpAddressEvent: successfully created duplicateIpAddress event for nodeid: {}", ifEntry.getNodeId());

        // Add event to the list of events to be sent out.
        return bldr.getEvent();
    }

    /**
     * This method is responsible for generating a nodeGainedService event and
     * adding it to the event list.
     * 
     * @param nodeEntry
     *            Entry of node which has gained a service
     * @param ifEntry
     *            Entry of interface which has gained a service
     * @param svcName
     *            Service name
     * @return 
     */
    private static Event createNodeGainedServiceEvent(final DbNodeEntry nodeEntry, final DbIpInterfaceEntry ifEntry, final String svcName) {
        final String ifAddress = str(ifEntry.getIfAddress());

        LOG.debug("createNodeGainedServiceEvent: nodeId: {} interface: {} service: {}", ifEntry.getNodeId(), ifAddress, svcName);

        EventBuilder bldr = serviceEventBuilder(EventConstants.NODE_GAINED_SERVICE_EVENT_UEI, ifEntry.getNodeId(), ifAddress, svcName);

        // Add ip host name
        bldr.addParam(EventConstants.PARM_IP_HOSTNAME, ifEntry.getHostname() == null ? "" : ifEntry.getHostname());

        // Add nodeSysName
        if (nodeEntry.getSystemName() != null) {
            bldr.addParam(EventConstants.PARM_NODE_SYSNAME, nodeEntry.getSystemName());
        }

        // Add nodeSysDescription
        if (nodeEntry.getSystemDescription() != null) {
            bldr.addParam(EventConstants.PARM_NODE_SYSDESCRIPTION, nodeEntry.getSystemDescription());
        }

        LOG.debug("createNodeGainedServiceEvent: successfully created nodeGainedService event for nodeid: {}", ifEntry.getNodeId());

        // Add event to the list of events to be sent out.
        return bldr.getEvent();
    }

    /**
     * This method is responsible for generating a suspendPollingService event
     * and adding it to the event list.
     * 
     * @param nodeEntry
     *            Entry of node for which a service is to be polled
     * @param ifEntry
     *            Entry of interface which a service is to be polled
     * @param svcName
     *            Service name
     * @return 
     */
    private static Event createSuspendPollingServiceEvent(final DbNodeEntry nodeEntry, final DbIpInterfaceEntry ifEntry, final String svcName) {
        final String ifAddress = str(ifEntry.getIfAddress());

        final EventBuilder bldr = serviceEventBuilder(EventConstants.SUSPEND_POLLING_SERVICE_EVENT_UEI, ifEntry.getNodeId(), ifAddress, svcName);

        // Add ip host name
        bldr.addParam(EventConstants.PARM_IP_HOSTNAME, ifEntry.getHostname() == null ? "" : ifEntry.getHostname());

        // Add nodeSysName
        if (nodeEntry.getSystemName() != null) {
            bldr.addParam(EventConstants.PARM_NODE_SYSNAME, nodeEntry.getSystemName());
        }

        // Add nodeSysDescription
        if (nodeEntry.getSystemDescription() != null) {
            bldr.addParam(EventConstants.PARM_NODE_SYSDESCRIPTION, nodeEntry.getSystemDescription());
        }

        LOG.debug("suspendPollingServiceEvent: Created suspendPollingService event for nodeid: {} interface: {} service: {}", ifEntry.getNodeId(), ifAddress, svcName);

        // Add event to the list of events to be sent out.
        return bldr.getEvent();
    }

    /**
     * This method is responsible for generating a resumePollingService event
     * and adding it to the event list.
     * 
     * @param nodeEntry
     *            Entry of node for which a service is to be polled
     * @param ifEntry
     *            Entry of interface which a service is to be polled
     * @param svcName
     *            Service name
     * @return 
     */
    private static Event createResumePollingServiceEvent(final DbNodeEntry nodeEntry, final DbIpInterfaceEntry ifEntry, final String svcName) {
        final String ifAddress = str(ifEntry.getIfAddress());

        final EventBuilder bldr = serviceEventBuilder(EventConstants.RESUME_POLLING_SERVICE_EVENT_UEI, ifEntry.getNodeId(), ifAddress, svcName);

        // Add ip host name
        bldr.addParam(EventConstants.PARM_IP_HOSTNAME, ifEntry.getHostname() == null ? "" : ifEntry.getHostname());

        // Add nodeSysName
        if (nodeEntry.getSystemName() != null) {
            bldr.addParam(EventConstants.PARM_NODE_SYSNAME, nodeEntry.getSystemName());
        }

        // Add nodeSysDescription
        if (nodeEntry.getSystemDescription() != null) {
            bldr.addParam(EventConstants.PARM_NODE_SYSDESCRIPTION, nodeEntry.getSystemDescription());
        }

        LOG.debug("resumePollingServiceEvent: Created resumePollingService event for nodeid: {} interface: {} service: {}", ifEntry.getNodeId(), ifAddress, svcName);

        // Add event to the list of events to be sent out.
        return bldr.getEvent();
    }

    /**
     * This method is responsible for generating a snmpConflictsWithDb event and
     * adding it to the event list.
     *
     * @param nodeEntry Entry of node for which a conflict exits
     * @return 
     */
    private static Event createSnmpConflictsWithDbEvent(final DbNodeEntry nodeEntry) {
        final EventBuilder bldr = nodeEventBuilder(EventConstants.SNMP_CONFLICTS_WITH_DB_EVENT_UEI, nodeEntry.getNodeId());

        // Add node label
        bldr.addParam(EventConstants.PARM_NODE_LABEL, nodeEntry.getLabel() == null ? "" : nodeEntry.getLabel());

        // Add nodeSysName
        if (nodeEntry.getSystemName() != null) {
            bldr.addParam(EventConstants.PARM_NODE_SYSNAME, nodeEntry.getSystemName());
        }

        // Add nodeSysDescription
        if (nodeEntry.getSystemDescription() != null) {
            bldr.addParam(EventConstants.PARM_NODE_SYSDESCRIPTION, nodeEntry.getSystemDescription());
        }

        LOG.debug("snmpConflictsWithDbEvent: Created snmpConflictsWithDbEvent for nodeid: {}", nodeEntry.getNodeId());

        // Add event to the list of events to be sent out.
        return bldr.getEvent();
    }

    /**
     * This method is responsible for generating a rescanCompleted event and
     * adding it to the event list.
     *
     * @param nodeEntry Entry of node which was rescanned
     * @return 
     */
    private static Event createRescanCompletedEvent(final DbNodeEntry nodeEntry) {
        final EventBuilder bldr = nodeEventBuilder(EventConstants.RESCAN_COMPLETED_EVENT_UEI, nodeEntry.getNodeId());

        // Add node label
        bldr.addParam(EventConstants.PARM_NODE_LABEL, nodeEntry.getLabel() == null ? "" : nodeEntry.getLabel());

        LOG.debug("rescanCompletedEvent: Created rescanCompletedEvent for nodeid: {}", nodeEntry.getNodeId());

        // Add event to the list of events to be sent out.
        return bldr.getEvent();
    }

    /**
     * This method is responsible for generating a interfaceSupportsSNMPEvent
     * event and adding it to the event list.
     * 
     * @param ifEntry
     *            Entry of interface which has gained a service
     * @return 
     */
    private static Event createInterfaceSupportsSNMPEvent(final DbIpInterfaceEntry ifEntry) {
        final String ifAddress = str(ifEntry.getIfAddress());

        LOG.debug("createInterfaceSupportsSNMPEvent: nodeId: {} interface: {}", ifEntry.getNodeId(), ifAddress);

        final EventBuilder bldr = interfaceEventBuilder(EventConstants.INTERFACE_SUPPORTS_SNMP_EVENT_UEI, ifEntry.getNodeId(), ifAddress);

        LOG.debug("interfaceSupportsSNMPEvent: successfully created interfaceSupportsSNMPEvent event for nodeid: {}", ifEntry.getNodeId());

        // Add event to the list of events to be sent out.
        return bldr.getEvent();
    }

    /**
     * This method is responsible for generating a
     * reinitializePrimarySnmpInterface event and adding it to the event list.
     * 
     * @param nodeId
     *            Nodeid of node being rescanned.
     * @param primarySnmpIf
     *            Primary SNMP interface address.
     * @return 
     */
    private static Event createReinitializePrimarySnmpInterfaceEvent(final int nodeId, final InetAddress primarySnmpIf) {
        final String primaryAddress = str(primarySnmpIf);

        LOG.debug("reinitializePrimarySnmpInterface: nodeId: {} interface: {}", nodeId, primaryAddress);

        final EventBuilder bldr = interfaceEventBuilder(EventConstants.REINITIALIZE_PRIMARY_SNMP_INTERFACE_EVENT_UEI, nodeId, primaryAddress);

        LOG.debug("createReinitializePrimarySnmpInterfaceEvent: successfully created reinitializePrimarySnmpInterface event for interface: {}", primaryAddress);

        // Add event to the list of events to be sent out.
        return bldr.getEvent();
    }


    /**
     * Responsible for setting the Set used to track rescans that
     * are already enqueued for processing.  Should be called once by Capsd
     * at startup.
     *
     * @param queuedRescanTracker
     *          The synchronized Set to use
     */
    public static synchronized void setQueuedRescansTracker(final Set<Integer> queuedRescanTracker) {
        s_queuedRescanTracker = Collections.synchronizedSet(queuedRescanTracker);
    }

    /**
     * Is a rescan already enqueued for a given node ID?
     *
     * @param nodeId a {@link java.lang.Integer} object.
     * @return a boolean.
     */
    public static boolean isRescanQueuedForNode(final Integer nodeId) {
        synchronized(s_queuedRescanTracker) {
            return (s_queuedRescanTracker.contains(nodeId));
        }
    }

} // end class
