/*******************************************************************************
 * This file is part of OpenNMS(R).
 *
 * Copyright (C) 2006-2012 The OpenNMS Group, Inc.
 * OpenNMS(R) is Copyright (C) 1999-2012 The OpenNMS Group, Inc.
 *
 * OpenNMS(R) is a registered trademark of The OpenNMS Group, Inc.
 *
 * OpenNMS(R) is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation, either version 3 of the License,
 * or (at your option) any later version.
 *
 * OpenNMS(R) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenNMS(R).  If not, see:
 *      http://www.gnu.org/licenses/
 *
 * For more information contact:
 *     OpenNMS(R) Licensing <license@opennms.org>
 *     http://www.opennms.org/
 *     http://www.opennms.com/
 *******************************************************************************/

package org.opennms.netmgt.poller.remote;

import java.beans.PropertyChangeListener;
import java.util.Collection;
import java.util.Date;
import java.util.List;

import org.opennms.netmgt.model.OnmsMonitoringLocationDefinition;

/**
 * <p>PollerFrontEnd interface.</p>
 *
 * @author <a href="mailto:brozow@opennms.org">Mathew Brozowski</a>
 * @version $Id: $
 */
public interface PollerFrontEnd {
    
    /**
     * <p>getMonitoringLocations</p>
     *
     * @return a {@link java.util.Collection} object.
     */
    public Collection<OnmsMonitoringLocationDefinition> getMonitoringLocations();
    
    /**
     * <p>getPolledServices</p>
     *
     * @return a {@link java.util.Collection} object.
     */
    public Collection<PolledService> getPolledServices();
    
    /**
     * Is the poller currently registered with the server.
     *
     * @return true if and only if the server has been registered
     */
    public boolean isRegistered();
    
    /**
     * Return the monitor name of the poller or null if none exist
     *
     * @return a {@link java.lang.String} object.
     */
    public String getMonitorName();
    
    /**
     * Register the poller if it has not been registered before.
     *
     * @param monitoringLocationName The name of the monitoring Location definition under
     * which to register this monitor
     */
    public void register(String monitoringLocationName);


    /**
     * Set the initial poll time for a polledService
     *
     * @param polledServiceId the id of the polledService whose pollTime we are setting
     * @param initialPollTime the time to set its initialPollTime to
     */
    public void setInitialPollTime(Integer polledServiceId, Date initialPollTime);
    
    /**
     * Poll the service with id polledServiceId and report the results to the server
     *
     * @param polledServiceId The serviceid of the polledService that needs to be polled
     */
    public void pollService(Integer polledServiceId);
    
    /**
     * Returns whether or not the poller has been started
     *
     * @return a boolean.
     */
    public boolean isStarted();
    
    /**
     * Returns whether some error occurred and an exit is necessary
     *
     * @return a boolean.
     */
    public boolean isExitNecessary();
    
    /**
     * Stop polling.  This should be called before the system exits.
     */
    public void stop();
    
    /**
     * Returns the state of polling in this monitor.
     *
     * @param polledServiceId a int.
     * @return a {@link org.opennms.netmgt.poller.remote.ServicePollState} object.
     */
    public ServicePollState getServicePollState(int polledServiceId);


    /**
     * <p>getPollerPollState</p>
     *
     * @return a {@link java.util.List} object.
     */
    public List<ServicePollState> getPollerPollState();
    /**
     * Register a listener to listen for events indication a change
     * in the poller configuration
     *
     * @param l a {@link org.opennms.netmgt.poller.remote.ConfigurationChangedListener} object.
     */
    public void addConfigurationChangedListener(ConfigurationChangedListener l);
    
    /**
     * Remove a config change listener
     *
     * @param l a {@link org.opennms.netmgt.poller.remote.ConfigurationChangedListener} object.
     */
    public void removeConfigurationChangedListener(ConfigurationChangedListener l);

    /**
     * Register a property change listener. (for exampe the 'registered' property)
     *
     * @param l a {@link java.beans.PropertyChangeListener} object.
     */
    public void addPropertyChangeListener(PropertyChangeListener l);
    /**
     * <p>removePropertyChangeListener</p>
     *
     * @param l a {@link java.beans.PropertyChangeListener} object.
     */
    public void removePropertyChangeListener(PropertyChangeListener l);

    /**
     * Register a listener for changes in an attribute of a PolledService
     *
     * @param l a {@link org.opennms.netmgt.poller.remote.ServicePollStateChangedListener} object.
     */
    public void addServicePollStateChangedListener(ServicePollStateChangedListener l);
    /**
     * <p>removeServicePollStateChangedListener</p>
     *
     * @param l a {@link org.opennms.netmgt.poller.remote.ServicePollStateChangedListener} object.
     */
    public void removeServicePollStateChangedListener(ServicePollStateChangedListener l);
	

}
