/*******************************************************************************
 * This file is part of OpenNMS(R).
 *
 * Copyright (C) 2006-2014 The OpenNMS Group, Inc.
 * OpenNMS(R) is Copyright (C) 1999-2014 The OpenNMS Group, Inc.
 *
 * OpenNMS(R) is a registered trademark of The OpenNMS Group, Inc.
 *
 * OpenNMS(R) is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License,
 * or (at your option) any later version.
 *
 * OpenNMS(R) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OpenNMS(R).  If not, see:
 *      http://www.gnu.org/licenses/
 *
 * For more information contact:
 *     OpenNMS(R) Licensing <license@opennms.org>
 *     http://www.opennms.org/
 *     http://www.opennms.com/
 *******************************************************************************/

package org.opennms.netmgt.dao.jaxb;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.opennms.netmgt.config.siteStatusViews.View;
import org.opennms.netmgt.dao.api.SiteStatusViewConfigDao;
import org.springframework.dao.DataRetrievalFailureException;

/**
 * <p>DefaultSiteStatusViewConfigDao class.</p>
 *
 * @author <a href="mailto:brozow@opennms.org">Mathew Brozowski</a>
 * @author <a href="mailto:david@opennms.org">David Hustace</a>
 */
public class DefaultSiteStatusViewConfigDao implements SiteStatusViewConfigDao {
    
    static {
        try {
            SiteStatusViewsFactory.init();
        } catch (FileNotFoundException e) {
            throw new DataRetrievalFailureException("Unable to locate site status view config file", e);
        } catch (IOException e) {
            throw new DataRetrievalFailureException("Error load site status view config file", e);
        }
    }
    
    /**
     * <p>Constructor for DefaultSiteStatusViewConfigDao.</p>
     */
    public DefaultSiteStatusViewConfigDao() {
    }

    /** {@inheritDoc} */
    @Override
    public View getView(String viewName) {
        try {
            return SiteStatusViewsFactory.getInstance().getView(viewName);
        } catch (IOException e) {
            throw new DataRetrievalFailureException("Error load site status view config file", e);
        }
    }

    /**
     * Use this method to get the generated Views class generated by the XSD.
     *
     * @return a {@link org.opennms.netmgt.config.siteStatusViews.Views} object.
     */
    @Override
    public List<View> getViews() {
        return SiteStatusViewsFactory.getConfig().getViews();
    }
    
    /**
     * Use this method to get a Map of view names to marshalled classes based on the generated View class
     * from the XSD.
     *
     * @return <code>Map</> of View classes.
     */
    public Map<String, View> getViewMap() {
        return Collections.unmodifiableMap(SiteStatusViewsFactory.getViewsMap());
    }

    /**
     * <p>getDefaultView</p>
     *
     * @return a {@link org.opennms.netmgt.config.siteStatusViews.View} object.
     */
    @Override
    public View getDefaultView() {
        final String defaultView = SiteStatusViewsFactory.getConfig().getDefaultView();
        return getView(defaultView);
    }

}
