/*******************************************************************************
 * This file is part of OpenNMS(R).
 *
 * Copyright (C) 2014 The OpenNMS Group, Inc.
 * OpenNMS(R) is Copyright (C) 1999-2014 The OpenNMS Group, Inc.
 *
 * OpenNMS(R) is a registered trademark of The OpenNMS Group, Inc.
 *
 * OpenNMS(R) is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License,
 * or (at your option) any later version.
 *
 * OpenNMS(R) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OpenNMS(R).  If not, see:
 *      http://www.gnu.org/licenses/
 *
 * For more information contact:
 *     OpenNMS(R) Licensing <license@opennms.org>
 *     http://www.opennms.org/
 *     http://www.opennms.com/
 *******************************************************************************/

package org.opennms.netmgt.config.poller;

import java.io.Serializable;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlRootElement;

/**
 * Critical service. Defining a critical service greatly reduces the traffic
 * generated by the poller when an interface is DOWN. When an interface is
 * DOWN only the critical service is polled. If and when the critical service
 * comes back UP then the interface's other services are polled to determine
 * their status. When an interface is UP all services are polled as expected.
 * If the critical service goes DOWN, all services are considered to be DOWN
 * and therefore the interface is also considered DOWN.
 */

@XmlRootElement(name="critical-service")
@XmlAccessorType(XmlAccessType.NONE)
public class CriticalService implements Serializable {
    private static final long serialVersionUID = 3608855241271932451L;

    /**
     * Name of the critical service (typically, "ICMP")
     */
    @XmlAttribute(name="name")
    private String m_name;


    public CriticalService() {
        super();
    }

    public CriticalService(final String name) {
        this();
        setName(name);
    }

    /**
     * Name of the critical service (typically, "ICMP")
     */
    public String getName() {
        return m_name;
    }

    public void setName(final String name) {
        m_name = name;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((m_name == null) ? 0 : m_name.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CriticalService)) {
            return false;
        }
        CriticalService other = (CriticalService) obj;
        if (m_name == null) {
            if (other.m_name != null) {
                return false;
            }
        } else if (!m_name.equals(other.m_name)) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        return "CriticalService[" + m_name + "]";
    }
}
