/*
 * Decompiled with CFR 0.152.
 */
package org.opennms.netmgt.flows.elastic;

import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.Timer;
import com.google.common.cache.CacheLoader;
import java.io.Serializable;
import java.net.InetAddress;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;
import org.opennms.core.cache.Cache;
import org.opennms.core.cache.CacheBuilder;
import org.opennms.core.cache.CacheConfig;
import org.opennms.core.utils.InetAddressUtils;
import org.opennms.netmgt.dao.api.InterfaceToNodeCache;
import org.opennms.netmgt.dao.api.NodeDao;
import org.opennms.netmgt.flows.api.FlowSource;
import org.opennms.netmgt.flows.classification.ClassificationEngine;
import org.opennms.netmgt.flows.classification.ClassificationRequest;
import org.opennms.netmgt.flows.classification.persistence.api.Protocols;
import org.opennms.netmgt.flows.elastic.ConversationKeyUtils;
import org.opennms.netmgt.flows.elastic.FlowDocument;
import org.opennms.netmgt.flows.elastic.Locality;
import org.opennms.netmgt.flows.elastic.NodeDocument;
import org.opennms.netmgt.model.OnmsCategory;
import org.opennms.netmgt.model.OnmsNode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.transaction.support.TransactionOperations;

public class DocumentEnricher {
    private static final Logger LOG = LoggerFactory.getLogger(DocumentEnricher.class);
    private final NodeDao nodeDao;
    private final InterfaceToNodeCache interfaceToNodeCache;
    private final TransactionOperations transactionOperations;
    private final ClassificationEngine classificationEngine;
    private final Cache<NodeInfoKey, Optional<NodeDocument>> nodeInfoCache;
    private final Timer nodeLoadTimer;

    public DocumentEnricher(MetricRegistry metricRegistry, NodeDao nodeDao, InterfaceToNodeCache interfaceToNodeCache, TransactionOperations transactionOperations, ClassificationEngine classificationEngine, CacheConfig cacheConfig) {
        this.nodeDao = Objects.requireNonNull(nodeDao);
        this.interfaceToNodeCache = Objects.requireNonNull(interfaceToNodeCache);
        this.transactionOperations = Objects.requireNonNull(transactionOperations);
        this.classificationEngine = Objects.requireNonNull(classificationEngine);
        this.nodeInfoCache = new CacheBuilder().withConfig(cacheConfig).withCacheLoader((CacheLoader)new CacheLoader<NodeInfoKey, Optional<NodeDocument>>(){

            public Optional<NodeDocument> load(NodeInfoKey key) {
                return DocumentEnricher.this.getNodeInfo(key.location, key.ipAddress);
            }
        }).build();
        this.nodeLoadTimer = metricRegistry.timer("nodeLoadTime");
    }

    public void enrich(List<FlowDocument> documents, FlowSource source) {
        if (documents.isEmpty()) {
            LOG.info("Nothing to enrich.");
            return;
        }
        this.transactionOperations.execute(callback -> {
            documents.forEach(document -> {
                document.setHost(source.getSourceAddress());
                document.setLocation(source.getLocation());
                this.getNodeInfoFromCache(source.getLocation(), source.getSourceAddress()).ifPresent(document::setNodeExporter);
                if (document.getDstAddr() != null) {
                    this.getNodeInfoFromCache(source.getLocation(), document.getDstAddr()).ifPresent(document::setNodeDst);
                }
                if (document.getSrcAddr() != null) {
                    this.getNodeInfoFromCache(source.getLocation(), document.getSrcAddr()).ifPresent(document::setNodeSrc);
                }
                if (document.getSrcAddr() != null) {
                    document.setSrcLocality(DocumentEnricher.isPrivateAddress(document.getSrcAddr()) ? Locality.PRIVATE : Locality.PUBLIC);
                }
                if (document.getDstAddr() != null) {
                    document.setDstLocality(DocumentEnricher.isPrivateAddress(document.getDstAddr()) ? Locality.PRIVATE : Locality.PUBLIC);
                }
                if (Locality.PUBLIC.equals((Object)document.getDstLocality()) || Locality.PUBLIC.equals((Object)document.getSrcLocality())) {
                    document.setFlowLocality(Locality.PUBLIC);
                } else if (Locality.PRIVATE.equals((Object)document.getDstLocality()) || Locality.PRIVATE.equals((Object)document.getSrcLocality())) {
                    document.setFlowLocality(Locality.PRIVATE);
                }
                ClassificationRequest classificationRequest = DocumentEnricher.createClassificationRequest(document);
                if (classificationRequest.isClassifiable()) {
                    document.setApplication(this.classificationEngine.classify(classificationRequest));
                }
                document.setConvoKey(ConversationKeyUtils.getConvoKeyAsJsonString(document));
            });
            return null;
        });
    }

    private static boolean isPrivateAddress(String ipAddress) {
        InetAddress inetAddress = InetAddressUtils.addr((String)ipAddress);
        return inetAddress.isLoopbackAddress() || inetAddress.isLinkLocalAddress() || inetAddress.isSiteLocalAddress();
    }

    private Optional<NodeDocument> getNodeInfoFromCache(String location, String ipAddress) {
        NodeInfoKey key = new NodeInfoKey(location, ipAddress);
        try {
            return (Optional)this.nodeInfoCache.get((Object)key);
        }
        catch (ExecutionException e) {
            LOG.error("Error while retrieving NodeDocument from NodeInfoCache: {}.", (Object)e.getMessage(), (Object)e);
            throw new RuntimeException(e);
        }
    }

    private Optional<NodeDocument> getNodeInfo(String location, String ipAddress) {
        return this.getNodeInfo(location, InetAddressUtils.addr((String)ipAddress));
    }

    private Optional<NodeDocument> getNodeInfo(String location, InetAddress ipAddress) {
        Optional nodeId = this.interfaceToNodeCache.getFirstNodeId(location, ipAddress);
        if (nodeId.isPresent()) {
            try (Timer.Context ctx = this.nodeLoadTimer.time();){
                OnmsNode onmsNode = (OnmsNode)this.nodeDao.get((Serializable)nodeId.get());
                if (onmsNode != null) {
                    NodeDocument nodeInfo = new NodeDocument();
                    nodeInfo.setForeignSource(onmsNode.getForeignSource());
                    nodeInfo.setForeignId(onmsNode.getForeignId());
                    nodeInfo.setNodeId((Integer)nodeId.get());
                    nodeInfo.setCategories(onmsNode.getCategories().stream().map(OnmsCategory::getName).collect(Collectors.toList()));
                    Optional<NodeDocument> optional = Optional.of(nodeInfo);
                    return optional;
                }
                LOG.warn("Node with id: {} at location: {} with IP address: {} is in the interface to node cache, but wasn't found in the database.");
            }
        }
        return Optional.empty();
    }

    protected static ClassificationRequest createClassificationRequest(FlowDocument document) {
        ClassificationRequest request = new ClassificationRequest();
        request.setProtocol(document.getProtocol() == null ? null : Protocols.getProtocol((int)document.getProtocol()));
        request.setLocation(document.getLocation());
        request.setExporterAddress(document.getHost());
        request.setDstAddress(document.getDstAddr());
        request.setDstPort(document.getDstPort());
        request.setSrcAddress(document.getSrcAddr());
        request.setSrcPort(document.getSrcPort());
        return request;
    }

    private static class NodeInfoKey {
        public final String location;
        public final String ipAddress;

        private NodeInfoKey(String location, String ipAddress) {
            this.location = location;
            this.ipAddress = ipAddress;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            NodeInfoKey that = (NodeInfoKey)o;
            return Objects.equals(this.location, that.location) && Objects.equals(this.ipAddress, that.ipAddress);
        }

        public int hashCode() {
            return Objects.hash(this.location, this.ipAddress);
        }
    }
}

