/*
 * Decompiled with CFR 0.152.
 */
package org.opennms.netmgt.capsd;

import java.lang.reflect.UndeclaredThrowableException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.Collections;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.RejectedExecutionException;
import org.opennms.core.db.DataSourceFactory;
import org.opennms.core.fiber.PausableFiber;
import org.opennms.core.utils.DBUtils;
import org.opennms.netmgt.capsd.ReparentViaSmb;
import org.opennms.netmgt.capsd.RescanProcessorFactory;
import org.opennms.netmgt.config.CapsdConfigFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class Scheduler
implements Runnable,
PausableFiber {
    private static final Logger LOG = LoggerFactory.getLogger(Scheduler.class);
    private static final String FIBER_NAME = "Capsd Scheduler";
    private static final String SQL_RETRIEVE_NODES = "SELECT nodeid FROM node WHERE nodetype != 'D'";
    private static final String SQL_GET_LAST_POLL_TIME = "SELECT iplastcapsdpoll FROM ipinterface WHERE nodeid=? AND (ismanaged = 'M' OR ismanaged = 'N')";
    private static final int SMB_REPARENTING_IDENTIFIER = -1;
    private int m_status;
    private Thread m_worker;
    private List<NodeInfo> m_knownNodes;
    private long m_interval;
    private long m_initialSleep;
    private ExecutorService m_rescanQ;
    private RescanProcessorFactory m_rescanProcessorFactory;

    Scheduler(ExecutorService rescanQ, RescanProcessorFactory rescanProcessorFactory) throws SQLException {
        this.m_rescanQ = rescanQ;
        this.m_rescanProcessorFactory = rescanProcessorFactory;
        this.m_status = 0;
        this.m_worker = null;
        this.m_knownNodes = Collections.synchronizedList(new LinkedList());
        this.m_interval = CapsdConfigFactory.getInstance().getRescanFrequency();
        LOG.debug("Scheduler: rescan interval(millis): {}", (Object)this.m_interval);
        this.m_initialSleep = CapsdConfigFactory.getInstance().getInitialSleepTime();
        LOG.debug("Scheduler: initial rescan sleep time(millis): {}", (Object)this.m_initialSleep);
        Date lastSmbReparenting = new Date();
        lastSmbReparenting.setTime(System.currentTimeMillis() - this.m_interval);
        LOG.debug("Scheduler: scheduling SMB reparenting...");
        NodeInfo smbInfo = new NodeInfo(-1, lastSmbReparenting, this.m_interval);
        this.m_knownNodes.add(smbInfo);
        this.loadKnownNodes();
        LOG.debug("Scheduler: done loading known nodes, node count: {}", (Object)this.m_knownNodes.size());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void loadKnownNodes() throws SQLException {
        Connection db = null;
        PreparedStatement nodeStmt = null;
        PreparedStatement ifStmt = null;
        ResultSet rs = null;
        ResultSet rset = null;
        DBUtils d = new DBUtils(this.getClass());
        try {
            db = DataSourceFactory.getInstance().getConnection();
            d.watch((Object)db);
            nodeStmt = db.prepareStatement(SQL_RETRIEVE_NODES);
            d.watch((Object)nodeStmt);
            ifStmt = db.prepareStatement(SQL_GET_LAST_POLL_TIME);
            d.watch((Object)ifStmt);
            rs = nodeStmt.executeQuery();
            d.watch((Object)rs);
            while (rs.next()) {
                int nodeId = rs.getInt(1);
                ifStmt.setInt(1, nodeId);
                LOG.debug("loadKnownNodes: retrieved nodeid {}, now getting last poll time.", (Object)nodeId);
                rset = ifStmt.executeQuery();
                d.watch((Object)rs);
                if (rset.next()) {
                    Timestamp lastPolled = rset.getTimestamp(1);
                    if (lastPolled == null || rset.wasNull()) continue;
                    LOG.debug("loadKnownNodes: adding node {} with last poll time {}", (Object)nodeId, (Object)lastPolled);
                    NodeInfo nodeInfo = new NodeInfo(nodeId, lastPolled, this.m_interval);
                    this.m_knownNodes.add(nodeInfo);
                    continue;
                }
                LOG.debug("Node w/ nodeid {} has no managed interfaces from which to retrieve a last poll time...it will not be scheduled.", (Object)nodeId);
            }
        }
        finally {
            d.cleanUp();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void scheduleNode(int nodeId) throws SQLException {
        Connection db = null;
        DBUtils d = new DBUtils(this.getClass());
        try {
            db = DataSourceFactory.getInstance().getConnection();
            d.watch((Object)db);
            PreparedStatement ifStmt = db.prepareStatement(SQL_GET_LAST_POLL_TIME);
            d.watch((Object)ifStmt);
            ifStmt.setInt(1, nodeId);
            ResultSet rset = ifStmt.executeQuery();
            d.watch((Object)rset);
            if (rset.next()) {
                Timestamp lastPolled = rset.getTimestamp(1);
                if (lastPolled != null && !rset.wasNull()) {
                    LOG.debug("scheduleNode: adding node {} with last poll time {}", (Object)nodeId, (Object)lastPolled);
                    this.m_knownNodes.add(new NodeInfo(nodeId, lastPolled, this.m_interval));
                }
            } else {
                LOG.warn("scheduleNode: Failed to retrieve last polled time from database for nodeid {}", (Object)nodeId);
            }
        }
        finally {
            d.cleanUp();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void unscheduleNode(int nodeId) {
        List<NodeInfo> list = this.m_knownNodes;
        synchronized (list) {
            for (NodeInfo nodeInfo : this.m_knownNodes) {
                if (nodeInfo.getNodeId() != nodeId) continue;
                LOG.debug("unscheduleNode: removing node {} from the scheduler.", (Object)nodeId);
                this.m_knownNodes.remove(nodeInfo);
                break;
            }
        }
    }

    void forceRescan(int nodeId) {
        try {
            this.m_rescanQ.execute(this.m_rescanProcessorFactory.createForcedRescanProcessor(nodeId));
        }
        catch (RejectedExecutionException e) {
            LOG.error("forceRescan: Failed to add node {} to the rescan queue.", (Object)nodeId, (Object)e);
        }
    }

    public synchronized void start() {
        if (this.m_worker != null) {
            throw new IllegalStateException("The fiber has already run or is running");
        }
        this.m_worker = new Thread((Runnable)this, this.getName());
        this.m_worker.setDaemon(true);
        this.m_worker.start();
        this.m_status = 1;
        LOG.debug("Scheduler.start: scheduler started");
    }

    public synchronized void stop() {
        if (this.m_worker == null) {
            throw new IllegalStateException("The fiber has never been started");
        }
        this.m_status = 3;
        this.m_worker.interrupt();
        LOG.debug("Scheduler.stop: scheduler stopped");
    }

    public synchronized void pause() {
        if (this.m_worker == null) {
            throw new IllegalStateException("The fiber has never been started");
        }
        if (this.m_status == 4 || this.m_status == 3) {
            throw new IllegalStateException("The fiber is not running or a stop is pending");
        }
        if (this.m_status == 6) {
            return;
        }
        this.m_status = 5;
        this.notifyAll();
    }

    public synchronized void resume() {
        if (this.m_worker == null) {
            throw new IllegalStateException("The fiber has never been started");
        }
        if (this.m_status == 4 || this.m_status == 3) {
            throw new IllegalStateException("The fiber is not running or a stop is pending");
        }
        if (this.m_status == 2) {
            return;
        }
        this.m_status = 7;
        this.notifyAll();
    }

    public synchronized int getStatus() {
        if (this.m_worker != null && !this.m_worker.isAlive()) {
            this.m_status = 4;
        }
        return this.m_status;
    }

    public String getName() {
        return FIBER_NAME;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void run() {
        Scheduler scheduler = this;
        synchronized (scheduler) {
            this.m_status = 2;
        }
        LOG.debug("Scheduler.run: scheduler running");
        boolean firstPass = true;
        while (true) {
            Scheduler scheduler2 = this;
            synchronized (scheduler2) {
                if (this.m_status != 2 && this.m_status != 6 && this.m_status != 5 && this.m_status != 7) {
                    LOG.debug("Scheduler.run: status = {}, time to exit", (Object)this.m_status);
                    break;
                }
            }
            if (firstPass) {
                firstPass = false;
                scheduler2 = this;
                synchronized (scheduler2) {
                    try {
                        LOG.debug("Scheduler.run: initial sleep configured for {}ms...sleeping...", (Object)this.m_initialSleep);
                        this.wait(this.m_initialSleep);
                    }
                    catch (InterruptedException ex) {
                        LOG.debug("Scheduler.run: interrupted exception during initial sleep...exiting.");
                        break;
                    }
                }
            }
            int added = 0;
            Object object = this.m_knownNodes;
            synchronized (object) {
                LOG.debug("Scheduler.run: iterating over known nodes list to schedule...");
                for (NodeInfo node : this.m_knownNodes) {
                    if (node.isScheduled() || !node.timeForRescan()) continue;
                    try {
                        node.setScheduled(true);
                        if (node.getNodeId() == -1) {
                            LOG.debug("Scheduler.run: time for reparenting via SMB...");
                            Connection db = null;
                            try {
                                db = DataSourceFactory.getInstance().getConnection();
                                ReparentViaSmb reparenter = new ReparentViaSmb(db);
                                try {
                                    reparenter.sync();
                                }
                                catch (SQLException sqlE) {
                                    LOG.error("Unexpected database error during SMB reparenting", (Throwable)sqlE);
                                }
                                catch (Throwable t) {
                                    LOG.error("Unexpected error during SMB reparenting", t);
                                }
                            }
                            catch (SQLException sqlE) {
                                LOG.error("Unable to get database connection from the factory.", (Throwable)sqlE);
                            }
                            finally {
                                if (db != null) {
                                    try {
                                        db.close();
                                    }
                                    catch (Throwable e) {}
                                }
                            }
                            node.setLastScanned(new Date());
                            node.setScheduled(false);
                            LOG.debug("Scheduler.run: SMB reparenting completed...");
                            continue;
                        }
                        LOG.debug("Scheduler.run: adding node {} to the rescan queue.", (Object)node.getNodeId());
                        this.m_rescanQ.execute(node);
                        ++added;
                    }
                    catch (RejectedExecutionException e) {
                        LOG.info("Scheduler.schedule: failed to add new node to rescan queue", (Throwable)e);
                        throw new UndeclaredThrowableException(e);
                    }
                }
            }
            object = this;
            synchronized (object) {
                if (added == 0) {
                    try {
                        this.wait(60000L);
                    }
                    catch (InterruptedException ex) {
                        break;
                    }
                }
            }
        }
        LOG.debug("Scheduler.run: scheduler exiting, state = STOPPED");
        Scheduler scheduler3 = this;
        synchronized (scheduler3) {
            this.m_status = 4;
        }
    }

    final class NodeInfo
    implements Runnable {
        int m_nodeId;
        Timestamp m_lastScanned;
        long m_interval;
        boolean m_scheduled;

        NodeInfo(int nodeId, Timestamp lastScanned, long interval) {
            this.m_nodeId = nodeId;
            this.m_lastScanned = lastScanned;
            this.m_interval = interval;
            this.m_scheduled = false;
        }

        NodeInfo(int nodeId, Date lastScanned, long interval) {
            this.m_nodeId = nodeId;
            this.m_lastScanned = new Timestamp(lastScanned.getTime());
            this.m_interval = interval;
            this.m_scheduled = false;
        }

        boolean isScheduled() {
            return this.m_scheduled;
        }

        int getNodeId() {
            return this.m_nodeId;
        }

        Timestamp getLastScanned() {
            return this.m_lastScanned;
        }

        long getRescanInterval() {
            return this.m_interval;
        }

        void setScheduled(boolean scheduled) {
            this.m_scheduled = scheduled;
        }

        void setLastScanned(Date lastScanned) {
            this.m_lastScanned = new Timestamp(lastScanned.getTime());
        }

        void setLastScanned(Timestamp lastScanned) {
            this.m_lastScanned = lastScanned;
        }

        boolean timeForRescan() {
            return System.currentTimeMillis() >= this.m_lastScanned.getTime() + this.m_interval;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            try {
                Scheduler.this.m_rescanProcessorFactory.createRescanProcessor(this.getNodeId()).run();
            }
            finally {
                this.setLastScanned(new Date());
                this.setScheduled(false);
            }
        }
    }
}

