/*******************************************************************************
 * This file is part of OpenNMS(R).
 *
 * Copyright (C) 2015-2016 The OpenNMS Group, Inc.
 * OpenNMS(R) is Copyright (C) 1999-2016 The OpenNMS Group, Inc.
 *
 * OpenNMS(R) is a registered trademark of The OpenNMS Group, Inc.
 *
 * OpenNMS(R) is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License,
 * or (at your option) any later version.
 *
 * OpenNMS(R) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OpenNMS(R).  If not, see:
 *      http://www.gnu.org/licenses/
 *
 * For more information contact:
 *     OpenNMS(R) Licensing <license@opennms.org>
 *     http://www.opennms.org/
 *     http://www.opennms.com/
 *******************************************************************************/

package org.opennms.netmgt.discovery;

import java.net.InetAddress;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import org.opennms.core.utils.InetAddressUtils;

/**
 * Given a list of managed IP addresses, this filter will match IP addresses not in that
 * list, hence it is an unmanaged IP address filter.
 */
public class UnmanagedInterfaceFilter implements IpAddressFilter {
    /**
     * a set of addresses to skip discovery on
     */
    private Set<String> m_managedAddresses = Collections.synchronizedSet(new HashSet<String>());

    public void addManagedAddress(String address) {
        synchronized(m_managedAddresses) {
            m_managedAddresses.add(address);
        }
    }

    public void removeManagedAddress(String address) {
        synchronized(m_managedAddresses) {
            m_managedAddresses.remove(address);
        }
    }

    public int size() {
        synchronized(m_managedAddresses) {
            return m_managedAddresses.size();
        }
    }

    public void setManagedAddresses(Set<String> addresses) {
        synchronized(m_managedAddresses) {
            m_managedAddresses.clear();
            m_managedAddresses.addAll(addresses);
        }
    }

    @Override
    public boolean matches(InetAddress address) {
        return matches(InetAddressUtils.str(address));
    }

    @Override
    public boolean matches(String address) {
        synchronized(m_managedAddresses) {
            return !m_managedAddresses.contains(address);
        }
    }
}
