/*
 * Decompiled with CFR 0.152.
 */
package org.opennms.core.ipc.sink.common;

import com.codahale.metrics.Counter;
import com.codahale.metrics.Gauge;
import com.codahale.metrics.Metric;
import com.codahale.metrics.MetricRegistry;
import com.google.common.annotations.VisibleForTesting;
import com.swrve.ratelimitedlogger.RateLimitedLog;
import java.time.Duration;
import java.util.AbstractMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;
import org.opennms.core.concurrent.LogPreservingThreadFactory;
import org.opennms.core.ipc.sink.api.AsyncDispatcher;
import org.opennms.core.ipc.sink.api.AsyncPolicy;
import org.opennms.core.ipc.sink.api.DispatchQueue;
import org.opennms.core.ipc.sink.api.DispatchQueueFactory;
import org.opennms.core.ipc.sink.api.Message;
import org.opennms.core.ipc.sink.api.SinkModule;
import org.opennms.core.ipc.sink.api.SyncDispatcher;
import org.opennms.core.ipc.sink.api.WriteFailedException;
import org.opennms.core.ipc.sink.common.DispatcherState;
import org.opennms.core.ipc.sink.offheap.DispatchQueueServiceLoader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AsyncDispatcherImpl<W, S extends Message, T extends Message>
implements AsyncDispatcher<S> {
    private static final Logger LOG = LoggerFactory.getLogger(AsyncDispatcherImpl.class);
    private final SyncDispatcher<S> syncDispatcher;
    private final AsyncPolicy asyncPolicy;
    private final Counter droppedCounter;
    private final Map<String, CompletableFuture<AsyncDispatcher.DispatchStatus>> futureMap = new ConcurrentHashMap<String, CompletableFuture<AsyncDispatcher.DispatchStatus>>();
    private final AtomicResultQueue<S> atomicResultQueue;
    private final AtomicLong missedFutures = new AtomicLong(0L);
    private final AtomicInteger activeDispatchers = new AtomicInteger(0);
    private final RateLimitedLog RATE_LIMITED_LOGGER = RateLimitedLog.withRateLimit((Logger)LOG).maxRate(5).every(Duration.ofSeconds(30L)).build();
    private final ExecutorService executor;

    public AsyncDispatcherImpl(DispatcherState<W, S, T> state, AsyncPolicy asyncPolicy, SyncDispatcher<S> syncDispatcher) {
        DispatchQueue dispatchQueue;
        Objects.requireNonNull(state);
        Objects.requireNonNull(asyncPolicy);
        Objects.requireNonNull(syncDispatcher);
        this.syncDispatcher = syncDispatcher;
        this.asyncPolicy = asyncPolicy;
        SinkModule<S, T> sinkModule = state.getModule();
        Optional<DispatchQueueFactory> factory = DispatchQueueServiceLoader.getDispatchQueueFactory();
        if (factory.isPresent()) {
            LOG.debug("Using queue from factory");
            dispatchQueue = factory.get().getQueue(asyncPolicy, sinkModule.getId(), arg_0 -> sinkModule.marshalSingleMessage(arg_0), arg_0 -> sinkModule.unmarshalSingleMessage(arg_0));
        } else {
            int size = asyncPolicy.getQueueSize();
            LOG.debug("Using default in memory queue of size {}", (Object)size);
            dispatchQueue = new DefaultQueue(size);
        }
        this.atomicResultQueue = new AtomicResultQueue(dispatchQueue);
        state.getMetrics().register(MetricRegistry.name((String)state.getModule().getId(), (String[])new String[]{"queue-size"}), (Metric)((Gauge)this.activeDispatchers::get));
        this.droppedCounter = state.getMetrics().counter(MetricRegistry.name((String)state.getModule().getId(), (String[])new String[]{"dropped"}));
        this.executor = Executors.newFixedThreadPool(asyncPolicy.getNumThreads(), (ThreadFactory)new LogPreservingThreadFactory("OpenNMS.Sink.AsyncDispatcher." + state.getModule().getId(), Integer.MAX_VALUE));
        this.startDrainingQueue();
    }

    private void dispatchFromQueue() {
        while (true) {
            try {
                while (true) {
                    LOG.trace("Asking dispatch queue for the next entry...");
                    Map.Entry<String, S> messageEntry = this.atomicResultQueue.dequeue();
                    LOG.trace("Received message entry from dispatch queue {}", messageEntry);
                    this.activeDispatchers.incrementAndGet();
                    LOG.trace("Sending message {} via sync dispatcher", messageEntry);
                    this.syncDispatcher.send(messageEntry.getValue());
                    LOG.trace("Successfully sent message {}", messageEntry);
                    if (messageEntry.getKey() != null) {
                        LOG.trace("Attempting to complete future for message {}", messageEntry);
                        CompletableFuture<AsyncDispatcher.DispatchStatus> messageFuture = this.futureMap.remove(messageEntry.getKey());
                        if (messageFuture != null) {
                            messageFuture.complete(AsyncDispatcher.DispatchStatus.DISPATCHED);
                            LOG.trace("Completed future for message {}", messageEntry);
                        } else {
                            this.RATE_LIMITED_LOGGER.warn("No future found for message {}", messageEntry);
                            this.missedFutures.incrementAndGet();
                        }
                    } else {
                        LOG.trace("Dequeued an entry with a null key");
                    }
                    this.activeDispatchers.decrementAndGet();
                }
            }
            catch (InterruptedException e) {
            }
            catch (Exception e) {
                this.RATE_LIMITED_LOGGER.warn("Encountered exception while taking from dispatch queue", (Throwable)e);
                continue;
            }
            break;
        }
    }

    private void startDrainingQueue() {
        for (int i = 0; i < this.asyncPolicy.getNumThreads(); ++i) {
            this.executor.execute(this::dispatchFromQueue);
        }
    }

    public CompletableFuture<AsyncDispatcher.DispatchStatus> send(S message) {
        CompletableFuture<AsyncDispatcher.DispatchStatus> sendFuture = new CompletableFuture<AsyncDispatcher.DispatchStatus>();
        if (!this.asyncPolicy.isBlockWhenFull() && this.atomicResultQueue.isFull()) {
            this.droppedCounter.inc();
            sendFuture.completeExceptionally(new RuntimeException("Dispatch queue full"));
            return sendFuture;
        }
        try {
            String newId = UUID.randomUUID().toString();
            this.atomicResultQueue.enqueue(message, newId, result -> {
                LOG.trace("Result of enqueueing for Id {} was {}", (Object)newId, result);
                if (result == DispatchQueue.EnqueueResult.DEFERRED) {
                    sendFuture.complete(AsyncDispatcher.DispatchStatus.QUEUED);
                } else {
                    this.futureMap.put(newId, sendFuture);
                }
            });
        }
        catch (WriteFailedException e) {
            sendFuture.completeExceptionally(e);
        }
        return sendFuture;
    }

    @VisibleForTesting
    public long getMissedFutures() {
        return this.missedFutures.get();
    }

    public int getQueueSize() {
        return this.atomicResultQueue.getSize();
    }

    public void close() throws Exception {
        this.syncDispatcher.close();
        this.executor.shutdown();
    }

    private static class DefaultQueue<T>
    implements DispatchQueue<T> {
        private final BlockingQueue<Map.Entry<String, T>> queue;
        private final Lock enqueueFairMutex = new ReentrantLock(true);

        DefaultQueue(int size) {
            this.queue = new ArrayBlockingQueue<Map.Entry<String, T>>(size);
        }

        public DispatchQueue.EnqueueResult enqueue(T message, String key) throws WriteFailedException {
            this.enqueueFairMutex.lock();
            try {
                this.queue.put(new AbstractMap.SimpleImmutableEntry<String, T>(key, message));
                DispatchQueue.EnqueueResult enqueueResult = DispatchQueue.EnqueueResult.IMMEDIATE;
                return enqueueResult;
            }
            catch (InterruptedException e) {
                throw new WriteFailedException((Throwable)e);
            }
            finally {
                this.enqueueFairMutex.unlock();
            }
        }

        public Map.Entry<String, T> dequeue() throws InterruptedException {
            return this.queue.take();
        }

        public boolean isFull() {
            return this.queue.remainingCapacity() <= 0;
        }

        public int getSize() {
            return this.queue.size();
        }
    }

    private static final class AtomicResultQueue<T> {
        private final Map<String, CountDownLatch> resultRecordedMap = new ConcurrentHashMap<String, CountDownLatch>();
        private final DispatchQueue<T> dispatchQueue;

        public AtomicResultQueue(DispatchQueue<T> dispatchQueue) {
            this.dispatchQueue = Objects.requireNonNull(dispatchQueue);
        }

        void enqueue(T message, String key, Consumer<DispatchQueue.EnqueueResult> onEnqueue) throws WriteFailedException {
            CountDownLatch resultRecorded = new CountDownLatch(1);
            this.resultRecordedMap.put(key, resultRecorded);
            DispatchQueue.EnqueueResult result = this.dispatchQueue.enqueue(message, key);
            if (result == DispatchQueue.EnqueueResult.DEFERRED) {
                this.resultRecordedMap.remove(key);
            }
            onEnqueue.accept(result);
            resultRecorded.countDown();
        }

        Map.Entry<String, T> dequeue() throws InterruptedException {
            Map.Entry messageEntry = this.dispatchQueue.dequeue();
            if (messageEntry.getKey() == null) {
                return messageEntry;
            }
            this.resultRecordedMap.remove(messageEntry.getKey()).await();
            return messageEntry;
        }

        boolean isFull() {
            return this.dispatchQueue.isFull();
        }

        int getSize() {
            return this.dispatchQueue.getSize();
        }
    }
}

