/*
 * Decompiled with CFR 0.152.
 */
package org.opennms.core.ipc.rpc.kafka;

import com.google.common.base.Strings;
import com.google.common.math.IntMath;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.protobuf.ByteString;
import com.google.protobuf.InvalidProtocolBufferException;
import com.swrve.ratelimitedlogger.RateLimitedLog;
import io.opentracing.Span;
import io.opentracing.SpanContext;
import io.opentracing.Tracer;
import io.opentracing.propagation.Format;
import io.opentracing.propagation.TextMapExtractAdapter;
import java.io.IOException;
import java.math.RoundingMode;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.Delayed;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.common.errors.WakeupException;
import org.apache.kafka.common.serialization.ByteArrayDeserializer;
import org.apache.kafka.common.serialization.ByteArraySerializer;
import org.apache.kafka.common.serialization.StringDeserializer;
import org.apache.kafka.common.serialization.StringSerializer;
import org.joda.time.Duration;
import org.opennms.core.camel.JmsQueueNameFactory;
import org.opennms.core.ipc.common.kafka.KafkaConfigProvider;
import org.opennms.core.ipc.common.kafka.KafkaRpcConstants;
import org.opennms.core.ipc.common.kafka.Utils;
import org.opennms.core.ipc.rpc.kafka.model.RpcMessageProtos;
import org.opennms.core.rpc.api.RpcModule;
import org.opennms.core.rpc.api.RpcRequest;
import org.opennms.core.rpc.api.RpcResponse;
import org.opennms.core.tracing.api.TracerRegistry;
import org.opennms.distributed.core.api.MinionIdentity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KafkaRpcServerManager {
    private static final Logger LOG = LoggerFactory.getLogger(KafkaRpcServerManager.class);
    private static final RateLimitedLog RATE_LIMITED_LOG = RateLimitedLog.withRateLimit((Logger)LOG).maxRate(5).every(Duration.standardSeconds((long)30L)).build();
    private final Map<String, RpcModule<RpcRequest, RpcResponse>> registerdModules = new ConcurrentHashMap<String, RpcModule<RpcRequest, RpcResponse>>();
    private final Properties kafkaConfig = new Properties();
    private final KafkaConfigProvider kafkaConfigProvider;
    private KafkaProducer<String, byte[]> producer;
    private MinionIdentity minionIdentity;
    private Integer maxBufferSize = 921600;
    private final ThreadFactory threadFactory = new ThreadFactoryBuilder().setNameFormat("rpc-server-kafka-consumer-%d").build();
    private final ExecutorService executor = Executors.newCachedThreadPool(this.threadFactory);
    private Map<RpcModule<RpcRequest, RpcResponse>, KafkaConsumerRunner> rpcModuleConsumers = new ConcurrentHashMap<RpcModule<RpcRequest, RpcResponse>, KafkaConsumerRunner>();
    private Map<String, ByteString> messageCache = new ConcurrentHashMap<String, ByteString>();
    private DelayQueue<RpcId> rpcIdQueue = new DelayQueue();
    private ExecutorService delayQueueExecutor = Executors.newSingleThreadExecutor();
    private Map<String, Integer> currentChunkCache = new ConcurrentHashMap<String, Integer>();
    private final TracerRegistry tracerRegistry;

    public KafkaRpcServerManager(KafkaConfigProvider configProvider, MinionIdentity minionIdentity, TracerRegistry tracerRegistry) {
        this.kafkaConfigProvider = configProvider;
        this.minionIdentity = minionIdentity;
        this.tracerRegistry = tracerRegistry;
    }

    public void init() throws IOException {
        this.kafkaConfig.put("group.id", this.minionIdentity.getLocation());
        this.kafkaConfig.put("auto.commit.interval.ms", "1000");
        this.kafkaConfig.put("key.deserializer", StringDeserializer.class.getCanonicalName());
        this.kafkaConfig.put("value.deserializer", ByteArrayDeserializer.class.getCanonicalName());
        this.kafkaConfig.put("key.serializer", StringSerializer.class.getCanonicalName());
        this.kafkaConfig.put("value.serializer", ByteArraySerializer.class.getCanonicalName());
        this.kafkaConfig.putAll((Map<?, ?>)this.kafkaConfigProvider.getProperties());
        LOG.info("initializing the Kafka producer with: {}", (Object)this.kafkaConfig);
        this.producer = (KafkaProducer)Utils.runWithGivenClassLoader(() -> new KafkaProducer(this.kafkaConfig), (ClassLoader)KafkaProducer.class.getClassLoader());
        this.maxBufferSize = KafkaRpcConstants.getMaxBufferSize((Properties)this.kafkaConfig);
        this.delayQueueExecutor.execute(() -> {
            try {
                while (true) {
                    RpcId rpcId = (RpcId)this.rpcIdQueue.take();
                    this.messageCache.remove(rpcId.getRpcId());
                    this.currentChunkCache.remove(rpcId.getRpcId());
                }
            }
            catch (InterruptedException e) {
                LOG.error("Delay Queue has been interrupted ", (Throwable)e);
                return;
            }
        });
        this.tracerRegistry.init(this.minionIdentity.getLocation() + "@" + this.minionIdentity.getId());
    }

    public void bind(RpcModule module) throws Exception {
        if (module != null) {
            RpcModule rpcModule = module;
            if (this.registerdModules.containsKey(rpcModule.getId())) {
                LOG.warn(" {} module is already registered", (Object)rpcModule.getId());
            } else {
                this.registerdModules.put(rpcModule.getId(), (RpcModule<RpcRequest, RpcResponse>)rpcModule);
                this.startConsumerForModule((RpcModule<RpcRequest, RpcResponse>)rpcModule);
            }
        }
    }

    protected void startConsumerForModule(RpcModule<RpcRequest, RpcResponse> rpcModule) {
        JmsQueueNameFactory topicNameFactory = new JmsQueueNameFactory("rpc-request", rpcModule.getId(), this.minionIdentity.getLocation());
        KafkaConsumer consumer = (KafkaConsumer)Utils.runWithGivenClassLoader(() -> new KafkaConsumer(this.kafkaConfig), (ClassLoader)KafkaConsumer.class.getClassLoader());
        KafkaConsumerRunner kafkaConsumerRunner = new KafkaConsumerRunner(rpcModule, (KafkaConsumer<String, byte[]>)consumer, topicNameFactory.getName());
        this.executor.execute(kafkaConsumerRunner);
        LOG.info("started kafka consumer for module : {}", (Object)rpcModule.getId());
        this.rpcModuleConsumers.put(rpcModule, kafkaConsumerRunner);
    }

    public void unbind(RpcModule module) throws Exception {
        if (module != null) {
            RpcModule rpcModule = module;
            this.registerdModules.remove(rpcModule.getId());
            this.stopConsumerForModule((RpcModule<RpcRequest, RpcResponse>)rpcModule);
        }
    }

    protected void stopConsumerForModule(RpcModule<RpcRequest, RpcResponse> rpcModule) {
        KafkaConsumerRunner kafkaConsumerRunner = this.rpcModuleConsumers.remove(rpcModule);
        LOG.info("stopped kafka consumer for module : {}", (Object)rpcModule.getId());
        kafkaConsumerRunner.shutdown();
    }

    public void destroy() {
        if (this.producer != null) {
            this.producer.close();
        }
        this.messageCache.clear();
        this.executor.shutdown();
        this.delayQueueExecutor.shutdown();
    }

    DelayQueue<RpcId> getRpcIdQueue() {
        return this.rpcIdQueue;
    }

    Map<RpcModule<RpcRequest, RpcResponse>, KafkaConsumerRunner> getRpcModuleConsumers() {
        return this.rpcModuleConsumers;
    }

    ExecutorService getExecutor() {
        return this.executor;
    }

    Properties getKafkaConfig() {
        return this.kafkaConfig;
    }

    Map<String, ByteString> getMessageCache() {
        return this.messageCache;
    }

    private class RpcId
    implements Delayed {
        private final long expirationTime;
        private final String rpcId;

        public RpcId(String rpcId, long expirationTime) {
            this.rpcId = rpcId;
            this.expirationTime = expirationTime;
        }

        @Override
        public int compareTo(Delayed other) {
            long myDelay = this.getDelay(TimeUnit.MILLISECONDS);
            long otherDelay = other.getDelay(TimeUnit.MILLISECONDS);
            return Long.compare(myDelay, otherDelay);
        }

        @Override
        public long getDelay(TimeUnit unit) {
            long now = System.currentTimeMillis();
            return unit.convert(this.expirationTime - now, TimeUnit.MILLISECONDS);
        }

        public String getRpcId() {
            return this.rpcId;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            RpcId that = (RpcId)o;
            return this.expirationTime == that.expirationTime && Objects.equals(this.rpcId, that.rpcId);
        }

        public int hashCode() {
            return Objects.hash(this.expirationTime, this.rpcId);
        }
    }

    class KafkaConsumerRunner
    implements Runnable {
        private final KafkaConsumer<String, byte[]> consumer;
        private final AtomicBoolean closed = new AtomicBoolean(false);
        private String topic;
        private RpcModule<RpcRequest, RpcResponse> module;

        public KafkaConsumerRunner(RpcModule<RpcRequest, RpcResponse> rpcModule, KafkaConsumer<String, byte[]> consumer, String topic) {
            this.consumer = consumer;
            this.topic = topic;
            this.module = rpcModule;
        }

        public void shutdown() {
            this.closed.set(true);
            this.consumer.wakeup();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            try {
                this.consumer.subscribe(Arrays.asList(this.topic));
                LOG.info("subscribed to topic {}", (Object)this.topic);
                while (!this.closed.get()) {
                    ConsumerRecords records = this.consumer.poll(java.time.Duration.ofMillis(Long.MAX_VALUE));
                    for (ConsumerRecord record : records) {
                        try {
                            boolean messageProcessed;
                            RpcMessageProtos.RpcMessage rpcMessage = RpcMessageProtos.RpcMessage.parseFrom((byte[])record.value());
                            String rpcId = rpcMessage.getRpcId();
                            long expirationTime = rpcMessage.getExpirationTime();
                            if (expirationTime < System.currentTimeMillis()) {
                                LOG.warn("ttl already expired for the request id = {}, won't process.", (Object)rpcMessage.getRpcId());
                                continue;
                            }
                            boolean hasSystemId = !Strings.isNullOrEmpty((String)rpcMessage.getSystemId());
                            String minionId = KafkaRpcServerManager.this.minionIdentity.getId();
                            if (hasSystemId && !minionId.equals(rpcMessage.getSystemId()) || hasSystemId && (messageProcessed = this.handleDirectedRPC(rpcMessage))) continue;
                            ByteString rpcContent = rpcMessage.getRpcContent();
                            if (rpcMessage.getTotalChunks() > 1) {
                                boolean allChunksReceived = this.handleChunks(rpcMessage);
                                if (!allChunksReceived) continue;
                                rpcContent = (ByteString)KafkaRpcServerManager.this.messageCache.get(rpcId);
                                KafkaRpcServerManager.this.messageCache.remove(rpcId);
                                KafkaRpcServerManager.this.currentChunkCache.remove(rpcId);
                            }
                            Tracer.SpanBuilder spanBuilder = this.buildSpanFromRpcMessage(rpcMessage);
                            Span minionSpan = spanBuilder.start();
                            RpcRequest request = this.module.unmarshalRequest(rpcContent.toStringUtf8());
                            this.setTagsOnMinion(rpcMessage, request, minionSpan);
                            CompletableFuture future = this.module.execute(request);
                            future.whenComplete((res, ex) -> {
                                RpcResponse response;
                                if (ex != null) {
                                    LOG.warn("An error occured while executing a call in {}.", (Object)this.module.getId(), ex);
                                    response = this.module.createResponseWithException(ex);
                                    minionSpan.log(ex.getMessage());
                                    minionSpan.setTag("failed", "true");
                                } else {
                                    response = res;
                                }
                                minionSpan.finish();
                                this.sendResponse(rpcId, response);
                            });
                        }
                        catch (InvalidProtocolBufferException e) {
                            LOG.error("error while parsing the request", (Throwable)e);
                        }
                    }
                }
            }
            catch (WakeupException e) {
                if (!this.closed.get()) {
                    throw e;
                }
            }
            finally {
                this.consumer.close();
            }
        }

        private void sendResponse(String rpcId, RpcResponse response) {
            try {
                JmsQueueNameFactory topicNameFactory = new JmsQueueNameFactory("rpc-response", this.module.getId());
                String responseAsString = this.module.marshalResponse(response);
                byte[] messageInBytes = responseAsString.getBytes();
                int totalChunks = IntMath.divide((int)messageInBytes.length, (int)KafkaRpcServerManager.this.maxBufferSize, (RoundingMode)RoundingMode.UP);
                RpcMessageProtos.RpcMessage.Builder builder = RpcMessageProtos.RpcMessage.newBuilder().setRpcId(rpcId);
                builder.setTotalChunks(totalChunks);
                for (int chunk = 0; chunk < totalChunks; ++chunk) {
                    int bufferSize = KafkaRpcConstants.getBufferSize((int)messageInBytes.length, (int)KafkaRpcServerManager.this.maxBufferSize, (int)chunk);
                    ByteString byteString = ByteString.copyFrom((byte[])messageInBytes, (int)(chunk * KafkaRpcServerManager.this.maxBufferSize), (int)bufferSize);
                    RpcMessageProtos.RpcMessage rpcMessage = builder.setCurrentChunkNumber(chunk).setRpcContent(byteString).build();
                    this.sendMessageToKafka(rpcMessage, topicNameFactory.getName(), responseAsString);
                }
            }
            catch (Throwable t) {
                LOG.error("Marshalling response in RPC module {} failed.", this.module, (Object)t);
            }
        }

        void sendMessageToKafka(RpcMessageProtos.RpcMessage rpcMessage, String topic, String responseAsString) {
            String rpcId = rpcMessage.getRpcId();
            int chunkNum = rpcMessage.getCurrentChunkNumber();
            ProducerRecord producerRecord = new ProducerRecord(topic, (Object)rpcMessage.getRpcId(), (Object)rpcMessage.toByteArray());
            KafkaRpcServerManager.this.producer.send(producerRecord, (recordMetadata, e) -> {
                if (e != null) {
                    RATE_LIMITED_LOG.error(" RPC response {} with id {} couldn't be sent to Kafka", new Object[]{rpcMessage, rpcId, e});
                } else if (LOG.isTraceEnabled()) {
                    LOG.trace("request with id {} executed, sending response {}, chunk number {} ", new Object[]{rpcId, responseAsString, chunkNum});
                }
            });
        }

        private boolean handleDirectedRPC(RpcMessageProtos.RpcMessage rpcMessage) {
            String messageId = rpcMessage.getRpcId();
            if (rpcMessage.getTotalChunks() > 1) {
                messageId = messageId + rpcMessage.getCurrentChunkNumber();
            }
            if (KafkaRpcServerManager.this.rpcIdQueue.contains(new RpcId(messageId, rpcMessage.getExpirationTime())) || rpcMessage.getExpirationTime() < System.currentTimeMillis()) {
                return true;
            }
            KafkaRpcServerManager.this.rpcIdQueue.offer(new RpcId(messageId, rpcMessage.getExpirationTime()));
            return false;
        }

        private boolean handleChunks(RpcMessageProtos.RpcMessage rpcMessage) {
            String rpcId = rpcMessage.getRpcId();
            KafkaRpcServerManager.this.currentChunkCache.putIfAbsent(rpcId, 0);
            Integer chunkNumber = (Integer)KafkaRpcServerManager.this.currentChunkCache.get(rpcId);
            if (chunkNumber.intValue() != rpcMessage.getCurrentChunkNumber()) {
                LOG.debug("Expected chunk = {} but got chunk = {}, ignoring.", (Object)chunkNumber, (Object)rpcMessage.getCurrentChunkNumber());
                return false;
            }
            ByteString byteString = (ByteString)KafkaRpcServerManager.this.messageCache.get(rpcId);
            if (byteString != null) {
                KafkaRpcServerManager.this.messageCache.put(rpcId, byteString.concat(rpcMessage.getRpcContent()));
            } else {
                KafkaRpcServerManager.this.messageCache.put(rpcId, rpcMessage.getRpcContent());
            }
            chunkNumber = chunkNumber + 1;
            KafkaRpcServerManager.this.currentChunkCache.put(rpcId, chunkNumber);
            return rpcMessage.getTotalChunks() == chunkNumber.intValue();
        }

        private Tracer.SpanBuilder buildSpanFromRpcMessage(RpcMessageProtos.RpcMessage rpcMessage) {
            Tracer tracer = KafkaRpcServerManager.this.tracerRegistry.getTracer();
            HashMap tracingInfoMap = new HashMap();
            rpcMessage.getTracingInfoList().forEach(tracingInfo -> tracingInfoMap.put(tracingInfo.getKey(), tracingInfo.getValue()));
            SpanContext context = tracer.extract(Format.Builtin.TEXT_MAP, (Object)new TextMapExtractAdapter(tracingInfoMap));
            Tracer.SpanBuilder spanBuilder = context != null ? tracer.buildSpan(this.module.getId()).asChildOf(context) : tracer.buildSpan(this.module.getId());
            return spanBuilder;
        }

        private void setTagsOnMinion(RpcMessageProtos.RpcMessage rpcMessage, RpcRequest request, Span minionSpan) {
            rpcMessage.getTracingInfoList().forEach(tracingInfo -> minionSpan.setTag(tracingInfo.getKey(), tracingInfo.getValue()));
            minionSpan.setTag("location", request.getLocation());
            if (request.getSystemId() != null) {
                minionSpan.setTag("systemId", request.getSystemId());
            }
        }
    }
}

