/*
 * Decompiled with CFR 0.152.
 */
package org.opennms.core.ipc.rpc.kafka;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.protobuf.ByteString;
import com.google.protobuf.InvalidProtocolBufferException;
import java.io.IOException;
import java.util.Arrays;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.common.errors.WakeupException;
import org.apache.kafka.common.serialization.ByteArrayDeserializer;
import org.apache.kafka.common.serialization.ByteArraySerializer;
import org.apache.kafka.common.serialization.StringDeserializer;
import org.apache.kafka.common.serialization.StringSerializer;
import org.opennms.core.camel.JmsQueueNameFactory;
import org.opennms.core.ipc.common.kafka.KafkaConfigProvider;
import org.opennms.core.ipc.common.kafka.Utils;
import org.opennms.core.ipc.rpc.kafka.model.RpcMessageProtos;
import org.opennms.core.rpc.api.RpcModule;
import org.opennms.core.rpc.api.RpcRequest;
import org.opennms.core.rpc.api.RpcResponse;
import org.opennms.distributed.core.api.MinionIdentity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KafkaRpcServerManager {
    private static final Logger LOG = LoggerFactory.getLogger(KafkaRpcServerManager.class);
    private final Map<String, RpcModule<RpcRequest, RpcResponse>> registerdModules = new ConcurrentHashMap<String, RpcModule<RpcRequest, RpcResponse>>();
    private final Properties kafkaConfig = new Properties();
    private final KafkaConfigProvider kafkaConfigProvider;
    private KafkaProducer<String, byte[]> producer;
    private MinionIdentity minionIdentity;
    private final ThreadFactory threadFactory = new ThreadFactoryBuilder().setNameFormat("rpc-server-kafka-consumer-%d").build();
    private final ExecutorService executor = Executors.newCachedThreadPool(this.threadFactory);
    private Map<RpcModule<RpcRequest, RpcResponse>, KafkaConsumerRunner> rpcModuleConsumers = new ConcurrentHashMap<RpcModule<RpcRequest, RpcResponse>, KafkaConsumerRunner>();
    private Cache<String, Long> rpcIdCache;

    public KafkaRpcServerManager(KafkaConfigProvider configProvider, MinionIdentity minionIdentity) {
        this.kafkaConfigProvider = configProvider;
        this.minionIdentity = minionIdentity;
    }

    public void init() throws IOException {
        this.kafkaConfig.put("group.id", this.minionIdentity.getLocation());
        this.kafkaConfig.put("enable.auto.commit", "true");
        this.kafkaConfig.put("key.deserializer", StringDeserializer.class.getCanonicalName());
        this.kafkaConfig.put("value.deserializer", ByteArrayDeserializer.class.getCanonicalName());
        this.kafkaConfig.put("auto.commit.interval.ms", "1000");
        this.kafkaConfig.put("auto.offset.reset", "earliest");
        this.kafkaConfig.put("key.serializer", StringSerializer.class.getCanonicalName());
        this.kafkaConfig.put("value.serializer", ByteArraySerializer.class.getCanonicalName());
        this.kafkaConfig.putAll((Map<?, ?>)this.kafkaConfigProvider.getProperties());
        LOG.info("initializing the Kafka producer with: {}", (Object)this.kafkaConfig);
        this.producer = (KafkaProducer)Utils.runWithGivenClassLoader(() -> new KafkaProducer(this.kafkaConfig), (ClassLoader)KafkaProducer.class.getClassLoader());
        String cacheConfig = this.kafkaConfig.getProperty("rpcid.cache.config", "maximumSize=1000,expireAfterWrite=10m");
        this.rpcIdCache = CacheBuilder.from((String)cacheConfig).build();
    }

    public void bind(RpcModule module) throws Exception {
        if (module != null) {
            RpcModule rpcModule = module;
            if (this.registerdModules.containsKey(rpcModule.getId())) {
                LOG.warn(" {} module is already registered", (Object)rpcModule.getId());
            } else {
                this.registerdModules.put(rpcModule.getId(), (RpcModule<RpcRequest, RpcResponse>)rpcModule);
                this.startConsumerForModule((RpcModule<RpcRequest, RpcResponse>)rpcModule);
            }
        }
    }

    private void startConsumerForModule(RpcModule<RpcRequest, RpcResponse> rpcModule) {
        JmsQueueNameFactory topicNameFactory = new JmsQueueNameFactory("rpc-request", rpcModule.getId(), this.minionIdentity.getLocation());
        KafkaConsumer consumer = (KafkaConsumer)Utils.runWithGivenClassLoader(() -> new KafkaConsumer(this.kafkaConfig), (ClassLoader)KafkaConsumer.class.getClassLoader());
        KafkaConsumerRunner kafkaConsumerRunner = new KafkaConsumerRunner(rpcModule, (KafkaConsumer<String, byte[]>)consumer, topicNameFactory.getName());
        this.executor.execute(kafkaConsumerRunner);
        LOG.info("started kafka consumer for module : {}", (Object)rpcModule.getId());
        this.rpcModuleConsumers.put(rpcModule, kafkaConsumerRunner);
    }

    public void unbind(RpcModule module) throws Exception {
        if (module != null) {
            RpcModule rpcModule = module;
            this.registerdModules.remove(rpcModule.getId());
            this.stopConsumerForModule((RpcModule<RpcRequest, RpcResponse>)rpcModule);
        }
    }

    private void stopConsumerForModule(RpcModule<RpcRequest, RpcResponse> rpcModule) {
        KafkaConsumerRunner kafkaConsumerRunner = this.rpcModuleConsumers.remove(rpcModule);
        LOG.info("stopped kafka consumer for module : {}", (Object)rpcModule.getId());
        kafkaConsumerRunner.shutdown();
    }

    public void destroy() {
    }

    public MinionIdentity getMinionIdentity() {
        return this.minionIdentity;
    }

    public Cache<String, Long> getRpcIdCache() {
        return this.rpcIdCache;
    }

    private class KafkaConsumerRunner
    implements Runnable {
        private final KafkaConsumer<String, byte[]> consumer;
        private final AtomicBoolean closed = new AtomicBoolean(false);
        private String topic;
        private RpcModule<RpcRequest, RpcResponse> module;

        public KafkaConsumerRunner(RpcModule<RpcRequest, RpcResponse> rpcModule, KafkaConsumer<String, byte[]> consumer, String topic) {
            this.consumer = consumer;
            this.topic = topic;
            this.module = rpcModule;
        }

        public void shutdown() {
            this.closed.set(true);
            this.consumer.wakeup();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            try {
                this.consumer.subscribe(Arrays.asList(this.topic));
                LOG.info("subscribed to topic {}", (Object)this.topic);
                while (!this.closed.get()) {
                    ConsumerRecords records = this.consumer.poll(Long.MAX_VALUE);
                    for (ConsumerRecord record : records) {
                        try {
                            RpcMessageProtos.RpcMessage rpcMessage = RpcMessageProtos.RpcMessage.parseFrom((byte[])record.value());
                            String rpcId = rpcMessage.getRpcId();
                            long expirationTime = rpcMessage.getExpirationTime();
                            if (expirationTime < System.currentTimeMillis()) {
                                LOG.debug("ttl already expired for the request id = {}, won't process.", (Object)rpcMessage.getRpcId());
                                continue;
                            }
                            boolean hasSystemId = rpcMessage.hasSystemId();
                            String minionId = KafkaRpcServerManager.this.getMinionIdentity().getId();
                            if (hasSystemId && !minionId.equals(rpcMessage.getSystemId())) {
                                LOG.debug("MinionIdentity {} doesn't match with systemId {}, ignore the request", (Object)minionId, (Object)rpcMessage.getSystemId());
                                continue;
                            }
                            if (hasSystemId) {
                                Long cachedTime = (Long)KafkaRpcServerManager.this.rpcIdCache.getIfPresent((Object)rpcId);
                                if (cachedTime != null) continue;
                                KafkaRpcServerManager.this.rpcIdCache.put((Object)rpcId, (Object)System.currentTimeMillis());
                            }
                            RpcRequest request = this.module.unmarshalRequest(rpcMessage.getRpcContent().toStringUtf8());
                            CompletableFuture future = this.module.execute(request);
                            future.whenComplete((res, ex) -> {
                                RpcResponse response;
                                if (ex != null) {
                                    LOG.warn("An error occured while executing a call in {}.", (Object)this.module.getId(), ex);
                                    response = this.module.createResponseWithException(ex);
                                } else {
                                    response = res;
                                }
                                String responseAsString = null;
                                try {
                                    responseAsString = this.module.marshalResponse(response);
                                    JmsQueueNameFactory topicNameFactory = new JmsQueueNameFactory("rpc-response", this.module.getId());
                                    RpcMessageProtos.RpcMessage rpcResponse = RpcMessageProtos.RpcMessage.newBuilder().setRpcId(rpcId).setRpcContent(ByteString.copyFromUtf8((String)responseAsString)).build();
                                    ProducerRecord producerRecord = new ProducerRecord(topicNameFactory.getName(), (Object)rpcId, (Object)rpcResponse.toByteArray());
                                    KafkaRpcServerManager.this.producer.send(producerRecord);
                                    LOG.debug("request with id {} executed, sending response {} ", (Object)rpcId, (Object)responseAsString);
                                }
                                catch (Throwable t) {
                                    LOG.error("Marshalling response in RPC module {} failed.", this.module, (Object)t);
                                }
                            });
                        }
                        catch (InvalidProtocolBufferException e) {
                            LOG.error("error while parsing the request", (Throwable)e);
                        }
                    }
                }
            }
            catch (WakeupException e) {
                if (!this.closed.get()) {
                    throw e;
                }
            }
            finally {
                this.consumer.close();
            }
        }
    }
}

