/*
 * Decompiled with CFR 0.152.
 */
package org.opennms.core.ipc.rpc.kafka;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.protobuf.ByteString;
import com.google.protobuf.InvalidProtocolBufferException;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.Delayed;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.common.errors.WakeupException;
import org.apache.kafka.common.serialization.ByteArrayDeserializer;
import org.apache.kafka.common.serialization.ByteArraySerializer;
import org.apache.kafka.common.serialization.StringDeserializer;
import org.apache.kafka.common.serialization.StringSerializer;
import org.opennms.core.camel.JmsQueueNameFactory;
import org.opennms.core.ipc.common.kafka.OnmsKafkaConfigProvider;
import org.opennms.core.ipc.rpc.kafka.ResponseCallback;
import org.opennms.core.ipc.rpc.kafka.model.RpcMessageProtos;
import org.opennms.core.logging.Logging;
import org.opennms.core.rpc.api.RemoteExecutionException;
import org.opennms.core.rpc.api.RequestTimedOutException;
import org.opennms.core.rpc.api.RpcClient;
import org.opennms.core.rpc.api.RpcClientFactory;
import org.opennms.core.rpc.api.RpcModule;
import org.opennms.core.rpc.api.RpcRequest;
import org.opennms.core.rpc.api.RpcResponse;
import org.opennms.core.utils.SystemInfoUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KafkaRpcClientFactory
implements RpcClientFactory {
    private static final Logger LOG = LoggerFactory.getLogger(KafkaRpcClientFactory.class);
    private static final Long TIMEOUT_FOR_KAFKA_RPC = Long.getLong(String.format("%sttl", "org.opennms.core.ipc.rpc.kafka."), 30000L);
    private String location;
    private KafkaProducer<String, byte[]> producer;
    private final Properties kafkaConfig = new Properties();
    private final ThreadFactory threadFactory = new ThreadFactoryBuilder().setNameFormat("rpc-client-kafka-consumer-%d").build();
    private final ThreadFactory timerThreadFactory = new ThreadFactoryBuilder().setNameFormat("rpc-client-timeout-tracker-%d").build();
    private final ExecutorService executor = Executors.newSingleThreadExecutor(this.threadFactory);
    private final ExecutorService timerExecutor = Executors.newSingleThreadExecutor(this.timerThreadFactory);
    private final Map<String, ResponseCallback> rpcResponseMap = new ConcurrentHashMap<String, ResponseCallback>();
    private KafkaConsumerRunner kafkaConsumerRunner;
    private DelayQueue<ResponseCallback> delayQueue = new DelayQueue();

    public <S extends RpcRequest, T extends RpcResponse> RpcClient<S, T> getClient(final RpcModule<S, T> module) {
        return new RpcClient<S, T>(){

            public CompletableFuture<T> execute(S request) {
                if (request.getLocation() == null || request.getLocation().equals(KafkaRpcClientFactory.this.location)) {
                    return module.execute(request);
                }
                JmsQueueNameFactory topicNameFactory = new JmsQueueNameFactory("rpc-request", module.getId(), request.getLocation());
                String requestTopic = topicNameFactory.getName();
                String marshalledRequest = module.marshalRequest(request);
                String rpcId = UUID.randomUUID().toString();
                long ttl = request.getTimeToLiveMs() != null ? request.getTimeToLiveMs().longValue() : TIMEOUT_FOR_KAFKA_RPC.longValue();
                ttl = ttl > 0L ? ttl : TIMEOUT_FOR_KAFKA_RPC;
                long expirationTime = System.currentTimeMillis() + ttl;
                CompletableFuture future = new CompletableFuture();
                Map loggingContext = Logging.getCopyOfContextMap();
                ResponseHandler responseHandler = new ResponseHandler(future, module, rpcId, expirationTime, loggingContext);
                KafkaRpcClientFactory.this.delayQueue.offer(responseHandler);
                KafkaRpcClientFactory.this.rpcResponseMap.put(rpcId, responseHandler);
                KafkaRpcClientFactory.this.kafkaConsumerRunner.startConsumingForModule(module.getId());
                if (request.getSystemId() != null) {
                    RpcMessageProtos.RpcMessage rpcMessage = RpcMessageProtos.RpcMessage.newBuilder().setRpcId(rpcId).setSystemId(request.getSystemId()).setExpirationTime(expirationTime).setRpcContent(ByteString.copyFromUtf8((String)marshalledRequest)).build();
                    List partitionInfo = KafkaRpcClientFactory.this.producer.partitionsFor(requestTopic);
                    partitionInfo.stream().forEach(partition -> {
                        ProducerRecord record = new ProducerRecord(requestTopic, Integer.valueOf(partition.partition()), (Object)rpcId, (Object)rpcMessage.toByteArray());
                        KafkaRpcClientFactory.this.producer.send(record);
                    });
                } else {
                    RpcMessageProtos.RpcMessage rpcMessage = RpcMessageProtos.RpcMessage.newBuilder().setRpcId(rpcId).setExpirationTime(expirationTime).setRpcContent(ByteString.copyFromUtf8((String)marshalledRequest)).build();
                    ProducerRecord record = new ProducerRecord(requestTopic, (Object)rpcId, (Object)rpcMessage.toByteArray());
                    KafkaRpcClientFactory.this.producer.send(record);
                }
                LOG.debug("RPC Request {} with id {} sent to minion at location {}", new Object[]{request, rpcId, request.getLocation()});
                return future;
            }
        };
    }

    public void setLocation(String location) {
        this.location = location;
    }

    public void start() {
        try (Logging.MDCCloseable mdc = Logging.withPrefixCloseable((String)"ipc");){
            this.kafkaConfig.clear();
            this.kafkaConfig.put("group.id", SystemInfoUtils.getInstanceId());
            this.kafkaConfig.put("enable.auto.commit", "true");
            this.kafkaConfig.put("key.deserializer", StringDeserializer.class.getCanonicalName());
            this.kafkaConfig.put("value.deserializer", ByteArrayDeserializer.class.getCanonicalName());
            this.kafkaConfig.put("auto.commit.interval.ms", "1000");
            this.kafkaConfig.put("auto.offset.reset", "earliest");
            this.kafkaConfig.put("key.serializer", StringSerializer.class.getCanonicalName());
            this.kafkaConfig.put("value.serializer", ByteArraySerializer.class.getCanonicalName());
            OnmsKafkaConfigProvider kafkaConfigProvider = new OnmsKafkaConfigProvider("org.opennms.core.ipc.rpc.kafka.");
            this.kafkaConfig.putAll((Map<?, ?>)kafkaConfigProvider.getProperties());
            this.producer = new KafkaProducer(this.kafkaConfig);
            LOG.info("initializing the Kafka producer with: {}", (Object)this.kafkaConfig);
            KafkaConsumer kafkaConsumer = new KafkaConsumer(this.kafkaConfig);
            this.kafkaConsumerRunner = new KafkaConsumerRunner((KafkaConsumer<String, byte[]>)kafkaConsumer);
            this.executor.execute(this.kafkaConsumerRunner);
            LOG.info("started  kafka consumer with : {}", (Object)this.kafkaConfig);
            this.timerExecutor.execute(() -> {
                while (true) {
                    try {
                        while (true) {
                            ResponseCallback responseCb;
                            if ((responseCb = (ResponseCallback)this.delayQueue.take()).isProcessed()) {
                                continue;
                            }
                            LOG.info("RPC request with id {} timedout ", (Object)responseCb.getRpcId());
                            responseCb.sendResponse(null);
                        }
                    }
                    catch (InterruptedException e) {
                        LOG.info("interrupted while waiting for an element from delayQueue {}", (Throwable)e);
                    }
                    catch (Exception e) {
                        LOG.warn("error while sending response from timeout handler {}", (Throwable)e);
                        continue;
                    }
                    break;
                }
            });
            LOG.info("started timeout tracker");
        }
    }

    public void stop() {
        LOG.info("stop kafka consumer runner");
        this.kafkaConsumerRunner.stop();
        this.executor.shutdown();
        this.timerExecutor.shutdown();
    }

    private class KafkaConsumerRunner
    implements Runnable {
        private final KafkaConsumer<String, byte[]> consumer;
        private final AtomicBoolean closed = new AtomicBoolean(false);
        private final Set<String> moduleIdsForTopics = new HashSet<String>();
        private final Set<String> topics = new HashSet<String>();
        private final AtomicBoolean topicAdded = new AtomicBoolean(false);
        private final CountDownLatch firstTopicAdded = new CountDownLatch(1);

        public KafkaConsumerRunner(KafkaConsumer<String, byte[]> consumer) {
            this.consumer = consumer;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            Logging.putPrefix((String)"ipc");
            if (this.topics.isEmpty()) {
                while (!this.topicAdded.get()) {
                    try {
                        this.firstTopicAdded.await(1L, TimeUnit.SECONDS);
                    }
                    catch (InterruptedException e) {
                        LOG.info("Interrupted before first topic was added. Terminating Kafka RPC consumer thread.");
                        return;
                    }
                }
                LOG.info("First topic is added, consumer will be started.");
            }
            while (!this.closed.get()) {
                if (this.topicAdded.get()) {
                    Set<String> e = this.topics;
                    synchronized (e) {
                        LOG.info("Subscribing Kafka RPC consumer to topics named: {}", this.topics);
                        this.consumer.subscribe(this.topics);
                        this.topicAdded.set(false);
                    }
                }
                try {
                    ConsumerRecords records = this.consumer.poll(Long.MAX_VALUE);
                    for (ConsumerRecord record : records) {
                        ResponseCallback responseCb = (ResponseCallback)KafkaRpcClientFactory.this.rpcResponseMap.get(record.key());
                        if (responseCb != null) {
                            RpcMessageProtos.RpcMessage rpcMessage = RpcMessageProtos.RpcMessage.parseFrom((byte[])record.value());
                            responseCb.sendResponse(rpcMessage.getRpcContent().toStringUtf8());
                            continue;
                        }
                        LOG.warn("Received a response for request with ID:{}, but no outstanding request was found with this id, The request may have timed out.", record.key());
                    }
                }
                catch (InvalidProtocolBufferException e) {
                    LOG.error("error while parsing response", (Throwable)e);
                }
                catch (WakeupException e) {
                    LOG.info(" consumer got wakeup exception, closed = {} ", (Object)this.closed.get(), (Object)e);
                }
            }
            this.consumer.close();
        }

        public void stop() {
            this.closed.set(true);
            this.consumer.wakeup();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void startConsumingForModule(String moduleId) {
            if (this.moduleIdsForTopics.contains(moduleId)) {
                return;
            }
            this.moduleIdsForTopics.add(moduleId);
            JmsQueueNameFactory topicNameFactory = new JmsQueueNameFactory("rpc-response", moduleId);
            Set<String> set = this.topics;
            synchronized (set) {
                if (this.topics.add(topicNameFactory.getName())) {
                    this.topicAdded.set(true);
                    this.firstTopicAdded.countDown();
                    this.consumer.wakeup();
                }
            }
        }
    }

    private class ResponseHandler<S extends RpcRequest, T extends RpcResponse>
    implements ResponseCallback {
        private final CompletableFuture<T> responseFuture;
        private final RpcModule<S, T> rpcModule;
        private final long expirationTime;
        private final String rpcId;
        private Map<String, String> loggingContext;
        private boolean isProcessed = false;

        public ResponseHandler(CompletableFuture<T> responseFuture, RpcModule<S, T> rpcModule, String rpcId, long timeout, Map<String, String> loggingContext) {
            this.responseFuture = responseFuture;
            this.rpcModule = rpcModule;
            this.expirationTime = timeout;
            this.rpcId = rpcId;
            this.loggingContext = loggingContext;
        }

        @Override
        public void sendResponse(String message) {
            RpcResponse response = null;
            try (Logging.MDCCloseable mdc = Logging.withContextMapCloseable(this.loggingContext);){
                if (message != null) {
                    response = this.rpcModule.unmarshalResponse(message);
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Received RPC response {} for id {}", (Object)message, (Object)this.rpcId);
                    }
                    if (response.getErrorMessage() != null) {
                        this.responseFuture.completeExceptionally((Throwable)new RemoteExecutionException(response.getErrorMessage()));
                    } else {
                        this.responseFuture.complete(response);
                    }
                    this.isProcessed = true;
                } else {
                    this.responseFuture.completeExceptionally((Throwable)new RequestTimedOutException((Throwable)new TimeoutException()));
                }
                KafkaRpcClientFactory.this.rpcResponseMap.remove(this.rpcId);
            }
            catch (Exception e) {
                LOG.warn("error while handling response for RPC request id {}", (Object)this.rpcId, (Object)e);
            }
        }

        @Override
        public int compareTo(Delayed other) {
            long otherDelay;
            long myDelay = this.getDelay(TimeUnit.MILLISECONDS);
            if (myDelay < (otherDelay = other.getDelay(TimeUnit.MILLISECONDS))) {
                return -1;
            }
            if (myDelay == otherDelay) {
                return 0;
            }
            return 1;
        }

        @Override
        public long getDelay(TimeUnit unit) {
            long now = System.currentTimeMillis();
            return unit.convert(this.expirationTime - now, TimeUnit.MILLISECONDS);
        }

        @Override
        public boolean isProcessed() {
            return this.isProcessed;
        }

        @Override
        public String getRpcId() {
            return this.rpcId;
        }
    }
}

