/*
 * Decompiled with CFR 0.152.
 */
package org.opennms.core.ipc.sink.common;

import com.codahale.metrics.Counter;
import com.codahale.metrics.Gauge;
import com.codahale.metrics.Metric;
import com.codahale.metrics.MetricRegistry;
import com.swrve.ratelimitedlogger.RateLimitedLog;
import java.util.AbstractMap;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.joda.time.Duration;
import org.opennms.core.concurrent.LogPreservingThreadFactory;
import org.opennms.core.ipc.sink.api.AsyncDispatcher;
import org.opennms.core.ipc.sink.api.AsyncPolicy;
import org.opennms.core.ipc.sink.api.Message;
import org.opennms.core.ipc.sink.api.OffHeapQueue;
import org.opennms.core.ipc.sink.api.SinkModule;
import org.opennms.core.ipc.sink.api.SyncDispatcher;
import org.opennms.core.ipc.sink.api.WriteFailedException;
import org.opennms.core.ipc.sink.common.DispatcherState;
import org.opennms.core.ipc.sink.offheap.OffHeapServiceLoader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AsyncDispatcherImpl<W, S extends Message, T extends Message>
implements AsyncDispatcher<S> {
    private static final Logger LOG = LoggerFactory.getLogger(AsyncDispatcherImpl.class);
    private final SyncDispatcher<S> syncDispatcher;
    private OffHeapAdapter offHeapAdapter;
    private ExecutorService offHeapAdapterExecutor = Executors.newSingleThreadExecutor();
    private final AsyncPolicy asyncPolicy;
    private OffHeapQueue offHeapQueue;
    private SinkModule<S, T> sinkModule;
    private DispatcherState<W, S, T> state;
    private boolean useOffHeap = false;
    final RateLimitedLog rateLimittedLogger = RateLimitedLog.withRateLimit((Logger)LOG).maxRate(5).every(Duration.standardSeconds((long)30L)).build();
    final LinkedBlockingQueue<Runnable> queue;
    final ExecutorService executor;

    public AsyncDispatcherImpl(DispatcherState<W, S, T> state, AsyncPolicy asyncPolicy, SyncDispatcher<S> syncDispatcher) {
        RejectedExecutionHandler rejectedExecutionHandler;
        Objects.requireNonNull(state);
        Objects.requireNonNull(asyncPolicy);
        this.syncDispatcher = Objects.requireNonNull(syncDispatcher);
        this.asyncPolicy = asyncPolicy;
        this.state = state;
        this.sinkModule = state.getModule();
        if (OffHeapServiceLoader.isOffHeapEnabled()) {
            this.offHeapQueue = OffHeapServiceLoader.getOffHeapQueue();
            if (this.offHeapQueue != null) {
                this.useOffHeap = true;
                LOG.info("Offheap storage enabled for sink module, {}", (Object)this.sinkModule.getId());
            }
        }
        if (asyncPolicy.isBlockWhenFull()) {
            this.queue = new OfferBlockingQueue<Runnable>(asyncPolicy.getQueueSize());
            rejectedExecutionHandler = new ThreadPoolExecutor.AbortPolicy();
        } else {
            this.queue = new LinkedBlockingQueue(asyncPolicy.getQueueSize());
            final Counter droppedCounter = state.getMetrics().counter(MetricRegistry.name((String)state.getModule().getId(), (String[])new String[]{"dropped"}));
            rejectedExecutionHandler = new RejectedExecutionHandler(){

                @Override
                public void rejectedExecution(Runnable r, ThreadPoolExecutor e) {
                    droppedCounter.inc();
                    throw new RejectedExecutionException("Task " + r.toString() + " rejected from " + e.toString());
                }
            };
        }
        state.getMetrics().register(MetricRegistry.name((String)state.getModule().getId(), (String[])new String[]{"queue-size"}), (Metric)new Gauge<Integer>(){

            public Integer getValue() {
                return AsyncDispatcherImpl.this.queue.size();
            }
        });
        this.executor = new ThreadPoolExecutor(asyncPolicy.getNumThreads(), asyncPolicy.getNumThreads(), 1000L, TimeUnit.MILLISECONDS, this.queue, (ThreadFactory)new LogPreservingThreadFactory("OpenNMS.Sink.AsyncDispatcher." + state.getModule().getId(), Integer.MAX_VALUE), rejectedExecutionHandler);
    }

    public CompletableFuture<S> send(S message) {
        if (this.useOffHeap && (this.asyncPolicy.getQueueSize() == this.getQueueSize() || this.offHeapAdapter != null && !this.offHeapAdapter.isOffHeapEmpty())) {
            if (this.offHeapAdapter == null) {
                this.offHeapAdapter = new OffHeapAdapter();
                this.offHeapAdapterExecutor.execute(this.offHeapAdapter);
                LOG.info("started drain thread for {}", (Object)this.sinkModule.getId());
            }
            try {
                return this.offHeapAdapter.writeMessage(message);
            }
            catch (WriteFailedException e) {
                this.rateLimittedLogger.error("OffHeap write failed ", (Throwable)e);
            }
        }
        try {
            return CompletableFuture.supplyAsync(() -> {
                this.syncDispatcher.send((Object)message);
                return message;
            }, this.executor);
        }
        catch (RejectedExecutionException ree) {
            CompletableFuture future = new CompletableFuture();
            future.completeExceptionally(ree);
            return future;
        }
    }

    public int getQueueSize() {
        return this.queue.size();
    }

    public void close() throws Exception {
        this.syncDispatcher.close();
        this.executor.shutdown();
        if (this.offHeapAdapter != null) {
            this.offHeapAdapter.shutdown();
            this.offHeapAdapterExecutor.shutdown();
        }
    }

    private class OffHeapAdapter
    implements Runnable {
        private Map<String, CompletableFuture<S>> offHeapFutureMap = new ConcurrentHashMap();
        private final CountDownLatch firstWrite = new CountDownLatch(1);
        private final AtomicBoolean closed = new AtomicBoolean(false);

        public OffHeapAdapter() {
            AsyncDispatcherImpl.this.state.getMetrics().register(MetricRegistry.name((String)AsyncDispatcherImpl.this.state.getModule().getId(), (String[])new String[]{"offheap-messages"}), (Metric)new Gauge<Integer>(){

                public Integer getValue() {
                    return AsyncDispatcherImpl.this.offHeapQueue.getNumOfMessages(AsyncDispatcherImpl.this.sinkModule.getId());
                }
            });
        }

        @Override
        public void run() {
            while (!this.closed.get()) {
                try {
                    this.firstWrite.await();
                    AbstractMap.SimpleImmutableEntry keyValue = AsyncDispatcherImpl.this.offHeapQueue.readNextMessage(AsyncDispatcherImpl.this.sinkModule.getId());
                    if (keyValue == null) continue;
                    AsyncDispatcherImpl.this.queue.put(() -> {
                        Message message = AsyncDispatcherImpl.this.sinkModule.unmarshalSingleMessage((byte[])keyValue.getValue());
                        AsyncDispatcherImpl.this.syncDispatcher.send((Object)message);
                        CompletableFuture future = this.offHeapFutureMap.get(keyValue.getKey());
                        future.complete(message);
                        this.offHeapFutureMap.remove(keyValue.getKey());
                    });
                }
                catch (InterruptedException e) {
                    LOG.warn("Interrupted while retrieving OffHeap Message for {} ", (Object)AsyncDispatcherImpl.this.sinkModule.getId(), (Object)e);
                }
            }
        }

        public CompletableFuture<S> writeMessage(S message) throws WriteFailedException {
            CompletableFuture future = new CompletableFuture();
            byte[] bytes = AsyncDispatcherImpl.this.sinkModule.marshalSingleMessage(message);
            String uuid = UUID.randomUUID().toString();
            this.offHeapFutureMap.put(uuid, future);
            AsyncDispatcherImpl.this.offHeapQueue.writeMessage(bytes, AsyncDispatcherImpl.this.sinkModule.getId(), uuid);
            this.firstWrite.countDown();
            return future;
        }

        public boolean isOffHeapEmpty() {
            return this.offHeapFutureMap.isEmpty();
        }

        public void shutdown() {
            this.firstWrite.countDown();
            this.closed.set(true);
        }
    }

    private static class OfferBlockingQueue<E>
    extends LinkedBlockingQueue<E> {
        private static final long serialVersionUID = 1L;

        public OfferBlockingQueue(int capacity) {
            super(capacity);
        }

        @Override
        public boolean offer(E e) {
            try {
                this.put(e);
                return true;
            }
            catch (InterruptedException ie) {
                Thread.currentThread().interrupt();
                return false;
            }
        }
    }
}

