/*******************************************************************************
 * This file is part of OpenNMS(R).
 *
 * Copyright (C) 2002-2014 The OpenNMS Group, Inc.
 * OpenNMS(R) is Copyright (C) 1999-2014 The OpenNMS Group, Inc.
 *
 * OpenNMS(R) is a registered trademark of The OpenNMS Group, Inc.
 *
 * OpenNMS(R) is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License,
 * or (at your option) any later version.
 *
 * OpenNMS(R) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OpenNMS(R).  If not, see:
 *      http://www.gnu.org/licenses/
 *
 * For more information contact:
 *     OpenNMS(R) Licensing <license@opennms.org>
 *     http://www.opennms.org/
 *     http://www.opennms.com/
 *******************************************************************************/

package org.opennms.netmgt.trapd;

import static org.opennms.core.utils.InetAddressUtils.str;

import java.net.InetAddress;
import java.util.Date;
import java.util.Objects;

import org.opennms.core.utils.InetAddressUtils;
import org.opennms.netmgt.config.api.EventConfDao;
import org.opennms.netmgt.dao.api.InterfaceToNodeCache;
import org.opennms.netmgt.dao.api.MonitoringLocationDao;
import org.opennms.netmgt.events.api.EventConstants;
import org.opennms.netmgt.model.events.EventBuilder;
import org.opennms.netmgt.model.events.snmp.SyntaxToEvent;
import org.opennms.netmgt.snmp.SnmpObjId;
import org.opennms.netmgt.snmp.SnmpResult;
import org.opennms.netmgt.snmp.SnmpValue;
import org.opennms.netmgt.xml.event.Event;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class EventCreator {
	
	private static final Logger LOG = LoggerFactory.getLogger(EventCreator.class);

    private final InterfaceToNodeCache cache;
    private final EventConfDao eventConfDao;

    public EventCreator(InterfaceToNodeCache cache, EventConfDao eventConfDao) {
        this.cache = Objects.requireNonNull(cache);
        this.eventConfDao = Objects.requireNonNull(eventConfDao);
    }

    public Event createEventFrom(final TrapDTO trapDTO, final String systemId, final String location, final InetAddress trapAddress) {
        LOG.debug("{} trap - trapInterface: {}", trapDTO.getVersion(), trapDTO.getAgentAddress());

        // Set event data
        final EventBuilder eventBuilder = new EventBuilder(null, "trapd");
        eventBuilder.setTime(new Date(trapDTO.getCreationTime()));
        eventBuilder.setCommunity(trapDTO.getCommunity());
        eventBuilder.setSnmpTimeStamp(trapDTO.getTimestamp());
        eventBuilder.setSnmpVersion(trapDTO.getVersion());
        eventBuilder.setSnmpHost(str(trapAddress));
        eventBuilder.setInterface(trapAddress);
        eventBuilder.setHost(InetAddressUtils.toIpAddrString(trapDTO.getAgentAddress()));

        // Handle trap identity
        final TrapIdentityDTO trapIdentity = trapDTO.getTrapIdentity();
        if (trapIdentity != null) {
            LOG.debug("Trap Identity {}", trapIdentity);
            eventBuilder.setGeneric(trapIdentity.getGeneric());
            eventBuilder.setSpecific(trapIdentity.getSpecific());
            eventBuilder.setEnterpriseId(trapIdentity.getEnterpriseId());
        }

        // Handle var bindings
        for (SnmpResult eachResult : trapDTO.getResults()) {
            final SnmpObjId name = eachResult.getBase();
            final SnmpValue value = eachResult.getValue();
            eventBuilder.addParam(SyntaxToEvent.processSyntax(name.toString(), value));
            if (EventConstants.OID_SNMP_IFINDEX.isPrefixOf(name)) {
                eventBuilder.setIfIndex(value.toInt());
            }
        }

        // Resolve Node id and set, if known by OpenNMS
        final long nodeId = resolveNodeId(location, trapAddress);
        if (nodeId != -1) {
            eventBuilder.setNodeid(nodeId);
        }

        // If there was no systemId in the trap message, assume that
        // it was generated by this system. Eventd will fill in the
        // systemId of the local system if it remains null here.
        if (systemId != null) {
            eventBuilder.setDistPoller(systemId);
        }

        // Get event template and set uei, if unknown
        final Event event = eventBuilder.getEvent();
        final org.opennms.netmgt.xml.eventconf.Event econf = eventConfDao.findByEvent(event);
        if (econf == null || econf.getUei() == null) {
            event.setUei("uei.opennms.org/default/trap");
        } else {
            event.setUei(econf.getUei());
        }
        return event;
    }

    private long resolveNodeId(String location, InetAddress trapAddress) {
        // If there was no location in the trap message, assume that
        // it was generated in the default location
        if (location == null) {
            return cache.getNodeId(MonitoringLocationDao.DEFAULT_MONITORING_LOCATION_ID, trapAddress);
        }
        return cache.getNodeId(location, trapAddress);
    }
}
