/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.client;

import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Deque;
import java.util.Queue;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReentrantLock;
import org.eclipse.jetty.client.api.Connection;
import org.eclipse.jetty.client.api.Destination;
import org.eclipse.jetty.util.BlockingArrayQueue;
import org.eclipse.jetty.util.Callback;
import org.eclipse.jetty.util.Promise;
import org.eclipse.jetty.util.annotation.ManagedAttribute;
import org.eclipse.jetty.util.annotation.ManagedObject;
import org.eclipse.jetty.util.component.ContainerLifeCycle;
import org.eclipse.jetty.util.component.Dumpable;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;
import org.eclipse.jetty.util.thread.Sweeper;

@ManagedObject(value="The connection pool")
public class DuplexConnectionPool
implements Closeable,
Dumpable,
Sweeper.Sweepable {
    private static final Logger LOG = Log.getLogger(DuplexConnectionPool.class);
    private final AtomicInteger connectionCount = new AtomicInteger();
    private final ReentrantLock lock = new ReentrantLock();
    private final Destination destination;
    private final int maxConnections;
    private final Callback requester;
    private final Deque<Connection> idleConnections;
    private final Queue<Connection> activeConnections;

    public DuplexConnectionPool(Destination destination, int maxConnections, Callback requester) {
        this.destination = destination;
        this.maxConnections = maxConnections;
        this.requester = requester;
        this.idleConnections = new LinkedBlockingDeque<Connection>(maxConnections);
        this.activeConnections = new BlockingArrayQueue(maxConnections);
    }

    @ManagedAttribute(value="The number of connections", readonly=true)
    public int getConnectionCount() {
        return this.connectionCount.get();
    }

    @ManagedAttribute(value="The number of idle connections", readonly=true)
    public int getIdleConnectionCount() {
        this.lock();
        try {
            int n = this.idleConnections.size();
            return n;
        }
        finally {
            this.unlock();
        }
    }

    @ManagedAttribute(value="The number of active connections", readonly=true)
    public int getActiveConnectionCount() {
        this.lock();
        try {
            int n = this.activeConnections.size();
            return n;
        }
        finally {
            this.unlock();
        }
    }

    public Queue<Connection> getIdleConnections() {
        return this.idleConnections;
    }

    public Queue<Connection> getActiveConnections() {
        return this.activeConnections;
    }

    public Connection acquire() {
        Connection connection = this.activateIdle();
        if (connection == null) {
            connection = this.tryCreate();
        }
        return connection;
    }

    private Connection tryCreate() {
        int next;
        int current;
        do {
            if ((next = (current = this.getConnectionCount()) + 1) <= this.maxConnections) continue;
            if (LOG.isDebugEnabled()) {
                LOG.debug("Max connections {}/{} reached", new Object[]{current, this.maxConnections});
            }
            return this.activateIdle();
        } while (!this.connectionCount.compareAndSet(current, next));
        if (LOG.isDebugEnabled()) {
            LOG.debug("Connection {}/{} creation", new Object[]{next, this.maxConnections});
        }
        this.destination.newConnection(new Promise<Connection>(){

            public void succeeded(Connection connection) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Connection {}/{} creation succeeded {}", new Object[]{next, DuplexConnectionPool.this.maxConnections, connection});
                }
                DuplexConnectionPool.this.idleCreated(connection);
                DuplexConnectionPool.this.proceed();
            }

            public void failed(Throwable x) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Connection " + next + "/" + DuplexConnectionPool.this.maxConnections + " creation failed", x);
                }
                DuplexConnectionPool.this.connectionCount.decrementAndGet();
                DuplexConnectionPool.this.requester.failed(x);
            }
        });
        return this.activateIdle();
    }

    protected void proceed() {
        this.requester.succeeded();
    }

    protected void idleCreated(Connection connection) {
        boolean idle;
        this.lock();
        try {
            idle = this.idleConnections.offerLast(connection);
        }
        finally {
            this.unlock();
        }
        this.idle(connection, idle);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Connection activateIdle() {
        boolean acquired;
        Connection connection;
        this.lock();
        try {
            connection = this.idleConnections.pollFirst();
            if (connection == null) {
                Connection connection2 = null;
                return connection2;
            }
            acquired = this.activeConnections.offer(connection);
        }
        finally {
            this.unlock();
        }
        if (acquired) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Connection active {}", new Object[]{connection});
            }
            this.acquired(connection);
            return connection;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Connection active overflow {}", new Object[]{connection});
        }
        connection.close();
        return null;
    }

    protected void acquired(Connection connection) {
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean release(Connection connection) {
        boolean idle;
        this.lock();
        try {
            if (!this.activeConnections.remove(connection)) {
                boolean bl = false;
                return bl;
            }
            idle = this.offerIdle(connection);
        }
        finally {
            this.unlock();
        }
        this.released(connection);
        return this.idle(connection, idle);
    }

    protected boolean offerIdle(Connection connection) {
        return this.idleConnections.offerFirst(connection);
    }

    protected boolean idle(Connection connection, boolean idle) {
        if (idle) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Connection idle {}", new Object[]{connection});
            }
            return true;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Connection idle overflow {}", new Object[]{connection});
        }
        connection.close();
        return false;
    }

    protected void released(Connection connection) {
    }

    public boolean remove(Connection connection) {
        return this.remove(connection, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected boolean remove(Connection connection, boolean force) {
        boolean removed;
        boolean idleRemoved;
        boolean activeRemoved;
        this.lock();
        try {
            activeRemoved = this.activeConnections.remove(connection);
            idleRemoved = this.idleConnections.remove(connection);
        }
        finally {
            this.unlock();
        }
        if (activeRemoved || force) {
            this.released(connection);
        }
        boolean bl = removed = activeRemoved || idleRemoved || force;
        if (removed) {
            int pooled = this.connectionCount.decrementAndGet();
            if (LOG.isDebugEnabled()) {
                LOG.debug("Connection removed {} - pooled: {}", new Object[]{connection, pooled});
            }
        }
        return removed;
    }

    public boolean isActive(Connection connection) {
        this.lock();
        try {
            boolean bl = this.activeConnections.contains(connection);
            return bl;
        }
        finally {
            this.unlock();
        }
    }

    public boolean isIdle(Connection connection) {
        this.lock();
        try {
            boolean bl = this.idleConnections.contains(connection);
            return bl;
        }
        finally {
            this.unlock();
        }
    }

    public boolean isEmpty() {
        return this.connectionCount.get() == 0;
    }

    @Override
    public void close() {
        ArrayList<Connection> idles = new ArrayList<Connection>();
        ArrayList<Connection> actives = new ArrayList<Connection>();
        this.lock();
        try {
            idles.addAll(this.idleConnections);
            this.idleConnections.clear();
            actives.addAll(this.activeConnections);
            this.activeConnections.clear();
        }
        finally {
            this.unlock();
        }
        this.connectionCount.set(0);
        for (Connection connection : idles) {
            connection.close();
        }
        for (Connection connection : actives) {
            connection.close();
        }
    }

    public String dump() {
        return ContainerLifeCycle.dump((Dumpable)this);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void dump(Appendable out, String indent) throws IOException {
        ArrayList<Connection> actives = new ArrayList<Connection>();
        ArrayList<Connection> idles = new ArrayList<Connection>();
        this.lock();
        try {
            actives.addAll(this.activeConnections);
            idles.addAll(this.idleConnections);
        }
        finally {
            this.unlock();
        }
        ContainerLifeCycle.dumpObject((Appendable)out, (Object)this);
        ContainerLifeCycle.dump((Appendable)out, (String)indent, (Collection[])new Collection[]{actives, idles});
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean sweep() {
        ArrayList<Connection> toSweep = new ArrayList<Connection>();
        this.lock();
        try {
            for (Connection connection : this.activeConnections) {
                if (!(connection instanceof Sweeper.Sweepable)) continue;
                toSweep.add(connection);
            }
        }
        finally {
            this.unlock();
        }
        for (Connection connection : toSweep) {
            if (!((Sweeper.Sweepable)connection).sweep()) continue;
            boolean removed = this.remove(connection, true);
            LOG.warn("Connection swept: {}{}{} from active connections{}{}", new Object[]{connection, System.lineSeparator(), removed ? "Removed" : "Not removed", System.lineSeparator(), this.dump()});
        }
        return false;
    }

    protected void lock() {
        this.lock.lock();
    }

    protected void unlock() {
        this.lock.unlock();
    }

    public String toString() {
        int idleSize;
        int activeSize;
        this.lock();
        try {
            activeSize = this.activeConnections.size();
            idleSize = this.idleConnections.size();
        }
        finally {
            this.unlock();
        }
        return String.format("%s[c=%d/%d,a=%d,i=%d]", this.getClass().getSimpleName(), this.connectionCount.get(), this.maxConnections, activeSize, idleSize);
    }
}

