/* XmpVar.java

   COPYRIGHT 2004,7 KRUPCZAK.ORG, LLC.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
   USA
 
   For more information, visit:
   http://www.krupczak.org/
*/

package org.krupczak.Xmp;

import java.net.InetAddress;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.io.PrintStream;
import javax.swing.*;

/** 
 * Base class serving as the internal representation of an XMP
 * variable and value.  Instances contain a MIB name, object name,
 * key, syntax, and value.  Values can be manipulated as strings and
 * as their internal representation (e.g. integer, boolean, floating
 * point, etc.)
 *
 * @see XmpMessage
 * @see Xmp
 * @author Bobby Krupczak, rdk@krupczak.org
 * @version $Id: XmpVar.java 15 2008-07-17 14:20:37Z rdk $ 
 **/

public class XmpVar {

  /* class variables and methods *********************** */

  /* instance variables **************************** */
  public String mibName;
  public String objName;
  public String key;      // used for table traversals
  public String valueStr;
  public int xmpSyntax;

  /* constructors  ***************************** */
  public XmpVar(String mibName, String objName, int xmpSyntax) {
     this.mibName = new String(mibName);
     this.objName = new String(objName);
     this.key = "";
     this.valueStr = "";
     this.xmpSyntax = xmpSyntax;  
  }
  public XmpVar(String mibName, String objName, String key) {
     this.mibName = new String(mibName);
     this.objName = new String(objName);
     this.key = new String(key);
     this.valueStr = null;
     this.xmpSyntax = Xmp.SYNTAX_INVALID;
  }
  public XmpVar(String mib, String obj, String key, String val, int syntax) {
     this.mibName = new String(mib);
     this.objName = new String(obj);
     this.key = new String(key);
     this.valueStr = new String(val);
     this.xmpSyntax = syntax;
  }
  public XmpVar() {
     this.mibName = "";
     this.objName = "";;
     this.key = "";
     this.valueStr = "";
     this.xmpSyntax = Xmp.SYNTAX_INVALID;
  }

  /* private methods **************************** */

  /* public methods ***************************** */
  public String getMibName() { return mibName; }
  public String getObjName() { return objName; }
  public String getKey() { return key; }

  public int getSyntax() { return xmpSyntax; }

  public int setMibName(String mib) { 
     this.mibName = new String(mib); return 0; 
  }
  public int setObjName(String obj) { 
     this.objName = new String(obj); return 0; 
  }
  public int setKey(String key) { this.key = new String(key); return 0; }
  public int setValue(String valStr) { 
     this.valueStr = new String(valStr); return 0; 
  }

  public int setSyntax(String syntaxStr) {
      this.xmpSyntax = Xmp.stringToSyntax(syntaxStr);
      return this.xmpSyntax;
  }
  public int setSyntax (int syntax) {
      if ((syntax >= Xmp.SYNTAX_INVALID) || 
          (syntax <= Xmp.SYNTAX_UNSUPPORTEDVAR))
         this.xmpSyntax = syntax;
      return this.xmpSyntax;
  }

  public ExtendedBoolean getValueExtendedBoolean()
  { 
      return new ExtendedBoolean(valueStr);
  }

  public int setValueExtendedBoolean(int val)
  {
      ExtendedBoolean b;

      this.xmpSyntax = Xmp.SYNTAX_EXTENDEDBOOLEAN;

      b = new ExtendedBoolean(val);

      this.valueStr = b.toString();

      return this.xmpSyntax;
  }

  public int setValueExtendedBoolean(String valStr)
  {
      ExtendedBoolean b;

      this.xmpSyntax = Xmp.SYNTAX_EXTENDEDBOOLEAN;

      /* should be true, false, unknown */
      b =  new ExtendedBoolean(valStr);

      this.valueStr = b.toString();

      return this.xmpSyntax;
  }

  public String getValue() { return valueStr; }
  public float getValueFloat() { return Float.valueOf(valueStr).floatValue();}
  public int getValueInteger() { return Integer.valueOf(valueStr).intValue(); }
  public long getValueLong() { return Long.valueOf(valueStr).longValue(); }
  public boolean getValueBoolean() { return Boolean.valueOf(valueStr).booleanValue(); }
  public int setValueStr(String valStr) {
     this.xmpSyntax = Xmp.SYNTAX_DISPLAYSTRING;
     this.valueStr = new String(valStr);
     return 0;
  }
  public int setValueCounter(long val) {
     this.xmpSyntax = Xmp.SYNTAX_COUNTER;
     this.valueStr = String.valueOf(val);
     return 0;
  }

  public int setValueIpV4Address(String ipaddrstr) {
     this.xmpSyntax = Xmp.SYNTAX_IPV4ADDRESS;
     this.valueStr = new String(ipaddrstr);
     return 0;
  }
  public int setValueIpV4Address(InetAddress addr) {
     this.xmpSyntax = Xmp.SYNTAX_IPV4ADDRESS;
     this.valueStr = addr.toString();
     return 0;
  }
  public int setValueIpV4Address(Inet4Address addr) {
     this.xmpSyntax = Xmp.SYNTAX_IPV4ADDRESS;
     this.valueStr = addr.toString();
     return 0;
  }

  public int setValueIpV6Address(String ipaddrstr) {
     this.xmpSyntax = Xmp.SYNTAX_IPV6ADDRESS;
     this.valueStr = new String(ipaddrstr);
     return 0;
  }
  public int setValueIpV6Address(InetAddress addr) {
     this.xmpSyntax = Xmp.SYNTAX_IPV6ADDRESS;
     this.valueStr = addr.toString();
     return 0;
  }
  public int setValueIpV6Address(Inet6Address addr) {
     this.xmpSyntax = Xmp.SYNTAX_IPV6ADDRESS;
     this.valueStr = addr.toString();
     return 0;
  }

  public int setValueDateTime(String datetime) {
     this.xmpSyntax = Xmp.SYNTAX_DATETIME;
     this.valueStr = new String(datetime);
     return 0;
  }
  public int setValueGauge(long val) {
     this.xmpSyntax = Xmp.SYNTAX_GAUGE;
     this.valueStr = String.valueOf(val);
     return 0;
  }
  public int setValueInteger(long val) {
     this.xmpSyntax = Xmp.SYNTAX_INTEGER;
     this.valueStr = String.valueOf(val);
     return 0;
  }
  public int setValueUInteger(long val) {
     this.xmpSyntax = Xmp.SYNTAX_UNSIGNEDINTEGER;
     this.valueStr = String.valueOf(val); // no unsigned in Java
     return 0;
  }
  public int setValueBoolean(boolean b) {
     this.xmpSyntax = Xmp.SYNTAX_BOOLEAN;
     this.valueStr = String.valueOf(b); // no unsigned in Java
     return 0;
  }
  public int setValueMacAddress(byte[] m) {
     this.xmpSyntax = Xmp.SYNTAX_MACADDRESS;
     return 0;
  }
  public int setValuePhysAddress(byte[] p) {
     this.xmpSyntax = Xmp.SYNTAX_PHYSADDRESS;
     return 0;
  }
  public int setValueBinaryString(byte[] s) {
     this.xmpSyntax = Xmp.SYNTAX_BINARYSTRING;
     return 0;
  }
  public int setValueNullSyntax() {
     this.xmpSyntax = Xmp.SYNTAX_NULLSYNTAX;
     this.valueStr = null;
     return 0;
  }
  public int setValueTable() {
     this.xmpSyntax = Xmp.SYNTAX_TABLE;
     this.valueStr = null;
     return 0;
  }
  public int setValueFloatingPoint(double d) {
     this.xmpSyntax = Xmp.SYNTAX_FLOATINGPOINT;
     this.valueStr = String.valueOf(d);
     return 0;
  }
  public int setUnsupportedVar() {
     this.xmpSyntax = Xmp.SYNTAX_UNSUPPORTEDVAR;
     this.valueStr = null;
     return 0;
  }

  public void dump(PrintStream out) {

     out.println("XmpVar name: "+mibName+"."+objName+".key='"+key+"'");
     out.println("XmpVar type: "+Xmp.syntaxToString(xmpSyntax)+
                 " Value: '"+valueStr+"'");
  }

  public void dump(JTextArea out) {
     out.append("XmpVar name: "+mibName+"."+objName+".key='"+key+"'\n");
     out.append("XmpVar type: "+Xmp.syntaxToString(xmpSyntax)+
                 " Value: '"+valueStr+"'\n");
  }     

} /* class XmpVar */
