/*******************************************************************************
 * This file is part of OpenNMS(R).
 *
 * Copyright (C) 2006-2011 The OpenNMS Group, Inc.
 * OpenNMS(R) is Copyright (C) 1999-2011 The OpenNMS Group, Inc.
 *
 * OpenNMS(R) is a registered trademark of The OpenNMS Group, Inc.
 *
 * OpenNMS(R) is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation, either version 3 of the License,
 * or (at your option) any later version.
 *
 * OpenNMS(R) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenNMS(R).  If not, see:
 *      http://www.gnu.org/licenses/
 *
 * For more information contact:
 *     OpenNMS(R) Licensing <license@opennms.org>
 *     http://www.opennms.org/
 *     http://www.opennms.com/
 *******************************************************************************/

package org.opennms.web.vulnerability;

import java.io.IOException;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import javax.servlet.RequestDispatcher;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.opennms.web.WebSecurityUtils;
import org.opennms.web.vulnerability.filter.Filter;

/**
 * A servlet that handles querying the vulnerabilities table by using filters to
 * create a vulnerability list and and then forwards that list to a JSP for
 * display.
 *
 * @author <A HREF="mailto:larry@opennms.org">Lawrence Karnowski </A>
 * @author <A HREF="http://www.opennms.org/">OpenNMS </A>
 * @author <A HREF="mailto:larry@opennms.org">Lawrence Karnowski </A>
 * @author <A HREF="http://www.opennms.org/">OpenNMS </A>
 * @version $Id: $
 * @since 1.8.1
 */
public class VulnerabilityFilterServlet extends HttpServlet {
    /**
     * 
     */
    private static final long serialVersionUID = -1885837409842764000L;

    /** Constant <code>DEFAULT_LIMIT=10</code> */
    public static final int DEFAULT_LIMIT = 10;

    /** Constant <code>DEFAULT_MULTIPLE=0</code> */
    public static final int DEFAULT_MULTIPLE = 0;

    /** Constant <code>DEFAULT_SORT_STYLE</code> */
    public static final VulnerabilityFactory.SortStyle DEFAULT_SORT_STYLE = VulnerabilityFactory.SortStyle.ID;

    /** Constant <code>DEFAULT_RESOLUTION_TYPE</code> */
    public static final VulnerabilityFactory.ResolutionType DEFAULT_RESOLUTION_TYPE = VulnerabilityFactory.ResolutionType.BOTH;

    /**
     * {@inheritDoc}
     *
     * Parses the query string to determine what types of filters to use (for
     * example, what to filter on or sort by), then does the database query
     * (through the VulnerabilityFactory) and then forwards the results to a JSP
     * for display.
     *
     * <p>
     * Sets the <em>vuls</em> and <em>parms</em> request attributes for the
     * forwardee JSP (or whatever gets called).
     * </p>
     */
    public void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        // handle the style sort parameter
        String sortStyleString = request.getParameter("sortby");
        VulnerabilityFactory.SortStyle sortStyle = DEFAULT_SORT_STYLE;
        if (sortStyleString != null) {
            Object temp = VulnerabilityUtil.getSortStyle(sortStyleString);
            if (temp != null) {
                sortStyle = (VulnerabilityFactory.SortStyle) temp;
            }
        }

        // handle the acknowledgement type parameter
        String resTypeString = request.getParameter("restype");
        VulnerabilityFactory.ResolutionType resType = DEFAULT_RESOLUTION_TYPE;
        if (resTypeString != null) {
            Object temp = VulnerabilityUtil.getResolutionType(resTypeString);
            if (temp != null) {
                resType = (VulnerabilityFactory.ResolutionType) temp;
            }
        }

        // handle the filter parameters
        String[] filterStrings = request.getParameterValues("filter");
        List<Filter> filterArray = new ArrayList<Filter>();
        if (filterStrings != null) {
            for (int i = 0; i < filterStrings.length; i++) {
                Filter filter = VulnerabilityUtil.getFilter(filterStrings[i], getServletContext());
                if (filter != null) {
                    filterArray.add(filter);
                }
            }
        }

        // handle the optional limit parameter
        String limitString = request.getParameter("limit");
        int limit = DEFAULT_LIMIT;

        if (limitString != null) {
            try {
                int newlimit = WebSecurityUtils.safeParseInt(limitString);
                if (newlimit > 0) {
                    limit = newlimit;
                }
            } catch (NumberFormatException e) {
                // do nothing, the default is aready set
            }
        }

        // handle the optional multiple parameter
        String multipleString = request.getParameter("multiple");
        int multiple = DEFAULT_MULTIPLE;
        if (multipleString != null) {
            try {
                multiple = Math.max(0, WebSecurityUtils.safeParseInt(multipleString));
            } catch (NumberFormatException e) {
            }
        }

        try {
            // put the parameters in a convenient struct
            VulnerabilityQueryParms parms = new VulnerabilityQueryParms();
            parms.sortStyle = sortStyle;
            parms.resType = resType;
            parms.filters = filterArray;
            parms.limit = limit;
            parms.multiple = multiple;

            // query the vulnerabilities with the new filters array
            Vulnerability[] vuls = VulnerabilityFactory.getVulnerabilities(sortStyle, resType, parms.getFilters(), limit, multiple * limit);

            // add the necessary data to the request so the
            // JSP (or whatever gets called) can create the view correctly
            request.setAttribute("vuls", vuls);
            request.setAttribute("parms", parms);

            // forward the request for proper display
            RequestDispatcher dispatcher = this.getServletContext().getRequestDispatcher("/vulnerability/list.jsp");
            dispatcher.forward(request, response);
        } catch (SQLException e) {
            throw new ServletException("", e);
        }
    }

}
