//
// This file is part of the OpenNMS(R) Application.
//
// OpenNMS(R) is Copyright (C) 2002-2009 The OpenNMS Group, Inc.  All rights reserved.
// OpenNMS(R) is a derivative work, containing both original code, included code and modified
// code that was published under the GNU General Public License. Copyrights for modified 
// and included code are below.
//
// OpenNMS(R) is a registered trademark of The OpenNMS Group, Inc.
//
// Modifications:
//
// 2009 Apr: refactoring to support ACL DAO work
// 2007 Jul 24: Java 5 generics. - dj@opennms.org
//
// Copyright (C) 1999-2001 Oculan Corp.  All rights reserved.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// For more information contact:
//      OpenNMS Licensing       <license@opennms.org>
//      http://www.opennms.org/
//      http://www.opennms.com/
//

package org.opennms.web.vulnerability;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;

import javax.servlet.http.HttpServletRequest;

import org.opennms.netmgt.model.OnmsSeverity;
import org.opennms.web.Util;
import org.opennms.web.WebSecurityUtils;
import org.opennms.web.vulnerability.filter.CreationDateAfterFilter;
import org.opennms.web.vulnerability.filter.CreationDateBeforeFilter;
import org.opennms.web.vulnerability.filter.Filter;
import org.opennms.web.vulnerability.filter.IPLikeFilter;
import org.opennms.web.vulnerability.filter.InterfaceFilter;
import org.opennms.web.vulnerability.filter.NegativeInterfaceFilter;
import org.opennms.web.vulnerability.filter.NegativeNodeFilter;
import org.opennms.web.vulnerability.filter.NegativePortFilter;
import org.opennms.web.vulnerability.filter.NegativeProtocolFilter;
import org.opennms.web.vulnerability.filter.NegativeSeverityFilter;
import org.opennms.web.vulnerability.filter.NodeFilter;
import org.opennms.web.vulnerability.filter.NodeNameLikeFilter;
import org.opennms.web.vulnerability.filter.PortFilter;
import org.opennms.web.vulnerability.filter.ProtocolFilter;
import org.opennms.web.vulnerability.filter.ResolvedDateAfterFilter;
import org.opennms.web.vulnerability.filter.ResolvedDateBeforeFilter;
import org.opennms.web.vulnerability.filter.SeverityFilter;

/**
 * Convenient Java methods and constants for use in the vulnerability user
 * interface.
 * 
 * <p>
 * <code>VulnerabilityUtil</code> is a place to put user interface-specific
 * code that is common among many JSPs or servlets.
 * </p>
 */
public abstract class VulnerabilityUtil extends Object {
    protected static final Map<OnmsSeverity, String> labels;

    protected static final Map<OnmsSeverity, String> icons;

    protected static final Map<String, VulnerabilityFactory.SortStyle> sortStyles;
    
    protected static final Map<VulnerabilityFactory.SortStyle, String> sortStylesString;

    protected static final Map<String, VulnerabilityFactory.ResolutionType> resTypes;
    
    protected static final Map<VulnerabilityFactory.ResolutionType, String> resTypesString;

    protected static final List<OnmsSeverity> severities;

    public static final String FILTER_SERVLET_URL_BASE = "vulnerability/list";

    static {
        // this lowest to highest order is mandatory, lots of code depends on it
        severities = new ArrayList<OnmsSeverity>();
        severities.add(OnmsSeverity.INDETERMINATE);
        severities.add(OnmsSeverity.CLEARED);
        severities.add(OnmsSeverity.NORMAL);
        severities.add(OnmsSeverity.WARNING);
        severities.add(OnmsSeverity.MINOR);
        severities.add(OnmsSeverity.MAJOR);
        severities.add(OnmsSeverity.CRITICAL);

        labels = new HashMap<OnmsSeverity, String>();
        labels.put(OnmsSeverity.INDETERMINATE, "Indeterminate");
        labels.put(OnmsSeverity.CLEARED, "Cleared");
        labels.put(OnmsSeverity.NORMAL, "Normal");
        labels.put(OnmsSeverity.WARNING, "Warning");
        labels.put(OnmsSeverity.MINOR, "Minor");
        labels.put(OnmsSeverity.MAJOR, "Major");
        labels.put(OnmsSeverity.CRITICAL, "Critical");

        // note several of these are null, for those use no icon
        icons = new HashMap<OnmsSeverity, String>();
        icons.put(OnmsSeverity.INDETERMINATE, "images/alert_indeterminate_s.gif");
        icons.put(OnmsSeverity.CLEARED, "images/alert_cleared_s.gif");
        icons.put(OnmsSeverity.NORMAL, "images/alert_normal_s.gif");
        icons.put(OnmsSeverity.WARNING, "images/alert_warning_s.gif");
        icons.put(OnmsSeverity.MINOR, "images/alert_minor_s.gif");
        icons.put(OnmsSeverity.MAJOR, "images/alert_major_s.gif");
        icons.put(OnmsSeverity.CRITICAL, "images/alert_critical_s.gif");

        sortStyles = new HashMap<String, VulnerabilityFactory.SortStyle>();
        sortStyles.put("severity", VulnerabilityFactory.SortStyle.SEVERITY);
        sortStyles.put("node", VulnerabilityFactory.SortStyle.NODE);
        sortStyles.put("interface", VulnerabilityFactory.SortStyle.INTERFACE);
        sortStyles.put("service", VulnerabilityFactory.SortStyle.SERVICE);
        sortStyles.put("id", VulnerabilityFactory.SortStyle.ID);
        sortStyles.put("ctime", VulnerabilityFactory.SortStyle.CREATE_TIME);
        sortStyles.put("rtime", VulnerabilityFactory.SortStyle.RESOLVED_TIME);
        sortStyles.put("port", VulnerabilityFactory.SortStyle.PORT);
        sortStyles.put("protocol", VulnerabilityFactory.SortStyle.PROTOCOL);
        sortStyles.put("rev_severity", VulnerabilityFactory.SortStyle.REVERSE_SEVERITY);
        sortStyles.put("rev_node", VulnerabilityFactory.SortStyle.REVERSE_NODE);
        sortStyles.put("rev_interface", VulnerabilityFactory.SortStyle.REVERSE_INTERFACE);
        sortStyles.put("rev_service", VulnerabilityFactory.SortStyle.REVERSE_SERVICE);
        sortStyles.put("rev_id", VulnerabilityFactory.SortStyle.REVERSE_ID);
        sortStyles.put("rev_ctime", VulnerabilityFactory.SortStyle.REVERSE_CREATE_TIME);
        sortStyles.put("rev_rtime", VulnerabilityFactory.SortStyle.REVERSE_RESOLVED_TIME);
        sortStyles.put("rev_port", VulnerabilityFactory.SortStyle.REVERSE_PORT);
        sortStyles.put("rev_protocol", VulnerabilityFactory.SortStyle.REVERSE_PROTOCOL);

        sortStylesString = new HashMap<VulnerabilityFactory.SortStyle, String>();
        sortStylesString.put(VulnerabilityFactory.SortStyle.SEVERITY, "severity");
        sortStylesString.put(VulnerabilityFactory.SortStyle.NODE, "node");
        sortStylesString.put(VulnerabilityFactory.SortStyle.INTERFACE, "interface");
        sortStylesString.put(VulnerabilityFactory.SortStyle.SERVICE, "service");
        sortStylesString.put(VulnerabilityFactory.SortStyle.ID, "id");
        sortStylesString.put(VulnerabilityFactory.SortStyle.CREATE_TIME, "ctime");
        sortStylesString.put(VulnerabilityFactory.SortStyle.RESOLVED_TIME, "rtime");
        sortStylesString.put(VulnerabilityFactory.SortStyle.PORT, "port");
        sortStylesString.put(VulnerabilityFactory.SortStyle.PROTOCOL, "protocol");
        sortStylesString.put(VulnerabilityFactory.SortStyle.REVERSE_SEVERITY, "rev_severity");
        sortStylesString.put(VulnerabilityFactory.SortStyle.REVERSE_NODE, "rev_node");
        sortStylesString.put(VulnerabilityFactory.SortStyle.REVERSE_INTERFACE, "rev_interface");
        sortStylesString.put(VulnerabilityFactory.SortStyle.REVERSE_SERVICE, "rev_service");
        sortStylesString.put(VulnerabilityFactory.SortStyle.REVERSE_ID, "rev_id");
        sortStylesString.put(VulnerabilityFactory.SortStyle.REVERSE_CREATE_TIME, "rev_ctime");
        sortStylesString.put(VulnerabilityFactory.SortStyle.REVERSE_RESOLVED_TIME, "rev_rtime");
        sortStylesString.put(VulnerabilityFactory.SortStyle.REVERSE_PORT, "rev_port");
        sortStylesString.put(VulnerabilityFactory.SortStyle.REVERSE_PROTOCOL, "rev_protocol");

        resTypes = new HashMap<String, VulnerabilityFactory.ResolutionType>();
        resTypes.put("res", VulnerabilityFactory.ResolutionType.RESOLVED);
        resTypes.put("open", VulnerabilityFactory.ResolutionType.OPEN);
        resTypes.put("both", VulnerabilityFactory.ResolutionType.BOTH);

        resTypesString = new HashMap<VulnerabilityFactory.ResolutionType, String>();
        resTypesString.put(VulnerabilityFactory.ResolutionType.RESOLVED, "res");
        resTypesString.put(VulnerabilityFactory.ResolutionType.OPEN, "open");
        resTypesString.put(VulnerabilityFactory.ResolutionType.BOTH, "both");
    }

    /**
     * Can return null, in that case, use no icon.
     */
    public static String getSeverityIcon(int severity) {
        return getSeverityIcon(new Integer(severity));
    }

    /**
     * Can return null, in that case, use no icon.
     */
    public static String getSeverityIcon(Integer severity) {
        if (severity == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        return icons.get(severity);
    }

    public static VulnerabilityFactory.SortStyle getSortStyle(String sortStyleString) {
        if (sortStyleString == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        return sortStyles.get(sortStyleString.toLowerCase());
    }

    public static String getSortStyleString(VulnerabilityFactory.SortStyle sortStyle) {
        if (sortStyle == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        return sortStylesString.get(sortStyle);
    }

    public static VulnerabilityFactory.ResolutionType getResolutionType(String resTypeString) {
        if (resTypeString == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        return resTypes.get(resTypeString.toLowerCase());
    }

    public static String getResolutionTypeString(VulnerabilityFactory.ResolutionType resType) {
        if (resType == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        return resTypesString.get(resType);
    }

    public static Filter getFilter(String filterString) {
        if (filterString == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        Filter filter = null;

        StringTokenizer tokens = new StringTokenizer(filterString, "=");
        String type = tokens.nextToken();
        String value = tokens.nextToken();

        if (type.equals(SeverityFilter.TYPE)) {
            filter = new SeverityFilter(WebSecurityUtils.safeParseInt(value));
        } else if (type.equals(NodeFilter.TYPE)) {
            filter = new NodeFilter(WebSecurityUtils.safeParseInt(value));
        } else if (type.equals(NodeNameLikeFilter.TYPE)) {
            filter = new NodeNameLikeFilter(value);
        } else if (type.equals(InterfaceFilter.TYPE)) {
            filter = new InterfaceFilter(value);
        } else if (type.equals(NegativeSeverityFilter.TYPE)) {
            filter = new NegativeSeverityFilter(WebSecurityUtils.safeParseInt(value));
        } else if (type.equals(NegativeNodeFilter.TYPE)) {
            filter = new NegativeNodeFilter(WebSecurityUtils.safeParseInt(value));
        } else if (type.equals(NegativeInterfaceFilter.TYPE)) {
            filter = new NegativeInterfaceFilter(value);
        } else if (type.equals(IPLikeFilter.TYPE)) {
            filter = new IPLikeFilter(value);
        } else if (type.equals(ProtocolFilter.TYPE)) {
            filter = new ProtocolFilter(value);
        } else if (type.equals(NegativeProtocolFilter.TYPE)) {
            filter = new NegativeProtocolFilter(value);
        } else if (type.equals(PortFilter.TYPE)) {
            filter = new PortFilter(WebSecurityUtils.safeParseInt(value));
        } else if (type.equals(NegativePortFilter.TYPE)) {
            filter = new NegativePortFilter(WebSecurityUtils.safeParseInt(value));
        } else if (type.equals(CreationDateBeforeFilter.TYPE)) {
            filter = new CreationDateBeforeFilter(WebSecurityUtils.safeParseLong(value));
        } else if (type.equals(CreationDateAfterFilter.TYPE)) {
            filter = new CreationDateAfterFilter(WebSecurityUtils.safeParseLong(value));
        } else if (type.equals(ResolvedDateBeforeFilter.TYPE)) {
            filter = new ResolvedDateBeforeFilter(WebSecurityUtils.safeParseLong(value));
        } else if (type.equals(ResolvedDateAfterFilter.TYPE)) {
            filter = new ResolvedDateAfterFilter(WebSecurityUtils.safeParseLong(value));
        }

        /*
         * else if( type.equals( ServiceFilter.TYPE )) { filter = new
         * ServiceFilter( WebSecurityUtils.safeParseInt( value )); } else if( type.equals(
         * NegativeServiceFilter.TYPE )) { filter = new NegativeServiceFilter(
         * WebSecurityUtils.safeParseInt( value )); } else if( type.equals(
         * LogMessageSubstringFilter.TYPE )) { filter = new
         * LogMessageSubstringFilter( value ); } else if( type.equals(
         * LogMessageMatchesAnyFilter.TYPE )) { filter = new
         * LogMessageMatchesAnyFilter(value); }
         */

        return filter;
    }

    public static String getFilterString(Filter filter) {
        if (filter == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        return filter.getDescription();
    }

    protected static final String[] LINK_IGNORES = new String[] { "sortby", "restype", "limit", "multiple", "filter" };

    public static String makeLink(HttpServletRequest request, VulnerabilityFactory.SortStyle sortStyle, VulnerabilityFactory.ResolutionType resType, List<Filter> filters, int limit) {
        if (request == null || sortStyle == null || resType == null || filters == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        if (limit < 1) {
            throw new IllegalArgumentException("Cannot take a zero or negative limit value.");
        }

        Map<String, Object> additions = new HashMap<String, Object>();
        additions.put("sortby", VulnerabilityUtil.getSortStyleString(sortStyle));
        additions.put("restype", VulnerabilityUtil.getResolutionTypeString(resType));
        additions.put("limit", Integer.toString(limit));

        if (filters != null) {
            String[] filterStrings = new String[filters.size()];

            for (int i = 0; i < filters.size(); i++) {
                filterStrings[i] = VulnerabilityUtil.getFilterString(filters.get(i));
            }

            additions.put("filter", filterStrings);
        }

        return FILTER_SERVLET_URL_BASE + "?" + Util.makeQueryString(request, additions, LINK_IGNORES, Util.IgnoreType.REQUEST_ONLY);
    }

    public static String makeLink(HttpServletRequest request, VulnerabilityQueryParms parms) {
        if (request == null || parms == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        return makeLink(request, parms.sortStyle, parms.resType, parms.filters, parms.limit);
    }

    public static String makeLink(HttpServletRequest request, VulnerabilityQueryParms parms, VulnerabilityFactory.SortStyle sortStyle) {
        if (request == null || parms == null || sortStyle == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        return makeLink(request, sortStyle, parms.resType, parms.filters, parms.limit);
    }

    public static String makeLink(HttpServletRequest request, VulnerabilityQueryParms parms, VulnerabilityFactory.ResolutionType resType) {
        if (request == null || parms == null || resType == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        return makeLink(request, parms.sortStyle, resType, parms.filters, parms.limit);
    }

    public static String makeLink(HttpServletRequest request, VulnerabilityQueryParms parms, List<Filter> filters) {
        if (request == null || parms == null || filters == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        return makeLink(request, parms.sortStyle, parms.resType, filters, parms.limit);
    }

    public static String makeLink(HttpServletRequest request, VulnerabilityQueryParms parms, Filter filter, boolean add) {
        if (request == null || parms == null || filter == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        List<Filter> newList = new ArrayList<Filter>(parms.filters);

        if (add) {
            newList.add(filter);
        } else {
            newList.remove(filter);
        }

        return makeLink(request, parms.sortStyle, parms.resType, newList, parms.limit);
    }

    public static String makeHiddenTags(HttpServletRequest request, VulnerabilityFactory.SortStyle sortStyle, VulnerabilityFactory.ResolutionType resType, List<Filter> filters, int limit) {
        if (request == null || sortStyle == null || resType == null || filters == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        if (limit < 1) {
            throw new IllegalArgumentException("Cannot take a zero or negative limit value.");
        }

        Map<String, Object> additions = new HashMap<String, Object>();
        additions.put("sortby", VulnerabilityUtil.getSortStyleString(sortStyle));
        additions.put("restype", VulnerabilityUtil.getResolutionTypeString(resType));
        additions.put("limit", Integer.toString(limit));

        if (filters != null) {
            String[] filterStrings = new String[filters.size()];

            for (int i = 0; i < filters.size(); i++) {
                filterStrings[i] = VulnerabilityUtil.getFilterString(filters.get(i));
            }

            additions.put("filter", filterStrings);
        }

        return Util.makeHiddenTags(request, additions, LINK_IGNORES, Util.IgnoreType.REQUEST_ONLY);
    }

    public static String makeHiddenTags(HttpServletRequest request, VulnerabilityQueryParms parms) {
        if (request == null || parms == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        return makeHiddenTags(request, parms.sortStyle, parms.resType, parms.filters, parms.limit);
    }

    public static String makeHiddenTags(HttpServletRequest request, VulnerabilityQueryParms parms, VulnerabilityFactory.SortStyle sortStyle) {
        if (request == null || parms == null || sortStyle == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        return makeHiddenTags(request, sortStyle, parms.resType, parms.filters, parms.limit);
    }

    public static String makeHiddenTags(HttpServletRequest request, VulnerabilityQueryParms parms, VulnerabilityFactory.ResolutionType resType) {
        if (request == null || parms == null || resType == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        return makeHiddenTags(request, parms.sortStyle, resType, parms.filters, parms.limit);
    }

    public static String makeHiddenTags(HttpServletRequest request, VulnerabilityQueryParms parms, List<Filter> filters) {
        if (request == null || parms == null || filters == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        return makeHiddenTags(request, parms.sortStyle, parms.resType, filters, parms.limit);
    }

    public static String makeHiddenTags(HttpServletRequest request, VulnerabilityQueryParms parms, Filter filter, boolean add) {
        if (request == null || parms == null || filter == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        List<Filter> newList = new ArrayList<Filter>(parms.filters);

        if (add) {
            newList.add(filter);
        } else {
            newList.remove(filter);
        }

        return makeHiddenTags(request, parms.sortStyle, parms.resType, newList, parms.limit);
    }

}
