//
// This file is part of the OpenNMS(R) Application.
//
// OpenNMS(R) is Copyright (C) 2002-2003 The OpenNMS Group, Inc.  All rights reserved.
// OpenNMS(R) is a derivative work, containing both original code, included code and modified
// code that was published under the GNU General Public License. Copyrights for modified 
// and included code are below.
//
// OpenNMS(R) is a registered trademark of The OpenNMS Group, Inc.
//
// Copyright (C) 1999-2001 Oculan Corp.  All rights reserved.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// For more information contact:
//      OpenNMS Licensing       <license@opennms.org>
//      http://www.opennms.org/
//      http://www.opennms.com/
//

package org.opennms.web.vulnerability;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.StringTokenizer;

import javax.servlet.http.HttpServletRequest;

import org.opennms.web.Util;
import org.opennms.web.event.Event;
import org.opennms.web.vulnerability.filter.CreationDateAfterFilter;
import org.opennms.web.vulnerability.filter.CreationDateBeforeFilter;
import org.opennms.web.vulnerability.filter.Filter;
import org.opennms.web.vulnerability.filter.IPLikeFilter;
import org.opennms.web.vulnerability.filter.InterfaceFilter;
import org.opennms.web.vulnerability.filter.NegativeInterfaceFilter;
import org.opennms.web.vulnerability.filter.NegativeNodeFilter;
import org.opennms.web.vulnerability.filter.NegativePortFilter;
import org.opennms.web.vulnerability.filter.NegativeProtocolFilter;
import org.opennms.web.vulnerability.filter.NegativeSeverityFilter;
import org.opennms.web.vulnerability.filter.NodeFilter;
import org.opennms.web.vulnerability.filter.NodeNameLikeFilter;
import org.opennms.web.vulnerability.filter.PortFilter;
import org.opennms.web.vulnerability.filter.ProtocolFilter;
import org.opennms.web.vulnerability.filter.ResolvedDateAfterFilter;
import org.opennms.web.vulnerability.filter.ResolvedDateBeforeFilter;
import org.opennms.web.vulnerability.filter.SeverityFilter;

/**
 * Convenient Java methods and constants for use in the vulnerability user
 * interface.
 * 
 * <p>
 * <code>VulnerabilityUtil</code> is a place to put user interface-specific
 * code that is common among many JSPs or servlets.
 * </p>
 */
public abstract class VulnerabilityUtil extends Object {
    protected static final HashMap colors;

    protected static final HashMap labels;

    protected static final HashMap icons;

    protected static final HashMap sortStyles;

    protected static final HashMap resTypes;

    protected static final List severities;

    public static final String FILTER_SERVLET_URL_BASE = "vulnerability/list";

    static {
        // this lowest to highest order is mandatory, lots of code depends on it
        severities = new java.util.ArrayList();
        severities.add(new Integer(Event.INDETERMINATE_SEVERITY));
        severities.add(new Integer(Event.CLEARED_SEVERITY));
        severities.add(new Integer(Event.NORMAL_SEVERITY));
        severities.add(new Integer(Event.WARNING_SEVERITY));
        severities.add(new Integer(Event.MINOR_SEVERITY));
        severities.add(new Integer(Event.MAJOR_SEVERITY));
        severities.add(new Integer(Event.CRITICAL_SEVERITY));

        colors = new java.util.HashMap();
        colors.put(new Integer(Event.INDETERMINATE_SEVERITY), "lightblue");
        colors.put(new Integer(Event.CLEARED_SEVERITY), "white");
        colors.put(new Integer(Event.NORMAL_SEVERITY), "green");
        colors.put(new Integer(Event.WARNING_SEVERITY), "cyan");
        colors.put(new Integer(Event.MINOR_SEVERITY), "yellow");
        colors.put(new Integer(Event.MAJOR_SEVERITY), "orange");
        colors.put(new Integer(Event.CRITICAL_SEVERITY), "red");

        labels = new java.util.HashMap();
        labels.put(new Integer(Event.INDETERMINATE_SEVERITY), "Indeterminate");
        labels.put(new Integer(Event.CLEARED_SEVERITY), "Cleared");
        labels.put(new Integer(Event.NORMAL_SEVERITY), "Normal");
        labels.put(new Integer(Event.WARNING_SEVERITY), "Warning");
        labels.put(new Integer(Event.MINOR_SEVERITY), "Minor");
        labels.put(new Integer(Event.MAJOR_SEVERITY), "Major");
        labels.put(new Integer(Event.CRITICAL_SEVERITY), "Critical");

        // note several of these are null, for those use no icon
        icons = new java.util.HashMap();
        icons.put(new Integer(Event.INDETERMINATE_SEVERITY), "images/alert_indeterminate_s.gif");
        icons.put(new Integer(Event.CLEARED_SEVERITY), "images/alert_cleared_s.gif");
        icons.put(new Integer(Event.NORMAL_SEVERITY), "images/alert_normal_s.gif");
        icons.put(new Integer(Event.WARNING_SEVERITY), "images/alert_warning_s.gif");
        icons.put(new Integer(Event.MINOR_SEVERITY), "images/alert_minor_s.gif");
        icons.put(new Integer(Event.MAJOR_SEVERITY), "images/alert_major_s.gif");
        icons.put(new Integer(Event.CRITICAL_SEVERITY), "images/alert_critical_s.gif");

        sortStyles = new java.util.HashMap();
        sortStyles.put("severity", VulnerabilityFactory.SortStyle.SEVERITY);
        sortStyles.put("node", VulnerabilityFactory.SortStyle.NODE);
        sortStyles.put("interface", VulnerabilityFactory.SortStyle.INTERFACE);
        sortStyles.put("service", VulnerabilityFactory.SortStyle.SERVICE);
        sortStyles.put("id", VulnerabilityFactory.SortStyle.ID);
        sortStyles.put("ctime", VulnerabilityFactory.SortStyle.CREATE_TIME);
        sortStyles.put("rtime", VulnerabilityFactory.SortStyle.RESOLVED_TIME);
        sortStyles.put("port", VulnerabilityFactory.SortStyle.PORT);
        sortStyles.put("protocol", VulnerabilityFactory.SortStyle.PROTOCOL);
        sortStyles.put("rev_severity", VulnerabilityFactory.SortStyle.REVERSE_SEVERITY);
        sortStyles.put("rev_node", VulnerabilityFactory.SortStyle.REVERSE_NODE);
        sortStyles.put("rev_interface", VulnerabilityFactory.SortStyle.REVERSE_INTERFACE);
        sortStyles.put("rev_service", VulnerabilityFactory.SortStyle.REVERSE_SERVICE);
        sortStyles.put("rev_id", VulnerabilityFactory.SortStyle.REVERSE_ID);
        sortStyles.put("rev_ctime", VulnerabilityFactory.SortStyle.REVERSE_CREATE_TIME);
        sortStyles.put("rev_rtime", VulnerabilityFactory.SortStyle.REVERSE_RESOLVED_TIME);
        sortStyles.put("rev_port", VulnerabilityFactory.SortStyle.REVERSE_PORT);
        sortStyles.put("rev_protocol", VulnerabilityFactory.SortStyle.REVERSE_PROTOCOL);

        sortStyles.put(VulnerabilityFactory.SortStyle.SEVERITY, "severity");
        sortStyles.put(VulnerabilityFactory.SortStyle.NODE, "node");
        sortStyles.put(VulnerabilityFactory.SortStyle.INTERFACE, "interface");
        sortStyles.put(VulnerabilityFactory.SortStyle.SERVICE, "service");
        sortStyles.put(VulnerabilityFactory.SortStyle.ID, "id");
        sortStyles.put(VulnerabilityFactory.SortStyle.CREATE_TIME, "ctime");
        sortStyles.put(VulnerabilityFactory.SortStyle.RESOLVED_TIME, "rtime");
        sortStyles.put(VulnerabilityFactory.SortStyle.PORT, "port");
        sortStyles.put(VulnerabilityFactory.SortStyle.PROTOCOL, "protocol");
        sortStyles.put(VulnerabilityFactory.SortStyle.REVERSE_SEVERITY, "rev_severity");
        sortStyles.put(VulnerabilityFactory.SortStyle.REVERSE_NODE, "rev_node");
        sortStyles.put(VulnerabilityFactory.SortStyle.REVERSE_INTERFACE, "rev_interface");
        sortStyles.put(VulnerabilityFactory.SortStyle.REVERSE_SERVICE, "rev_service");
        sortStyles.put(VulnerabilityFactory.SortStyle.REVERSE_ID, "rev_id");
        sortStyles.put(VulnerabilityFactory.SortStyle.REVERSE_CREATE_TIME, "rev_ctime");
        sortStyles.put(VulnerabilityFactory.SortStyle.REVERSE_RESOLVED_TIME, "rev_rtime");
        sortStyles.put(VulnerabilityFactory.SortStyle.REVERSE_PORT, "rev_port");
        sortStyles.put(VulnerabilityFactory.SortStyle.REVERSE_PROTOCOL, "rev_protocol");

        resTypes = new java.util.HashMap();
        resTypes.put("res", VulnerabilityFactory.ResolutionType.RESOLVED);
        resTypes.put("open", VulnerabilityFactory.ResolutionType.OPEN);
        resTypes.put("both", VulnerabilityFactory.ResolutionType.BOTH);
        resTypes.put(VulnerabilityFactory.ResolutionType.RESOLVED, "res");
        resTypes.put(VulnerabilityFactory.ResolutionType.OPEN, "open");
        resTypes.put(VulnerabilityFactory.ResolutionType.BOTH, "both");
    }

    public static List getSeverityList() {
        return severities;
    }

    public static int getSeverityId(int index) {
        return ((Integer) severities.get(index)).intValue();
    }

    public static String getSeverityColor(int severity) {
        return ((String) colors.get(new Integer(severity)));
    }

    public static String getSeverityLabel(int severity) {
        return getSeverityLabel(new Integer(severity));
    }

    public static String getSeverityLabel(Integer severity) {
        if (severity == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        return (String) labels.get(severity);
    }

    /**
     * Can return null, in that case, use no icon.
     */
    public static String getSeverityIcon(int severity) {
        return getSeverityIcon(new Integer(severity));
    }

    /**
     * Can return null, in that case, use no icon.
     */
    public static String getSeverityIcon(Integer severity) {
        if (severity == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        return (String) icons.get(severity);
    }

    public static VulnerabilityFactory.SortStyle getSortStyle(String sortStyleString) {
        if (sortStyleString == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        return (VulnerabilityFactory.SortStyle) sortStyles.get(sortStyleString.toLowerCase());
    }

    public static String getSortStyleString(VulnerabilityFactory.SortStyle sortStyle) {
        if (sortStyle == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        return (String) sortStyles.get(sortStyle);
    }

    public static VulnerabilityFactory.ResolutionType getResolutionType(String resTypeString) {
        if (resTypeString == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        return (VulnerabilityFactory.ResolutionType) resTypes.get(resTypeString.toLowerCase());
    }

    public static String getResolutionTypeString(VulnerabilityFactory.ResolutionType resType) {
        if (resType == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        return (String) resTypes.get(resType);
    }

    public static Filter getFilter(String filterString) {
        if (filterString == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        Filter filter = null;

        StringTokenizer tokens = new StringTokenizer(filterString, "=");
        String type = tokens.nextToken();
        String value = tokens.nextToken();

        if (type.equals(SeverityFilter.TYPE)) {
            filter = new SeverityFilter(Integer.parseInt(value));
        } else if (type.equals(NodeFilter.TYPE)) {
            filter = new NodeFilter(Integer.parseInt(value));
        } else if (type.equals(NodeNameLikeFilter.TYPE)) {
            filter = new NodeNameLikeFilter(value);
        } else if (type.equals(InterfaceFilter.TYPE)) {
            filter = new InterfaceFilter(value);
        } else if (type.equals(NegativeSeverityFilter.TYPE)) {
            filter = new NegativeSeverityFilter(Integer.parseInt(value));
        } else if (type.equals(NegativeNodeFilter.TYPE)) {
            filter = new NegativeNodeFilter(Integer.parseInt(value));
        } else if (type.equals(NegativeInterfaceFilter.TYPE)) {
            filter = new NegativeInterfaceFilter(value);
        } else if (type.equals(IPLikeFilter.TYPE)) {
            filter = new IPLikeFilter(value);
        } else if (type.equals(ProtocolFilter.TYPE)) {
            filter = new ProtocolFilter(value);
        } else if (type.equals(NegativeProtocolFilter.TYPE)) {
            filter = new NegativeProtocolFilter(value);
        } else if (type.equals(PortFilter.TYPE)) {
            filter = new PortFilter(Integer.parseInt(value));
        } else if (type.equals(NegativePortFilter.TYPE)) {
            filter = new NegativePortFilter(Integer.parseInt(value));
        } else if (type.equals(CreationDateBeforeFilter.TYPE)) {
            filter = new CreationDateBeforeFilter(Long.parseLong(value));
        } else if (type.equals(CreationDateAfterFilter.TYPE)) {
            filter = new CreationDateAfterFilter(Long.parseLong(value));
        } else if (type.equals(ResolvedDateBeforeFilter.TYPE)) {
            filter = new ResolvedDateBeforeFilter(Long.parseLong(value));
        } else if (type.equals(ResolvedDateAfterFilter.TYPE)) {
            filter = new ResolvedDateAfterFilter(Long.parseLong(value));
        }

        /*
         * else if( type.equals( ServiceFilter.TYPE )) { filter = new
         * ServiceFilter( Integer.parseInt( value )); } else if( type.equals(
         * NegativeServiceFilter.TYPE )) { filter = new NegativeServiceFilter(
         * Integer.parseInt( value )); } else if( type.equals(
         * LogMessageSubstringFilter.TYPE )) { filter = new
         * LogMessageSubstringFilter( value ); } else if( type.equals(
         * LogMessageMatchesAnyFilter.TYPE )) { filter = new
         * LogMessageMatchesAnyFilter(value); }
         */

        return (filter);
    }

    public static String getFilterString(Filter filter) {
        if (filter == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        return (filter.getDescription());
    }

    protected static final String[] LINK_IGNORES = new String[] { "sortby", "restype", "limit", "multiple", "filter" };

    public static String makeLink(HttpServletRequest request, VulnerabilityFactory.SortStyle sortStyle, VulnerabilityFactory.ResolutionType resType, ArrayList filters, int limit) {
        if (request == null || sortStyle == null || resType == null || filters == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        if (limit < 1) {
            throw new IllegalArgumentException("Cannot take a zero or negative limit value.");
        }

        HashMap additions = new HashMap();
        additions.put("sortby", VulnerabilityUtil.getSortStyleString(sortStyle));
        additions.put("restype", VulnerabilityUtil.getResolutionTypeString(resType));
        additions.put("limit", Integer.toString(limit));

        if (filters != null) {
            String[] filterStrings = new String[filters.size()];

            for (int i = 0; i < filters.size(); i++) {
                filterStrings[i] = VulnerabilityUtil.getFilterString((Filter) filters.get(i));
            }

            additions.put("filter", filterStrings);
        }

        return FILTER_SERVLET_URL_BASE + "?" + Util.makeQueryString(request, additions, LINK_IGNORES, Util.IgnoreType.REQUEST_ONLY);
    }

    public static String makeLink(HttpServletRequest request, VulnerabilityQueryParms parms) {
        if (request == null || parms == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        return makeLink(request, parms.sortStyle, parms.resType, parms.filters, parms.limit);
    }

    public static String makeLink(HttpServletRequest request, VulnerabilityQueryParms parms, VulnerabilityFactory.SortStyle sortStyle) {
        if (request == null || parms == null || sortStyle == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        return makeLink(request, sortStyle, parms.resType, parms.filters, parms.limit);
    }

    public static String makeLink(HttpServletRequest request, VulnerabilityQueryParms parms, VulnerabilityFactory.ResolutionType resType) {
        if (request == null || parms == null || resType == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        return makeLink(request, parms.sortStyle, resType, parms.filters, parms.limit);
    }

    public static String makeLink(HttpServletRequest request, VulnerabilityQueryParms parms, ArrayList filters) {
        if (request == null || parms == null || filters == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        return makeLink(request, parms.sortStyle, parms.resType, filters, parms.limit);
    }

    public static String makeLink(HttpServletRequest request, VulnerabilityQueryParms parms, Filter filter, boolean add) {
        if (request == null || parms == null || filter == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        ArrayList newList = new ArrayList(parms.filters);

        if (add) {
            newList.add(filter);
        } else {
            newList.remove(filter);
        }

        return makeLink(request, parms.sortStyle, parms.resType, newList, parms.limit);
    }

    public static String makeHiddenTags(HttpServletRequest request, VulnerabilityFactory.SortStyle sortStyle, VulnerabilityFactory.ResolutionType resType, ArrayList filters, int limit) {
        if (request == null || sortStyle == null || resType == null || filters == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        if (limit < 1) {
            throw new IllegalArgumentException("Cannot take a zero or negative limit value.");
        }

        HashMap additions = new HashMap();
        additions.put("sortby", VulnerabilityUtil.getSortStyleString(sortStyle));
        additions.put("restype", VulnerabilityUtil.getResolutionTypeString(resType));
        additions.put("limit", Integer.toString(limit));

        if (filters != null) {
            String[] filterStrings = new String[filters.size()];

            for (int i = 0; i < filters.size(); i++) {
                filterStrings[i] = VulnerabilityUtil.getFilterString((Filter) filters.get(i));
            }

            additions.put("filter", filterStrings);
        }

        return Util.makeHiddenTags(request, additions, LINK_IGNORES, Util.IgnoreType.REQUEST_ONLY);
    }

    public static String makeHiddenTags(HttpServletRequest request, VulnerabilityQueryParms parms) {
        if (request == null || parms == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        return makeHiddenTags(request, parms.sortStyle, parms.resType, parms.filters, parms.limit);
    }

    public static String makeHiddenTags(HttpServletRequest request, VulnerabilityQueryParms parms, VulnerabilityFactory.SortStyle sortStyle) {
        if (request == null || parms == null || sortStyle == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        return makeHiddenTags(request, sortStyle, parms.resType, parms.filters, parms.limit);
    }

    public static String makeHiddenTags(HttpServletRequest request, VulnerabilityQueryParms parms, VulnerabilityFactory.ResolutionType resType) {
        if (request == null || parms == null || resType == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        return makeHiddenTags(request, parms.sortStyle, resType, parms.filters, parms.limit);
    }

    public static String makeHiddenTags(HttpServletRequest request, VulnerabilityQueryParms parms, ArrayList filters) {
        if (request == null || parms == null || filters == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        return makeHiddenTags(request, parms.sortStyle, parms.resType, filters, parms.limit);
    }

    public static String makeHiddenTags(HttpServletRequest request, VulnerabilityQueryParms parms, Filter filter, boolean add) {
        if (request == null || parms == null || filter == null) {
            throw new IllegalArgumentException("Cannot take null parameters.");
        }

        ArrayList newList = new ArrayList(parms.filters);

        if (add) {
            newList.add(filter);
        } else {
            newList.remove(filter);
        }

        return makeHiddenTags(request, parms.sortStyle, parms.resType, newList, parms.limit);
    }

}
