//
// This file is part of the OpenNMS(R) Application.
//
// OpenNMS(R) is Copyright (C) 2002-2003 The OpenNMS Group, Inc.  All rights reserved.
// OpenNMS(R) is a derivative work, containing both original code, included code and modified
// code that was published under the GNU General Public License. Copyrights for modified 
// and included code are below.
//
// OpenNMS(R) is a registered trademark of The OpenNMS Group, Inc.
//
// Copyright (C) 1999-2001 Oculan Corp.  All rights reserved.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.                                                            
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//       
// For more information contact: 
//      OpenNMS Licensing       <license@opennms.org>
//      http://www.opennms.org/
//      http://www.opennms.com/
//

package org.opennms.core.utils;

import org.opennms.core.utils.ThreadCategory;

/**
 * Convenience classes for parsing and manipulating Strings.
 *
 * @author <a href="mailto:mike@opennms.org">Mike Davidson </a>
 * @author <a href="http://www.opennms.org">OpenNMS </a>
 * @author <a href="mailto:mike@opennms.org">Mike Davidson </a>
 * @author <a href="http://www.opennms.org">OpenNMS </a>
 * @version $Id: $
 */
public class AlphaNumeric extends Object {

    /**
     * Any character in the passed string which does not match one of the
     * following values is replaced by the specified replacement character.
     *
     * Ascii chars: 0 - 9 (Decimal 48 - 57) A - Z (Decimal 65 - 90) a - z
     * (Decimal 97 - 122)
     *
     * For example: 'Ethernet 10/100' is converted to 'Ethernet_10_100'
     *
     * @param str
     *            string to be converted
     * @param replacement
     *            replacement character
     * @return Converted value which can be used in a file name.
     */
    public static String parseAndReplace(String str, char replacement) {
	return parseAndReplaceExcept(str, replacement, new Character(replacement).toString());
    }

    /**
     * Any character in the passed string which does not match one of the
     * following values is replaced by the specified replacement character,
     * unless it is contained in the exception string.
     *
     * Ascii chars: 0 - 9 (Decimal 48 - 57) A - Z (Decimal 65 - 90) a - z
     * (Decimal 97 - 122)
     *
     * For example: 'Ethernet 10/100' is converted to 'Ethernet_10_100'
     *
     * @param str
     *            string to be converted
     * @param replacement
     *            replacement character
     * @param except
     *            string containing exception characters
     * @return Converted value which can be used in a file name.
     */
    public static String parseAndReplaceExcept(String str, char replacement, String except) {
        if (str == null) {
            return "";
        } else {
            boolean replacedChar = false;
            byte[] bytes = str.getBytes();
            byte[] exBytes = except.getBytes();

            blat: for (int x = 0; x < bytes.length; x++) {
                if (!((bytes[x] >= 48 && bytes[x] <= 57) || (bytes[x] >= 65 && bytes[x] <= 90) || (bytes[x] >= 97 && bytes[x] <= 122))) {
		    for (int y = 0; y < exBytes.length; y++) {
			if (bytes[x] == exBytes[y]) {
			    continue blat;
			}
		    }
                    bytes[x] = (byte) replacement;
                    replacedChar = true;
                }
            }

            String temp = new String(bytes);

            // Log4j category
            //
            ThreadCategory log = ThreadCategory.getInstance(AlphaNumeric.class);
            if (log.isDebugEnabled()) {
                if (replacedChar)
                    log.debug("parseAndReplace: original='" + str + "'" + " new='" + temp + "'");
            }

            return temp;
        }
    }

    /**
     * Any character in the passed string which does not match one of the
     * following values is replaced by an Ascii space and then trimmed from the
     * resulting string.
     *
     * Ascii chars: 0 - 9 (Decimal 48 - 57) A - Z (Decimal 65 - 90) a - z
     * (Decimal 97 - 122)
     *
     * @param str
     *            string to be converted
     * @return Converted value.
     */
    public static String parseAndTrim(String str) {
        if (str == null) {
            return "";
        } else {
            byte[] bytes = str.getBytes();

            for (int x = 0; x < bytes.length; x++) {
                if (!((bytes[x] == 32) || (bytes[x] >= 48 && bytes[x] <= 57) || (bytes[x] >= 65 && bytes[x] <= 90) || (bytes[x] >= 97 && bytes[x] <= 122))) {
                    bytes[x] = 32; // Ascii space
                }
            }

            String temp = new String(bytes);
            temp = temp.trim();

            return temp;
        }
    }
}
