/*
 * This file is part of the OpenNMS(R) Application.
 *
 * OpenNMS(R) is Copyright (C) 2007 The OpenNMS Group, Inc. All rights reserved.
 * OpenNMS(R) is a derivative work, containing both original code, included code and modified
 * code that was published under the GNU General Public License. Copyrights for modified
 * and included code are below.
 *
 * OpenNMS(R) is a registered trademark of The OpenNMS Group, Inc.
 *
 * Modifications:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * For more information contact:
 *      OpenNMS Licensing <license@opennms.org>
 *      http://www.opennms.org/
 *      http://www.opennms.com/
 */
package org.opennms.netmgt.ticketd;

import java.util.HashMap;
import java.util.Map;

/**
 * OpenNMS Trouble Ticket Model class used to contain common ticket data
 * by implementations of <code>TicketerPlugin</code> API.
 * 
 * @author <a href="mailto:brozow@opennms.org">Mathew Brozowski</a>
 * @author <a href="mailto:david@opennms.org">David Hustace</a>
 *
 */
public class Ticket {
    
    /**
     * Enumeration for representation of a Ticket's state.
     * @author <a href="mailto:brozow@opennms.org">Mathew Brozowski</a>
     *
     */
    public enum State {
        /**
         * Ticket is concidered to be in an Open State in the HelpDesk system.
         */
        OPEN,
        /**
         * Ticket is concidered to be in a Canceled State in the HelpDesk system.
         */
        CANCELLED,
        /**
         * Ticket is concidered to be in an Closed State in the HelpDesk system.
         */
        CLOSED
    }
    
    private String m_id;
    private State m_state = State.OPEN;
    private String m_summary;
    private String m_details;
    private String m_user;
    private String m_modificationTimestamp;
    private Map<String, String> m_attributes;
    
    /**
     * @return a Map of free from attributes of the ticket.  Typically,
     * from OnmsAlarm attributes.
     */
    public Map<String, String> getAttributes() {
        return m_attributes;
    }
    
    /**
     * Store a list of free form attributes in the Ticket.  Typically, from
     * the OnmsAlarm attributes.
     * 
     * @param attributes
     */
    public void setAttributes(Map<String, String> attributes) {
        m_attributes = attributes;
    }
    
    /**
     * Adds a single free form attribute to the Ticket.
     * 
     * @param key
     * @param value
     */
    public void addAttribute(String key, String value) {
        if (m_attributes == null) {
            m_attributes = new HashMap<String, String>();
        }
        m_attributes.put(key, value);
    }
    
    /**
     * Gets a single free form attribute from a Ticket.
     * 
     * @param key
     * @return
     */
    public String getAttribute(String key) {
        if (m_attributes == null) {
            return null;
        }
        return m_attributes.get(key);
    }
    
    /**
     * Returns a simple high level summary about the ticket that is generated
     * from the Alarm logmsg.
     * 
     * @return A string containing the summary of the ticket.
     */
    public String getSummary() {
        return m_summary;
    }
    
    /**
     * Set a summary into the ticket.  Typically the alarm's logmsg.
     * 
     * @param summary
     */
    public void setSummary(String summary) {
        m_summary = summary;
    }
    
    /**
     * TODO: This should probably turn into a collection of comments.
     * 
     * @return A string of details about the Ticket.
     */
    public String getDetails() {
        return m_details;
    }
    /**
     * TODO: This should probably turn into a collection of comments or some such thing.
     * @param details
     */
    public void setDetails(String details) {
        m_details = details;
    }
    
    /**
     * This should be the ticket ID assigned by the HelpDesk system.
     * @return
     */
    public String getId() {
        return m_id;
    }
    
    /**
     * The TicketerPlugin should set the ID.
     * @param id
     */
    public void setId(String id) {
        m_id = id;
    }
    
    /**
     * User name owning/createing the ticket.  Will be set initially to
     * the user name set in the parameter of the create ticket event.
     * 
     * @return a String containing the user name that owns the ticket.
     */
    public String getUser() {
        return m_user;
    }
    
    /**
     * Set the user name owning the ticket.  Typically set by the TicketerServiceLayer
     * implemenation as the user name parameter from the create ticket event.
     * 
     * @param user
     */
    public void setUser(String user) {
        m_user = user;
    }
    
    /**
     * Returns the current <code>Ticket.State</code>
     * 
     * @return the ticket state enum.
     */
    public State getState() {
        return m_state;
    }
    
    /**
     * Sets the Ticket state to one of the <code>Ticket.State</code> Enums.
     * 
     * @param state
     */
    public void setState(State state) {
        m_state = state;
    }

    /**
     * A timestamp to be used for optimistic locking with the trouble ticketing system
     * @return
     */
    public String getModificationTimestamp() {
        return m_modificationTimestamp;
    }

    public void setModificationTimestamp(String modificationTimestamp) {
        m_modificationTimestamp = modificationTimestamp;
    }
    
}
