/*
 * Decompiled with CFR 0.152.
 */
package org.opennms.netmgt.flows.elastic;

import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.Timer;
import com.google.common.base.Strings;
import com.google.common.cache.CacheLoader;
import java.io.Serializable;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;
import org.opennms.core.cache.Cache;
import org.opennms.core.cache.CacheBuilder;
import org.opennms.core.cache.CacheConfig;
import org.opennms.core.cache.CacheConfigBuilder;
import org.opennms.core.rpc.utils.mate.ContextKey;
import org.opennms.core.utils.InetAddressUtils;
import org.opennms.netmgt.dao.api.InterfaceToNodeCache;
import org.opennms.netmgt.dao.api.NodeDao;
import org.opennms.netmgt.dao.api.SessionUtils;
import org.opennms.netmgt.flows.api.Flow;
import org.opennms.netmgt.flows.api.FlowSource;
import org.opennms.netmgt.flows.classification.ClassificationEngine;
import org.opennms.netmgt.flows.classification.ClassificationRequest;
import org.opennms.netmgt.flows.classification.persistence.api.Protocols;
import org.opennms.netmgt.flows.elastic.ConversationKeyUtils;
import org.opennms.netmgt.flows.elastic.FlowDocument;
import org.opennms.netmgt.flows.elastic.Locality;
import org.opennms.netmgt.flows.elastic.NodeDocument;
import org.opennms.netmgt.model.OnmsCategory;
import org.opennms.netmgt.model.OnmsNode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DocumentEnricher {
    private static final Logger LOG = LoggerFactory.getLogger(DocumentEnricher.class);
    private static final String NODE_METADATA_CACHE = "flows.node.metadata";
    private final NodeDao nodeDao;
    private final InterfaceToNodeCache interfaceToNodeCache;
    private final SessionUtils sessionUtils;
    private final ClassificationEngine classificationEngine;
    private final Cache<Integer, Optional<NodeDocument>> nodeInfoCache;
    private final Cache<NodeMetadataKey, Optional<NodeDocument>> nodeMetadataCache;
    private final Timer nodeLoadTimer;
    private final long clockSkewCorrectionThreshold;

    public DocumentEnricher(MetricRegistry metricRegistry, NodeDao nodeDao, InterfaceToNodeCache interfaceToNodeCache, SessionUtils sessionUtils, ClassificationEngine classificationEngine, CacheConfig cacheConfig, long clockSkewCorrectionThreshold) {
        this.nodeDao = Objects.requireNonNull(nodeDao);
        this.interfaceToNodeCache = Objects.requireNonNull(interfaceToNodeCache);
        this.sessionUtils = Objects.requireNonNull(sessionUtils);
        this.classificationEngine = Objects.requireNonNull(classificationEngine);
        this.nodeInfoCache = new CacheBuilder().withConfig(cacheConfig).withCacheLoader((CacheLoader)new CacheLoader<Integer, Optional<NodeDocument>>(){

            public Optional<NodeDocument> load(Integer nodeId) {
                return DocumentEnricher.this.getNodeInfo(nodeId);
            }
        }).build();
        CacheConfig nodeMetadataCacheConfig = this.buildMetadataCacheConfig(cacheConfig);
        this.nodeMetadataCache = new CacheBuilder().withConfig(nodeMetadataCacheConfig).withCacheLoader((CacheLoader)new CacheLoader<NodeMetadataKey, Optional<NodeDocument>>(){

            public Optional<NodeDocument> load(NodeMetadataKey key) {
                return DocumentEnricher.this.getNodeInfoFromMetadataContext(key.contextKey, key.value);
            }
        }).build();
        this.nodeLoadTimer = metricRegistry.timer("nodeLoadTime");
        this.clockSkewCorrectionThreshold = clockSkewCorrectionThreshold;
    }

    public List<FlowDocument> enrich(Collection<Flow> flows, FlowSource source) {
        if (flows.isEmpty()) {
            LOG.info("Nothing to enrich.");
            return Collections.emptyList();
        }
        return (List)this.sessionUtils.withTransaction(() -> flows.stream().map(flow -> {
            long skew;
            FlowDocument document = FlowDocument.from(flow);
            document.setHost(source.getSourceAddress());
            document.setLocation(source.getLocation());
            this.getNodeInfoFromCache(source.getLocation(), source.getSourceAddress(), source.getContextKey(), flow.getNodeIdentifier()).ifPresent(document::setNodeExporter);
            if (document.getDstAddr() != null) {
                this.getNodeInfoFromCache(source.getLocation(), document.getDstAddr(), null, null).ifPresent(document::setNodeDst);
            }
            if (document.getSrcAddr() != null) {
                this.getNodeInfoFromCache(source.getLocation(), document.getSrcAddr(), null, null).ifPresent(document::setNodeSrc);
            }
            if (document.getSrcAddr() != null) {
                document.setSrcLocality(DocumentEnricher.isPrivateAddress(document.getSrcAddr()) ? Locality.PRIVATE : Locality.PUBLIC);
            }
            if (document.getDstAddr() != null) {
                document.setDstLocality(DocumentEnricher.isPrivateAddress(document.getDstAddr()) ? Locality.PRIVATE : Locality.PUBLIC);
            }
            if (Locality.PUBLIC.equals((Object)document.getDstLocality()) || Locality.PUBLIC.equals((Object)document.getSrcLocality())) {
                document.setFlowLocality(Locality.PUBLIC);
            } else if (Locality.PRIVATE.equals((Object)document.getDstLocality()) || Locality.PRIVATE.equals((Object)document.getSrcLocality())) {
                document.setFlowLocality(Locality.PRIVATE);
            }
            ClassificationRequest classificationRequest = DocumentEnricher.createClassificationRequest(document);
            if (classificationRequest.isClassifiable()) {
                document.setApplication(this.classificationEngine.classify(classificationRequest));
            }
            document.setConvoKey(ConversationKeyUtils.getConvoKeyAsJsonString(document));
            if (this.clockSkewCorrectionThreshold > 0L && Math.abs(skew = flow.getTimestamp() - flow.getReceivedAt()) >= this.clockSkewCorrectionThreshold) {
                document.setClockCorrection(-skew);
                document.setTimestamp(document.getTimestamp() - skew);
                document.setFirstSwitched(document.getFirstSwitched() - skew);
                document.setDeltaSwitched(document.getDeltaSwitched() - skew);
                document.setLastSwitched(document.getLastSwitched() - skew);
            }
            return document;
        }).collect(Collectors.toList()));
    }

    private static boolean isPrivateAddress(String ipAddress) {
        InetAddress inetAddress = InetAddressUtils.addr((String)ipAddress);
        return inetAddress.isLoopbackAddress() || inetAddress.isLinkLocalAddress() || inetAddress.isSiteLocalAddress();
    }

    private Optional<NodeDocument> getNodeInfoFromCache(String location, String ipAddress, ContextKey contextKey, String value) {
        Optional nodeId;
        Optional nodeDocument = Optional.empty();
        if (contextKey != null && !Strings.isNullOrEmpty((String)value)) {
            NodeMetadataKey metadataKey = new NodeMetadataKey(contextKey, value);
            try {
                nodeDocument = (Optional)this.nodeMetadataCache.get((Object)metadataKey);
            }
            catch (ExecutionException e) {
                LOG.error("Error while retrieving NodeDocument from NodeMetadataCache: {}.", (Object)e.getMessage(), (Object)e);
                throw new RuntimeException(e);
            }
            if (nodeDocument.isPresent()) {
                return nodeDocument;
            }
        }
        if ((nodeId = this.interfaceToNodeCache.getFirstNodeId(location, InetAddressUtils.addr((String)ipAddress))).isPresent()) {
            try {
                return (Optional)this.nodeInfoCache.get(nodeId.get());
            }
            catch (ExecutionException e) {
                LOG.error("Error while retrieving NodeDocument from NodeInfoCache: {}.", (Object)e.getMessage(), (Object)e);
                throw new RuntimeException(e);
            }
        }
        return nodeDocument;
    }

    private Optional<NodeDocument> getNodeInfoFromMetadataContext(ContextKey contextKey, String value) {
        List nodes = new ArrayList();
        try (Timer.Context ctx = this.nodeLoadTimer.time();){
            nodes = this.nodeDao.findNodeWithMetaData(contextKey.getContext(), contextKey.getKey(), value);
        }
        if (nodes.isEmpty()) {
            return Optional.empty();
        }
        return this.mapOnmsNodeToNodeDocument((OnmsNode)nodes.get(0));
    }

    private Optional<NodeDocument> getNodeInfo(Integer nodeId) {
        OnmsNode onmsNode = null;
        try (Timer.Context ctx = this.nodeLoadTimer.time();){
            onmsNode = (OnmsNode)this.nodeDao.get((Serializable)nodeId);
        }
        return this.mapOnmsNodeToNodeDocument(onmsNode);
    }

    private Optional<NodeDocument> mapOnmsNodeToNodeDocument(OnmsNode onmsNode) {
        if (onmsNode != null) {
            NodeDocument nodeDocument = new NodeDocument();
            nodeDocument.setForeignSource(onmsNode.getForeignSource());
            nodeDocument.setForeignId(onmsNode.getForeignId());
            nodeDocument.setNodeId(onmsNode.getId());
            nodeDocument.setCategories(onmsNode.getCategories().stream().map(OnmsCategory::getName).collect(Collectors.toList()));
            return Optional.of(nodeDocument);
        }
        return Optional.empty();
    }

    protected static ClassificationRequest createClassificationRequest(FlowDocument document) {
        ClassificationRequest request = new ClassificationRequest();
        request.setProtocol(document.getProtocol() == null ? null : Protocols.getProtocol((int)document.getProtocol()));
        request.setLocation(document.getLocation());
        request.setExporterAddress(document.getHost());
        request.setDstAddress(document.getDstAddr());
        request.setDstPort(document.getDstPort());
        request.setSrcAddress(document.getSrcAddr());
        request.setSrcPort(document.getSrcPort());
        return request;
    }

    private CacheConfig buildMetadataCacheConfig(CacheConfig cacheConfig) {
        CacheConfig metadataCacheConfig = new CacheConfigBuilder().withName(NODE_METADATA_CACHE).withMaximumSize(cacheConfig.getMaximumSize().longValue()).withExpireAfterWrite(cacheConfig.getExpireAfterWrite().longValue()).build();
        cacheConfig.setRecordStats(true);
        cacheConfig.setMetricRegistry(cacheConfig.getMetricRegistry());
        return metadataCacheConfig;
    }

    private static class NodeMetadataKey {
        public final ContextKey contextKey;
        public final String value;

        private NodeMetadataKey(ContextKey contextKey, String value) {
            this.contextKey = contextKey;
            this.value = value;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            NodeMetadataKey that = (NodeMetadataKey)o;
            return Objects.equals(this.contextKey, that.contextKey) && Objects.equals(this.value, that.value);
        }

        public int hashCode() {
            return Objects.hash(this.contextKey, this.value);
        }
    }
}

