/*
 * Decompiled with CFR 0.152.
 */
package org.opennms.features.newts.converter;

import com.google.common.base.Optional;
import com.google.common.base.Throwables;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.common.primitives.UnsignedLong;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.NavigableSet;
import java.util.Properties;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.cli.PosixParser;
import org.apache.commons.io.FilenameUtils;
import org.joda.time.Interval;
import org.joda.time.Period;
import org.joda.time.ReadablePeriod;
import org.joda.time.format.PeriodFormatter;
import org.joda.time.format.PeriodFormatterBuilder;
import org.opennms.core.db.DataSourceFactory;
import org.opennms.features.newts.converter.NewtsConverterError;
import org.opennms.features.newts.converter.OnmsProperties;
import org.opennms.netmgt.model.ResourcePath;
import org.opennms.netmgt.newts.support.NewtsUtils;
import org.opennms.netmgt.rrd.model.AbstractDS;
import org.opennms.netmgt.rrd.model.AbstractRRA;
import org.opennms.netmgt.rrd.model.AbstractRRD;
import org.opennms.netmgt.rrd.model.Row;
import org.opennms.netmgt.rrd.model.RrdConvertUtils;
import org.opennms.netmgt.rrd.model.v3.RRDv3;
import org.opennms.newts.api.Counter;
import org.opennms.newts.api.Gauge;
import org.opennms.newts.api.MetricType;
import org.opennms.newts.api.Resource;
import org.opennms.newts.api.Sample;
import org.opennms.newts.api.SampleRepository;
import org.opennms.newts.api.Timestamp;
import org.opennms.newts.api.ValueType;
import org.opennms.newts.api.search.Indexer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.support.ClassPathXmlApplicationContext;

public class NewtsConverter
implements AutoCloseable {
    private static final Logger LOG = LoggerFactory.getLogger(NewtsConverter.class);
    private static final String CMD_SYNTAX = "newts-converter [options]";
    private static final Timestamp EPOCH = Timestamp.fromEpochMillis((long)0L);
    private static final ValueType<?> ZERO = ValueType.compose((Number)0, (MetricType)MetricType.GAUGE);
    private final ClassPathXmlApplicationContext context;
    private final Path onmsHome;
    private final Path rrdDir;
    private final Path rrdBinary;
    private final StorageStrategy storageStrategy;
    private final StorageTool storageTool;
    private final Map<Integer, ForeignId> foreignIds = Maps.newHashMap();
    private final SampleRepository repository;
    private final Indexer indexer;
    private static AtomicLong processedMetrics = new AtomicLong(0L);
    private static AtomicLong processedSamples = new AtomicLong(0L);
    private int batchSize;
    private final ExecutorService executor;

    public static void main(String ... args) {
        long start;
        try (NewtsConverter converter = new NewtsConverter(args);){
            start = System.currentTimeMillis();
            converter.execute();
        }
        catch (NewtsConverterError e) {
            LOG.error(e.getMessage(), (Throwable)e);
            System.exit(1);
            throw null;
        }
        Period period = new Interval(start, System.currentTimeMillis()).toPeriod();
        PeriodFormatter formatter = new PeriodFormatterBuilder().appendDays().appendSuffix(" days ").appendHours().appendSuffix(" hours ").appendMinutes().appendSuffix(" min ").appendSeconds().appendSuffix(" sec ").printZeroNever().toFormatter();
        LOG.info("Conversion Finished: metrics: {}, samples: {}, time: {}", new Object[]{processedMetrics, processedSamples, formatter.print((ReadablePeriod)period)});
        System.exit(0);
    }

    private NewtsConverter(String ... args) {
        int threads;
        CommandLine cmd;
        Options options = new Options();
        Option helpOption = new Option("h", "help", false, "Print this help");
        options.addOption(helpOption);
        Option opennmsHomeOption = new Option("o", "onms-home", true, "OpenNMS Home Directory (defaults to /opt/opennms)");
        options.addOption(opennmsHomeOption);
        Option rrdPathOption = new Option("r", "rrd-dir", true, "The path to the RRD data (defaults to ONMS-HOME/share/rrd)");
        options.addOption(rrdPathOption);
        Option rrdToolOption = new Option("t", "rrd-tool", true, "Whether to use rrdtool or JRobin (defaults to use rrdtool)");
        options.addOption(rrdToolOption);
        Option rrdBinaryOption = new Option("T", "rrd-binary", true, "The binary path to the rrdtool command (defaults to /usr/bin/rrdtool, only used if rrd-tool is set)");
        options.addOption(rrdBinaryOption);
        Option storeByGroupOption = new Option("s", "storage-strategy", true, "Whether store by group was enabled or not");
        storeByGroupOption.setRequired(true);
        options.addOption(storeByGroupOption);
        Option threadsOption = new Option("n", "threads", true, "Number of conversion threads (defaults to number of CPUs)");
        options.addOption(threadsOption);
        PosixParser parser = new PosixParser();
        try {
            cmd = parser.parse(options, args);
        }
        catch (ParseException e) {
            new HelpFormatter().printHelp(80, CMD_SYNTAX, String.format("ERROR: %s%n", e.getMessage()), options, null);
            System.exit(1);
            throw null;
        }
        if (cmd.hasOption('h')) {
            new HelpFormatter().printHelp(80, CMD_SYNTAX, null, options, null);
            System.exit(0);
        }
        Path path = this.onmsHome = cmd.hasOption('o') ? Paths.get(cmd.getOptionValue('o'), new String[0]) : Paths.get("/opt/opennms", new String[0]);
        if (!Files.exists(this.onmsHome, new LinkOption[0]) || !Files.isDirectory(this.onmsHome, new LinkOption[0])) {
            new HelpFormatter().printHelp(80, CMD_SYNTAX, String.format("ERROR: Directory %s doesn't exist%n", this.onmsHome.toAbsolutePath()), options, null);
            System.exit(1);
            throw null;
        }
        System.setProperty("opennms.home", this.onmsHome.toAbsolutePath().toString());
        Path path2 = this.rrdDir = cmd.hasOption('r') ? Paths.get(cmd.getOptionValue('r'), new String[0]) : this.onmsHome.resolve("share").resolve("rrd");
        if (!Files.exists(this.rrdDir, new LinkOption[0]) || !Files.isDirectory(this.rrdDir, new LinkOption[0])) {
            new HelpFormatter().printHelp(80, CMD_SYNTAX, String.format("ERROR: Directory %s doesn't exist%n", this.rrdDir.toAbsolutePath()), options, null);
            System.exit(1);
            throw null;
        }
        if (!cmd.hasOption('s')) {
            new HelpFormatter().printHelp(80, CMD_SYNTAX, String.format("ERROR: Option for storage-strategy must be spcified%n", new Object[0]), options, null);
            System.exit(1);
            throw null;
        }
        switch (cmd.getOptionValue('s').toLowerCase()) {
            case "storeByMetric": 
            case "sbm": 
            case "false": {
                this.storageStrategy = StorageStrategy.STORE_BY_METRIC;
                break;
            }
            case "storeByGroup": 
            case "sbg": 
            case "true": {
                this.storageStrategy = StorageStrategy.STORE_BY_GROUP;
                break;
            }
            default: {
                new HelpFormatter().printHelp(80, CMD_SYNTAX, String.format("ERROR: Invalid value for storage-strategy%n", new Object[0]), options, null);
                System.exit(1);
                throw null;
            }
        }
        if (!cmd.hasOption('t')) {
            new HelpFormatter().printHelp(80, CMD_SYNTAX, String.format("ERROR: Option rrd-tool must be specified%n", new Object[0]), options, null);
            System.exit(1);
            throw null;
        }
        switch (cmd.getOptionValue('t').toLowerCase()) {
            case "rrdtool": 
            case "rrd": 
            case "true": {
                this.storageTool = StorageTool.RRDTOOL;
                break;
            }
            case "jrobin": 
            case "jrb": 
            case "false": {
                this.storageTool = StorageTool.JROBIN;
                break;
            }
            default: {
                new HelpFormatter().printHelp(80, CMD_SYNTAX, String.format("ERROR: Invalid value for rrd-tool%n", new Object[0]), options, null);
                System.exit(1);
                throw null;
            }
        }
        Path path3 = this.rrdBinary = cmd.hasOption('T') ? Paths.get(cmd.getOptionValue('T'), new String[0]) : Paths.get("/usr/bin/rrdtool", new String[0]);
        if (!Files.exists(this.rrdBinary, new LinkOption[0]) || !Files.isExecutable(this.rrdBinary)) {
            new HelpFormatter().printHelp(80, CMD_SYNTAX, String.format("ERROR: RRDtool command %s doesn't exist%n", this.rrdBinary.toAbsolutePath()), options, null);
            System.exit(1);
            throw null;
        }
        System.setProperty("rrd.binary", this.rrdBinary.toString());
        try {
            threads = cmd.hasOption('n') ? Integer.parseInt(cmd.getOptionValue('n')) : Runtime.getRuntime().availableProcessors();
            this.executor = new ForkJoinPool(threads, ForkJoinPool.defaultForkJoinWorkerThreadFactory, null, true);
        }
        catch (Exception e) {
            new HelpFormatter().printHelp(80, CMD_SYNTAX, String.format("ERROR: Invalid number of threads: %s%n", e.getMessage()), options, null);
            System.exit(1);
            throw null;
        }
        OnmsProperties.initialize();
        String host = System.getProperty("org.opennms.newts.config.hostname", "localhost");
        String keyspace = System.getProperty("org.opennms.newts.config.keyspace", "newts");
        int ttl = Integer.parseInt(System.getProperty("org.opennms.newts.config.ttl", "31540000"));
        int port = Integer.parseInt(System.getProperty("org.opennms.newts.config.port", "9042"));
        this.batchSize = Integer.parseInt(System.getProperty("org.opennms.newts.config.max_batch_size", "16"));
        LOG.info("OpenNMS Home: {}", (Object)this.onmsHome);
        LOG.info("RRD Directory: {}", (Object)this.rrdDir);
        LOG.info("Use RRDtool Tool: {}", (Object)this.storageTool);
        LOG.info("RRDtool CLI: {}", (Object)this.rrdBinary);
        LOG.info("StoreByGroup: {}", (Object)this.storageStrategy);
        LOG.info("Conversion Threads: {}", (Object)threads);
        LOG.info("Cassandra Host: {}", (Object)host);
        LOG.info("Cassandra Port: {}", (Object)port);
        LOG.info("Cassandra Keyspace: {}", (Object)keyspace);
        LOG.info("Newts Max Batch Size: {}", (Object)this.batchSize);
        LOG.info("Newts TTL: {}", (Object)ttl);
        if (!"newts".equals(System.getProperty("org.opennms.timeseries.strategy", "rrd"))) {
            throw NewtsConverterError.create("The configured timeseries strategy must be 'newts' on opennms.properties (org.opennms.timeseries.strategy)", new Object[0]);
        }
        if (!"true".equals(System.getProperty("org.opennms.rrd.storeByForeignSource", "false"))) {
            throw NewtsConverterError.create("The option storeByForeignSource must be enabled in opennms.properties (org.opennms.rrd.storeByForeignSource)", new Object[0]);
        }
        try {
            this.context = new ClassPathXmlApplicationContext(new String[]{"classpath:/META-INF/opennms/applicationContext-soa.xml", "classpath:/META-INF/opennms/applicationContext-timeseries-newts.xml"});
            this.repository = (SampleRepository)this.context.getBean(SampleRepository.class);
            this.indexer = (Indexer)this.context.getBean(Indexer.class);
        }
        catch (Exception e) {
            throw NewtsConverterError.create(e, "Cannot connect to the Cassandra/Newts backend: {}", e.getMessage());
        }
        try (Connection conn = DataSourceFactory.getInstance().getConnection();
             Statement st = conn.createStatement();
             ResultSet rs = st.executeQuery("SELECT nodeid, foreignsource, foreignid from node n");){
            while (rs.next()) {
                this.foreignIds.put(rs.getInt("nodeid"), new ForeignId(rs.getString("foreignsource"), rs.getString("foreignid")));
            }
        }
        catch (Exception e) {
            throw NewtsConverterError.create(e, "Failed to connect to database: {}", e.getMessage());
        }
        LOG.trace("Found {} nodes on the database", (Object)this.foreignIds.size());
    }

    public void execute() {
        LOG.trace("Starting Conversion...");
        this.processStoreByGroupResources(this.rrdDir.resolve("response"));
        this.processStringsProperties(this.rrdDir.resolve("snmp"));
        switch (this.storageStrategy) {
            case STORE_BY_GROUP: {
                this.processStoreByGroupResources(this.rrdDir.resolve("snmp"));
                break;
            }
            case STORE_BY_METRIC: {
                this.processStoreByMetricResources(this.rrdDir.resolve("snmp"));
            }
        }
    }

    private void processStoreByGroupResources(Path path) {
        try {
            Files.walk(path, new FileVisitOption[0]).filter(p -> p.endsWith("ds.properties")).forEach(p -> this.processStoreByGroupResource(p.getParent()));
        }
        catch (Exception e) {
            LOG.error("Error while reading RRD files", (Throwable)e);
            return;
        }
    }

    private void processStoreByGroupResource(Path path) {
        Properties ds = new Properties();
        try (BufferedReader r = Files.newBufferedReader(path.resolve("ds.properties"));){
            ds.load(r);
        }
        catch (IOException e) {
            LOG.error("No group information found - please verify storageStrategy settings");
            return;
        }
        Sets.newHashSet((Iterable)Iterables.transform(ds.values(), Object::toString)).forEach(group -> this.executor.execute(() -> this.processResource(path, (String)group, (String)group)));
    }

    private void processStoreByMetricResources(Path path) {
        try {
            Files.walk(path, new FileVisitOption[0]).filter(p -> p.getFileName().toString().endsWith(".meta")).forEach(this::processStoreByMetricResource);
        }
        catch (Exception e) {
            LOG.error("Error while reading RRD files", (Throwable)e);
            return;
        }
    }

    private void processStoreByMetricResource(Path metaPath) {
        Path path = metaPath.getParent();
        String metric = FilenameUtils.removeExtension((String)metaPath.getFileName().toString());
        Properties meta = new Properties();
        try (BufferedReader r = Files.newBufferedReader(metaPath);){
            meta.load(r);
        }
        catch (IOException e) {
            LOG.error("Failed to read .meta file: {}", (Object)metaPath, (Object)e);
            return;
        }
        String group = meta.getProperty("GROUP");
        if (group == null) {
            LOG.warn("No group information found - please verify storageStrategy settings");
            return;
        }
        this.executor.execute(() -> this.processResource(path, metric, group));
    }

    private void processResource(Path resourceDir, String fileName, String group) {
        RRDv3 rrd;
        Path file;
        LOG.info("Processing resource: dir={}, file={}, group={}", new Object[]{resourceDir, fileName, group});
        ResourcePath resourcePath = this.buildResourcePath(resourceDir);
        if (resourcePath == null) {
            return;
        }
        switch (this.storageTool) {
            case RRDTOOL: {
                file = resourceDir.resolve(fileName + ".rrd");
                break;
            }
            case JROBIN: {
                file = resourceDir.resolve(fileName + ".jrb");
                break;
            }
            default: {
                file = null;
            }
        }
        if (!Files.exists(file, new LinkOption[0])) {
            LOG.error("File not found: {}", (Object)file);
            return;
        }
        try {
            switch (this.storageTool) {
                case RRDTOOL: {
                    rrd = RrdConvertUtils.dumpRrd((File)file.toFile());
                    break;
                }
                case JROBIN: {
                    rrd = RrdConvertUtils.dumpJrb((File)file.toFile());
                    break;
                }
                default: {
                    rrd = null;
                    break;
                }
            }
        }
        catch (Exception e) {
            LOG.error("Can't parse JRB/RRD file: {}", (Object)file, (Object)e);
            return;
        }
        try {
            this.injectSamplesToNewts(resourcePath, group, rrd.getDataSources(), NewtsConverter.generateSamples((AbstractRRD)rrd));
        }
        catch (Exception e) {
            LOG.error("Failed to convert file: {}", (Object)file, (Object)e);
            return;
        }
    }

    protected static Sample toSample(AbstractDS ds, Resource resource, Timestamp timestamp, double value) {
        String metric = ds.getName();
        MetricType type = ds.isCounter() ? MetricType.COUNTER : MetricType.GAUGE;
        Counter valueType = ds.isCounter() ? new Counter(UnsignedLong.valueOf((BigInteger)BigDecimal.valueOf(value).toBigInteger())) : new Gauge(value);
        return new Sample(timestamp, resource, metric, type, (ValueType)valueType);
    }

    private void injectSamplesToNewts(ResourcePath resourcePath, String group, List<? extends AbstractDS> dataSources, SortedMap<Long, List<Double>> samples) {
        ResourcePath groupPath = ResourcePath.get((ResourcePath)resourcePath, (String[])new String[]{group});
        String groupId = NewtsUtils.toResourceId((ResourcePath)groupPath);
        HashMap attributes = Maps.newHashMap();
        NewtsUtils.addIndicesToAttributes((ResourcePath)groupPath, (Map)attributes);
        Resource resource = new Resource(groupId, Optional.of((Object)attributes));
        ArrayList<Sample> batch = new ArrayList<Sample>(this.batchSize);
        for (Map.Entry<Long, List<Double>> s : samples.entrySet()) {
            for (int i = 0; i < dataSources.size(); ++i) {
                double value = s.getValue().get(i);
                if (Double.isNaN(value)) continue;
                AbstractDS ds = dataSources.get(i);
                Timestamp timestamp = Timestamp.fromEpochSeconds((long)s.getKey());
                try {
                    batch.add(NewtsConverter.toSample(ds, resource, timestamp, value));
                }
                catch (IllegalArgumentException e) {
                    continue;
                }
                if (batch.size() < this.batchSize) continue;
                this.repository.insert(batch, true);
                processedSamples.getAndAdd(batch.size());
                batch = new ArrayList(this.batchSize);
            }
        }
        if (!batch.isEmpty()) {
            this.repository.insert(batch, true);
            processedSamples.getAndAdd(batch.size());
        }
        processedMetrics.getAndAdd(dataSources.size());
        LOG.trace("Stats: {} / {}", (Object)processedMetrics, (Object)processedSamples);
    }

    private void processStringsProperties(Path path) {
        try {
            Files.walk(path, new FileVisitOption[0]).filter(p -> p.endsWith("strings.properties")).forEach(p -> {
                Properties properties = new Properties();
                try (BufferedReader r = Files.newBufferedReader(p);){
                    properties.load(r);
                }
                catch (IOException e) {
                    throw Throwables.propagate((Throwable)e);
                }
                ResourcePath resourcePath = this.buildResourcePath(p.getParent());
                if (resourcePath == null) {
                    return;
                }
                this.injectStringPropertiesToNewts(resourcePath, (Map<String, String>)Maps.fromProperties((Properties)properties));
            });
        }
        catch (Exception e) {
            LOG.error("Error while reading string.properties", (Throwable)e);
            return;
        }
    }

    private void injectStringPropertiesToNewts(ResourcePath resourcePath, Map<String, String> stringProperties) {
        Resource resource = new Resource(NewtsUtils.toResourceId((ResourcePath)resourcePath), Optional.of(stringProperties));
        Sample sample = new Sample(EPOCH, resource, "strings", MetricType.GAUGE, ZERO);
        this.indexer.update((Collection)Lists.newArrayList((Object[])new Sample[]{sample}));
    }

    private ResourcePath buildResourcePath(Path resourceDir) {
        ResourcePath resourcePath;
        Path relativeResourceDir = this.rrdDir.relativize(resourceDir);
        if (relativeResourceDir.startsWith(Paths.get("snmp", new String[0])) && !relativeResourceDir.startsWith(Paths.get("snmp", "fs"))) {
            int nodeId = Integer.parseInt(relativeResourceDir.getName(1).toString());
            ForeignId foreignId = this.foreignIds.get(nodeId);
            if (foreignId == null) {
                return null;
            }
            resourcePath = ResourcePath.get((ResourcePath)ResourcePath.get((ResourcePath)ResourcePath.get((String[])new String[]{"snmp", "fs"}), (String[])new String[]{foreignId.foreignSource, foreignId.foreignId}), (Iterable)Iterables.transform((Iterable)Iterables.skip((Iterable)relativeResourceDir, (int)2), Path::toString));
        } else {
            resourcePath = ResourcePath.get((Iterable)Iterables.transform((Iterable)relativeResourceDir, Path::toString));
        }
        return resourcePath;
    }

    @Override
    public void close() {
        this.executor.shutdown();
        while (!this.executor.isTerminated()) {
            try {
                this.executor.awaitTermination(10L, TimeUnit.SECONDS);
            }
            catch (InterruptedException interruptedException) {}
        }
        this.context.close();
    }

    public static SortedMap<Long, List<Double>> generateSamples(AbstractRRD rrd) {
        NavigableSet rras = Stream.concat(rrd.getRras().stream().filter(rra -> rra.getPdpPerRow() == 1L).max(Comparator.comparingInt(rra -> rra.getRows().size())).map(Stream::of).orElseGet(Stream::empty), rrd.getRras().stream().filter(AbstractRRA::hasAverageAsCF)).collect(Collectors.toCollection(() -> new TreeSet<AbstractRRA>(Comparator.comparing(AbstractRRA::getPdpPerRow))));
        TreeMap<Long, List<Double>> collected = new TreeMap<Long, List<Double>>();
        for (AbstractRRA rra2 : rras) {
            NavigableMap<Long, List<Double>> samples = NewtsConverter.generateSamples(rrd, rra2);
            AbstractRRA lowerRra = rras.lower(rra2);
            if (lowerRra != null) {
                long lowerRraStart = rrd.getLastUpdate() - lowerRra.getPdpPerRow() * rrd.getStep() * (long)lowerRra.getRows().size();
                long rraStep = rra2.getPdpPerRow() * rrd.getStep();
                samples = samples.headMap((long)((int)Math.ceil((double)lowerRraStart / (double)rraStep)) * rraStep, false);
            }
            if (samples.isEmpty()) continue;
            AbstractRRA higherRra = rras.higher(rra2);
            if (higherRra != null) {
                long higherRraStep = higherRra.getPdpPerRow() * rrd.getStep();
                samples = samples.tailMap((long)((int)Math.ceil((double)((Long)samples.firstKey()).longValue() / (double)higherRraStep)) * higherRraStep, true);
            }
            collected.putAll(samples);
        }
        return collected;
    }

    public static NavigableMap<Long, List<Double>> generateSamples(AbstractRRD rrd, AbstractRRA rra) {
        long step = rra.getPdpPerRow() * rrd.getStep();
        long start = rrd.getStartTimestamp(rra);
        long end = rrd.getEndTimestamp(rra);
        TreeMap<Long, List<Double>> valuesMap = new TreeMap<Long, List<Double>>();
        for (long ts = start; ts <= end; ts += step) {
            ArrayList<Double> values = new ArrayList<Double>();
            for (int i = 0; i < rrd.getDataSources().size(); ++i) {
                values.add(Double.NaN);
            }
            valuesMap.put(ts, values);
        }
        ArrayList<Double> lastValues = new ArrayList<Double>();
        for (AbstractDS ds : rrd.getDataSources()) {
            double v = ds.getLastDs() == null ? 0.0 : ds.getLastDs();
            lastValues.add(v - v % (double)step);
        }
        for (int i = 0; i < rrd.getDataSources().size(); ++i) {
            if (!rrd.getDataSource(i).isCounter()) continue;
            ((List)valuesMap.get(end)).set(i, lastValues.get(i));
        }
        long ts = end - step;
        for (int j = rra.getRows().size() - 1; j >= 0; --j) {
            Row row = (Row)rra.getRows().get(j);
            for (int i = 0; i < rrd.getDataSources().size(); ++i) {
                if (rrd.getDataSource(i).isCounter()) {
                    if (j <= 0) continue;
                    Double last = (Double)lastValues.get(i);
                    Double current = row.getValue(i).isNaN() ? 0.0 : row.getValue(i);
                    Double value = last - current * (double)step;
                    if (value < 0.0) {
                        value = value + Math.pow(2.0, 64.0);
                    }
                    lastValues.set(i, value);
                    if (row.getValue(i).isNaN()) continue;
                    ((List)valuesMap.get(ts)).set(i, value);
                    continue;
                }
                if (row.getValue(i).isNaN()) continue;
                ((List)valuesMap.get(ts + step)).set(i, row.getValue(i));
            }
            ts -= step;
        }
        return valuesMap;
    }

    private static class ForeignId {
        private final String foreignSource;
        private final String foreignId;

        public ForeignId(String foreignSource, String foreignId) {
            this.foreignSource = foreignSource;
            this.foreignId = foreignId;
        }
    }

    private static enum StorageTool {
        RRDTOOL,
        JROBIN;

    }

    private static enum StorageStrategy {
        STORE_BY_METRIC,
        STORE_BY_GROUP;

    }
}

