/*
 * Decompiled with CFR 0.152.
 */
package org.opennms.core.ipc.sink.kafka.client;

import com.codahale.metrics.MetricRegistry;
import com.google.common.math.IntMath;
import com.google.protobuf.ByteString;
import io.opentracing.Span;
import io.opentracing.Tracer;
import io.opentracing.propagation.Format;
import io.opentracing.util.GlobalTracer;
import java.io.IOException;
import java.math.RoundingMode;
import java.util.Map;
import java.util.Properties;
import java.util.UUID;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.clients.producer.RecordMetadata;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.serialization.ByteArraySerializer;
import org.apache.kafka.common.serialization.StringSerializer;
import org.opennms.core.camel.JmsQueueNameFactory;
import org.opennms.core.ipc.common.kafka.OsgiKafkaConfigProvider;
import org.opennms.core.ipc.common.kafka.Utils;
import org.opennms.core.ipc.sink.api.Message;
import org.opennms.core.ipc.sink.api.SinkModule;
import org.opennms.core.ipc.sink.common.AbstractMessageDispatcherFactory;
import org.opennms.core.ipc.sink.model.SinkMessage;
import org.opennms.core.logging.Logging;
import org.opennms.core.tracing.api.TracerRegistry;
import org.opennms.core.tracing.util.TracingInfoCarrier;
import org.opennms.distributed.core.api.Identity;
import org.osgi.framework.BundleContext;
import org.osgi.service.cm.ConfigurationAdmin;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KafkaRemoteMessageDispatcherFactory
extends AbstractMessageDispatcherFactory<String> {
    private static final Logger LOG = LoggerFactory.getLogger(KafkaRemoteMessageDispatcherFactory.class);
    private static final int INVALID_PARTITION = -1;
    private final Properties kafkaConfig = new Properties();
    private ConfigurationAdmin configAdmin;
    private BundleContext bundleContext;
    private KafkaProducer<String, byte[]> producer;
    private TracerRegistry tracerRegistry;
    private MetricRegistry metrics;
    private Identity identity;
    private int maxBufferSize;

    public <S extends Message, T extends Message> String getModuleMetadata(SinkModule<S, T> module) {
        JmsQueueNameFactory topicNameFactory = new JmsQueueNameFactory("Sink", module.getId());
        return topicNameFactory.getName();
    }

    public <S extends Message, T extends Message> void dispatch(SinkModule<S, T> module, String topic, T message) {
        try (Logging.MDCCloseable mdc = Logging.withPrefixCloseable((String)"ipc");){
            LOG.trace("dispatch({}): sending message {}", (Object)topic, message);
            byte[] sinkMessageContent = module.marshal(message);
            String messageId = UUID.randomUUID().toString();
            String messageKey = module.getRoutingKey(message).orElse(messageId);
            boolean partitionChanged = false;
            while (partitionChanged = this.sendMessage(topic, messageId, messageKey, sinkMessageContent)) {
            }
        }
    }

    private boolean sendMessage(String topic, String messageId, String messageKey, byte[] sinkMessageContent) {
        int partitionNum = -1;
        boolean partitionChanged = false;
        int totalChunks = IntMath.divide((int)sinkMessageContent.length, (int)this.maxBufferSize, (RoundingMode)RoundingMode.UP);
        for (int chunk = 0; chunk < totalChunks; ++chunk) {
            byte[] messageInBytes = this.wrapMessageToProto(messageId, chunk, totalChunks, sinkMessageContent);
            ProducerRecord record = new ProducerRecord(topic, (Object)messageKey, (Object)messageInBytes);
            Span activeSpan = this.getTracer().activeSpan();
            if (activeSpan != null && chunk + 1 == totalChunks) {
                activeSpan.setTag("topic", topic);
                activeSpan.setTag("messageSize", (Number)sinkMessageContent.length);
                activeSpan.setTag("thread", Thread.currentThread().getName());
            }
            int partition = this.sendMessageChunkToKafka(topic, (ProducerRecord<String, byte[]>)record);
            if (totalChunks > 1 && chunk == 0) {
                partitionNum = partition;
                continue;
            }
            if (totalChunks <= 1 || partitionNum == partition) continue;
            partitionChanged = true;
            break;
        }
        return partitionChanged;
    }

    private int sendMessageChunkToKafka(String topic, ProducerRecord<String, byte[]> record) {
        while (true) {
            try {
                Future future = this.producer.send(record);
                RecordMetadata recordMetadata = (RecordMetadata)future.get();
                return recordMetadata.partition();
            }
            catch (InterruptedException e) {
                LOG.warn("Interrupted while sending message to topic {}.", (Object)topic, (Object)e);
                Thread.currentThread().interrupt();
            }
            catch (ExecutionException e) {
                if (e.getCause() != null && e.getCause() instanceof TimeoutException) {
                    LOG.warn("Timeout occured while sending message to topic {}, it will be attempted again.", (Object)topic);
                    continue;
                }
                LOG.error("Exception occured while sending message to topic {} ", (Throwable)e);
            }
            break;
        }
        return -1;
    }

    private byte[] wrapMessageToProto(String messageId, int chunk, int totalChunks, byte[] sinkMessageContent) {
        int bufferSize = this.getRemainingBufferSize(sinkMessageContent.length, chunk);
        ByteString byteString = ByteString.copyFrom((byte[])sinkMessageContent, (int)(chunk * this.maxBufferSize), (int)bufferSize);
        SinkMessage.Builder sinkMessageBuilder = SinkMessage.newBuilder().setMessageId(messageId).setCurrentChunkNumber(chunk).setTotalChunks(totalChunks).setContent(byteString);
        Tracer tracer = this.getTracer();
        if (tracer.activeSpan() != null && chunk + 1 == totalChunks) {
            TracingInfoCarrier tracingInfoCarrier = new TracingInfoCarrier();
            tracer.inject(tracer.activeSpan().context(), Format.Builtin.TEXT_MAP, (Object)tracingInfoCarrier);
            tracer.activeSpan().setTag("location", this.identity.getLocation());
            tracer.activeSpan().setTag("thread", Thread.currentThread().getName());
            tracingInfoCarrier.getTracingInfoMap().forEach((arg_0, arg_1) -> ((SinkMessage.Builder)sinkMessageBuilder).putTracingInfo(arg_0, arg_1));
        }
        return sinkMessageBuilder.build().toByteArray();
    }

    public void init() throws IOException {
        try (Logging.MDCCloseable mdc = Logging.withPrefixCloseable((String)"ipc");){
            this.kafkaConfig.clear();
            this.kafkaConfig.put("key.serializer", StringSerializer.class.getCanonicalName());
            this.kafkaConfig.put("value.serializer", ByteArraySerializer.class.getCanonicalName());
            OsgiKafkaConfigProvider configProvider = new OsgiKafkaConfigProvider("org.opennms.core.ipc.sink.kafka", this.configAdmin);
            this.kafkaConfig.putAll((Map<?, ?>)configProvider.getProperties());
            LOG.info("KafkaRemoteMessageDispatcherFactory: initializing the Kafka producer with: {}", (Object)this.kafkaConfig);
            this.producer = (KafkaProducer)Utils.runWithGivenClassLoader(() -> new KafkaProducer(this.kafkaConfig), (ClassLoader)KafkaProducer.class.getClassLoader());
            this.maxBufferSize = this.getMaxBufferSize();
            if (this.tracerRegistry != null && this.identity != null) {
                this.tracerRegistry.init(this.identity.getLocation() + "@" + this.identity.getId());
            }
            this.onInit();
        }
    }

    public void destroy() {
        this.onDestroy();
        if (this.producer != null) {
            this.producer.close();
            this.producer = null;
        }
    }

    public String getMetricDomain() {
        return "org.opennms.core.ipc.sink.producer";
    }

    public BundleContext getBundleContext() {
        return this.bundleContext;
    }

    public void setConfigAdmin(ConfigurationAdmin configAdmin) {
        this.configAdmin = configAdmin;
    }

    public void setBundleContext(BundleContext bundleContext) {
        this.bundleContext = bundleContext;
    }

    private int getRemainingBufferSize(int messageSize, int chunk) {
        int bufferSize = messageSize;
        if (messageSize > this.maxBufferSize) {
            int remaining = messageSize - chunk * this.maxBufferSize;
            bufferSize = remaining > this.maxBufferSize ? this.maxBufferSize : remaining;
        }
        return bufferSize;
    }

    public Integer getMaxBufferSize() {
        int maxBufferSize = 921600;
        String bufferSize = this.kafkaConfig.getProperty("max.buffer.size");
        if (bufferSize != null) {
            try {
                maxBufferSize = Integer.parseInt(bufferSize);
            }
            catch (NumberFormatException ex) {
                LOG.warn("Configured max buffer size is not a number");
            }
        }
        return Math.min(921600, maxBufferSize);
    }

    public void setTracerRegistry(TracerRegistry tracerRegistry) {
        this.tracerRegistry = tracerRegistry;
    }

    public TracerRegistry getTracerRegistry() {
        return this.tracerRegistry;
    }

    public Tracer getTracer() {
        if (this.getTracerRegistry() != null) {
            return this.getTracerRegistry().getTracer();
        }
        return GlobalTracer.get();
    }

    public Identity getIdentity() {
        return this.identity;
    }

    public void setIdentity(Identity identity) {
        this.identity = identity;
    }

    public MetricRegistry getMetrics() {
        if (this.metrics == null) {
            this.metrics = new MetricRegistry();
        }
        return this.metrics;
    }

    public void setMetrics(MetricRegistry metrics) {
        this.metrics = metrics;
    }
}

