/*
 * Decompiled with CFR 0.152.
 */
package org.opennms.core.ipc.grpc.client;

import com.codahale.metrics.MetricRegistry;
import com.google.common.base.Strings;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.protobuf.ByteString;
import io.grpc.Channel;
import io.grpc.ConnectivityState;
import io.grpc.ManagedChannel;
import io.grpc.netty.shaded.io.grpc.netty.GrpcSslContexts;
import io.grpc.netty.shaded.io.grpc.netty.NegotiationType;
import io.grpc.netty.shaded.io.grpc.netty.NettyChannelBuilder;
import io.grpc.netty.shaded.io.netty.handler.ssl.SslContextBuilder;
import io.grpc.stub.StreamObserver;
import io.opentracing.Span;
import io.opentracing.SpanContext;
import io.opentracing.Tracer;
import io.opentracing.propagation.Format;
import io.opentracing.propagation.TextMapExtractAdapter;
import io.opentracing.util.GlobalTracer;
import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import javax.net.ssl.SSLException;
import org.opennms.core.ipc.grpc.client.EmptyMessageReceiver;
import org.opennms.core.ipc.grpc.common.ConfigUtils;
import org.opennms.core.ipc.grpc.common.OpenNMSIpcGrpc;
import org.opennms.core.ipc.grpc.common.RpcRequestProto;
import org.opennms.core.ipc.grpc.common.RpcResponseProto;
import org.opennms.core.ipc.grpc.common.SinkMessage;
import org.opennms.core.ipc.sink.api.Message;
import org.opennms.core.ipc.sink.api.SinkModule;
import org.opennms.core.ipc.sink.common.AbstractMessageDispatcherFactory;
import org.opennms.core.logging.Logging;
import org.opennms.core.rpc.api.RpcModule;
import org.opennms.core.rpc.api.RpcRequest;
import org.opennms.core.rpc.api.RpcResponse;
import org.opennms.core.tracing.api.TracerRegistry;
import org.opennms.core.tracing.util.TracingInfoCarrier;
import org.opennms.core.utils.PropertiesUtils;
import org.opennms.distributed.core.api.MinionIdentity;
import org.osgi.framework.BundleContext;
import org.osgi.service.cm.ConfigurationAdmin;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MinionGrpcClient
extends AbstractMessageDispatcherFactory<String> {
    private static final Logger LOG = LoggerFactory.getLogger(MinionGrpcClient.class);
    private static final long SINK_BLOCKING_TIMEOUT = 3000L;
    private static final int SINK_BLOCKING_THREAD_POOL_SIZE = 100;
    private ManagedChannel channel;
    private OpenNMSIpcGrpc.OpenNMSIpcStub asyncStub;
    private Properties properties;
    private BundleContext bundleContext;
    private MinionIdentity minionIdentity;
    private ConfigurationAdmin configAdmin;
    private StreamObserver<RpcResponseProto> rpcStream;
    private StreamObserver<SinkMessage> sinkStream;
    private ConnectivityState currentChannelState;
    private MetricRegistry metrics;
    private TracerRegistry tracerRegistry;
    private final ThreadFactory requestHandlerThreadFactory = new ThreadFactoryBuilder().setNameFormat("rpc-request-handler-%d").build();
    private final ThreadFactory blockingSinkMessageThreadFactory = new ThreadFactoryBuilder().setNameFormat("blocking-sink-message-%d").build();
    private final ExecutorService requestHandlerExecutor = Executors.newCachedThreadPool(this.requestHandlerThreadFactory);
    private final Map<String, RpcModule<RpcRequest, RpcResponse>> registerdModules = new ConcurrentHashMap<String, RpcModule<RpcRequest, RpcResponse>>();
    private final ScheduledExecutorService blockingSinkMessageScheduler = Executors.newScheduledThreadPool(100, this.blockingSinkMessageThreadFactory);

    public MinionGrpcClient(MinionIdentity identity, ConfigurationAdmin configAdmin) {
        this.minionIdentity = identity;
        this.configAdmin = configAdmin;
    }

    public void start() throws IOException {
        this.properties = ConfigUtils.getPropertiesFromConfig((ConfigurationAdmin)this.configAdmin, (String)"org.opennms.core.ipc.grpc.client");
        String host = PropertiesUtils.getProperty((Properties)this.properties, (String)"host", (String)"localhost");
        int port = PropertiesUtils.getProperty((Properties)this.properties, (String)"port", (int)8990);
        boolean tlsEnabled = PropertiesUtils.getProperty((Properties)this.properties, (String)"tls.enabled", (boolean)false);
        int maxInboundMessageSize = PropertiesUtils.getProperty((Properties)this.properties, (String)"max.message.size", (int)0xA00000);
        NettyChannelBuilder channelBuilder = (NettyChannelBuilder)NettyChannelBuilder.forAddress((String)host, (int)port).keepAliveWithoutCalls(true).maxInboundMessageSize(maxInboundMessageSize);
        if (tlsEnabled) {
            this.channel = channelBuilder.negotiationType(NegotiationType.TLS).sslContext(this.buildSslContext().build()).build();
            LOG.info("TLS enabled for gRPC");
        } else {
            this.channel = channelBuilder.usePlaintext().build();
        }
        this.asyncStub = OpenNMSIpcGrpc.newStub((Channel)this.channel);
        this.initializeRpcStub();
        this.initializeSinkStub();
        if (this.tracerRegistry != null) {
            this.tracerRegistry.init(this.minionIdentity.getLocation() + "@" + this.minionIdentity.getId());
        }
        LOG.info("Minion at location {} with systemId {} started", (Object)this.minionIdentity.getLocation(), (Object)this.minionIdentity.getId());
    }

    private SslContextBuilder buildSslContext() throws SSLException {
        SslContextBuilder builder = GrpcSslContexts.forClient();
        String clientCertChainFilePath = this.properties.getProperty("client.cert.filepath");
        String clientPrivateKeyFilePath = this.properties.getProperty("client.private.key.filepath");
        String trustCertCollectionFilePath = this.properties.getProperty("trust.cert.filepath");
        if (trustCertCollectionFilePath != null) {
            builder.trustManager(new File(trustCertCollectionFilePath));
        }
        if (clientCertChainFilePath != null && clientPrivateKeyFilePath != null) {
            builder.keyManager(new File(clientCertChainFilePath), new File(clientPrivateKeyFilePath));
        }
        return builder;
    }

    private void initializeRpcStub() {
        if (this.getChannelState().equals((Object)ConnectivityState.READY)) {
            this.rpcStream = this.asyncStub.rpcStreaming((StreamObserver)new RpcMessageHandler());
            this.sendMinionHeaders();
            LOG.info("Initialized RPC stream");
        } else {
            LOG.warn("gRPC IPC server is not in ready state");
        }
    }

    private void initializeSinkStub() {
        if (this.getChannelState().equals((Object)ConnectivityState.READY)) {
            this.sinkStream = this.asyncStub.sinkStreaming((StreamObserver)new EmptyMessageReceiver());
            LOG.info("Initialized Sink stream");
        } else {
            LOG.warn("gRPC IPC server is not in ready state");
        }
    }

    public void bind(RpcModule module) throws Exception {
        if (module != null) {
            RpcModule rpcModule = module;
            if (this.registerdModules.containsKey(rpcModule.getId())) {
                LOG.warn(" {} module is already registered", (Object)rpcModule.getId());
            } else {
                this.registerdModules.put(rpcModule.getId(), (RpcModule<RpcRequest, RpcResponse>)rpcModule);
                LOG.info("Registered module {} with gRPC IPC client", (Object)rpcModule.getId());
            }
        }
    }

    public void unbind(RpcModule module) throws Exception {
        if (module != null) {
            RpcModule rpcModule = module;
            this.registerdModules.remove(rpcModule.getId());
            LOG.info("Removing module {} from gRPC IPC client.", (Object)rpcModule.getId());
        }
    }

    private boolean hasChangedToReadyState() {
        ConnectivityState prevState = this.currentChannelState;
        return !prevState.equals((Object)ConnectivityState.READY) && this.getChannelState().equals((Object)ConnectivityState.READY);
    }

    public void shutdown() {
        this.requestHandlerExecutor.shutdown();
        this.blockingSinkMessageScheduler.shutdown();
        this.registerdModules.clear();
        if (this.rpcStream != null) {
            this.rpcStream.onCompleted();
        }
        this.channel.shutdown();
        LOG.info("Minion at location {} with systemId {} stopped", (Object)this.minionIdentity.getLocation(), (Object)this.minionIdentity.getId());
    }

    public void setBundleContext(BundleContext bundleContext) {
        this.bundleContext = bundleContext;
    }

    public String getMetricDomain() {
        return "org.opennms.core.ipc.sink.producer";
    }

    public BundleContext getBundleContext() {
        return this.bundleContext;
    }

    public Tracer getTracer() {
        if (this.tracerRegistry != null) {
            return this.tracerRegistry.getTracer();
        }
        return GlobalTracer.get();
    }

    public MetricRegistry getMetrics() {
        if (this.metrics == null) {
            return new MetricRegistry();
        }
        return this.metrics;
    }

    public void setMetrics(MetricRegistry metrics) {
        this.metrics = metrics;
    }

    public TracerRegistry getTracerRegistry() {
        return this.tracerRegistry;
    }

    public void setTracerRegistry(TracerRegistry tracerRegistry) {
        this.tracerRegistry = tracerRegistry;
    }

    ConnectivityState getChannelState() {
        this.currentChannelState = this.channel.getState(true);
        return this.currentChannelState;
    }

    public <S extends Message, T extends Message> void dispatch(SinkModule<S, T> module, String metadata, T message) {
        try (Logging.MDCCloseable mdc = Logging.withPrefixCloseable((String)"ipc");){
            byte[] sinkMessageContent = module.marshal(message);
            String messageId = UUID.randomUUID().toString();
            SinkMessage.Builder sinkMessageBuilder = SinkMessage.newBuilder().setMessageId(messageId).setLocation(this.minionIdentity.getLocation()).setModuleId(module.getId()).setContent(ByteString.copyFrom((byte[])sinkMessageContent));
            if (module.getId().equals("Heartbeat") && (this.rpcStream == null || this.sinkStream == null || this.hasChangedToReadyState())) {
                this.initializeSinkStub();
                this.initializeRpcStub();
            }
            this.setTagsForSink(sinkMessageBuilder);
            if (module.getAsyncPolicy() != null) {
                this.sendBlockingSinkMessage(sinkMessageBuilder.build());
            } else {
                this.sendSinkMessage(sinkMessageBuilder.build());
            }
        }
    }

    private void sendBlockingSinkMessage(SinkMessage sinkMessage) {
        boolean succeeded = this.sendSinkMessage(sinkMessage);
        if (succeeded) {
            return;
        }
        this.scheduleSinkMessageAfterDelay(sinkMessage);
    }

    private boolean scheduleSinkMessageAfterDelay(SinkMessage sinkMessage) {
        ScheduledFuture<Boolean> future = this.blockingSinkMessageScheduler.schedule(() -> this.sendSinkMessage(sinkMessage), 3000L, TimeUnit.MILLISECONDS);
        try {
            boolean succeeded = (Boolean)future.get();
            if (succeeded) {
                return true;
            }
        }
        catch (InterruptedException | ExecutionException e) {
            LOG.error("Error while attempting to send sink message with id {} from module {} to gRPC IPC server", new Object[]{sinkMessage.getMessageId(), sinkMessage.getModuleId(), e});
        }
        return this.scheduleSinkMessageAfterDelay(sinkMessage);
    }

    private synchronized boolean sendSinkMessage(SinkMessage sinkMessage) {
        if (this.getChannelState().equals((Object)ConnectivityState.READY)) {
            if (this.sinkStream != null) {
                try {
                    this.sinkStream.onNext((Object)sinkMessage);
                    return true;
                }
                catch (Throwable e) {
                    LOG.error("Exception while sending sinkMessage to gRPC IPC server", e);
                }
            }
        } else {
            LOG.info("gRPC IPC server is not in ready state");
        }
        return false;
    }

    private void sendMinionHeaders() {
        RpcResponseProto rpcHeader = RpcResponseProto.newBuilder().setLocation(this.minionIdentity.getLocation()).setSystemId(this.minionIdentity.getId()).setModuleId("MINION_HEADERS").setRpcId(this.minionIdentity.getId()).build();
        this.sendRpcResponse(rpcHeader);
        LOG.info("Sending Minion Headers from SystemId {} to gRPC server", (Object)this.minionIdentity.getId());
    }

    private void processRpcRequest(RpcRequestProto requestProto) {
        long currentTime = requestProto.getExpirationTime();
        if (requestProto.getExpirationTime() < currentTime) {
            LOG.debug("ttl already expired for the request id = {}, won't process.", (Object)requestProto.getRpcId());
            return;
        }
        String moduleId = requestProto.getModuleId();
        if (Strings.isNullOrEmpty((String)moduleId)) {
            return;
        }
        LOG.debug("Received RPC request with RpcID:{} for module {}", (Object)requestProto.getRpcId(), (Object)requestProto.getModuleId());
        RpcModule<RpcRequest, RpcResponse> rpcModule = this.registerdModules.get(moduleId);
        if (rpcModule == null) {
            return;
        }
        Tracer.SpanBuilder spanBuilder = this.buildSpanFromRpcMessage(requestProto);
        Span minionSpan = spanBuilder.start();
        this.setTagsForRpc(requestProto, minionSpan);
        RpcRequest rpcRequest = rpcModule.unmarshalRequest(requestProto.getRpcContent().toStringUtf8());
        CompletableFuture future = rpcModule.execute(rpcRequest);
        future.whenComplete((res, ex) -> {
            RpcResponse rpcResponse;
            if (ex != null) {
                LOG.warn("An error occured while executing a call in {}.", (Object)rpcModule.getId(), ex);
                rpcResponse = rpcModule.createResponseWithException(ex);
                minionSpan.log(ex.getMessage());
                minionSpan.setTag("failed", Boolean.TRUE.toString());
            } else {
                rpcResponse = res;
            }
            minionSpan.finish();
            String responseAsString = rpcModule.marshalResponse(rpcResponse);
            RpcResponseProto responseProto = RpcResponseProto.newBuilder().setRpcId(requestProto.getRpcId()).setSystemId(this.minionIdentity.getId()).setLocation(requestProto.getLocation()).setModuleId(requestProto.getModuleId()).setRpcContent(ByteString.copyFrom((String)responseAsString, (Charset)StandardCharsets.UTF_8)).build();
            if (this.getChannelState().equals((Object)ConnectivityState.READY)) {
                try {
                    this.sendRpcResponse(responseProto);
                    LOG.debug("Request with RpcId:{} for module {} handled successfully, and response was sent", (Object)responseProto.getRpcId(), (Object)responseProto.getModuleId());
                }
                catch (Throwable e) {
                    LOG.error("Error while sending RPC response {}", (Object)responseProto, (Object)e);
                }
            } else {
                LOG.warn("gRPC IPC server is not in ready state");
            }
        });
    }

    private Tracer.SpanBuilder buildSpanFromRpcMessage(RpcRequestProto requestProto) {
        Tracer tracer = this.getTracer();
        HashMap tracingInfoMap = new HashMap();
        requestProto.getTracingInfoMap().forEach(tracingInfoMap::put);
        SpanContext context = tracer.extract(Format.Builtin.TEXT_MAP, (Object)new TextMapExtractAdapter(tracingInfoMap));
        Tracer.SpanBuilder spanBuilder = context != null ? tracer.buildSpan(requestProto.getModuleId()).asChildOf(context) : tracer.buildSpan(requestProto.getModuleId());
        return spanBuilder;
    }

    private void setTagsForRpc(RpcRequestProto requestProto, Span minionSpan) {
        requestProto.getTracingInfoMap().forEach((arg_0, arg_1) -> ((Span)minionSpan).setTag(arg_0, arg_1));
        minionSpan.setTag("location", requestProto.getLocation());
        if (!Strings.isNullOrEmpty((String)requestProto.getSystemId())) {
            minionSpan.setTag("systemId", requestProto.getSystemId());
        }
    }

    private void setTagsForSink(SinkMessage.Builder sinkMessageBuilder) {
        Tracer tracer = this.getTracer();
        if (tracer.activeSpan() != null) {
            TracingInfoCarrier tracingInfoCarrier = new TracingInfoCarrier();
            tracer.inject(tracer.activeSpan().context(), Format.Builtin.TEXT_MAP, (Object)tracingInfoCarrier);
            tracer.activeSpan().setTag("location", this.minionIdentity.getLocation());
            tracer.activeSpan().setTag("thread", Thread.currentThread().getName());
            tracingInfoCarrier.getTracingInfoMap().forEach((arg_0, arg_1) -> ((SinkMessage.Builder)sinkMessageBuilder).putTracingInfo(arg_0, arg_1));
        }
    }

    private synchronized void sendRpcResponse(RpcResponseProto rpcResponseProto) {
        if (this.rpcStream != null) {
            try {
                this.rpcStream.onNext((Object)rpcResponseProto);
            }
            catch (Exception e) {
                LOG.error("Exception while sending RPC response : {}", (Object)rpcResponseProto);
            }
        } else {
            throw new RuntimeException("RPC response handler not found");
        }
    }

    private class RpcMessageHandler
    implements StreamObserver<RpcRequestProto> {
        private RpcMessageHandler() {
        }

        public void onNext(RpcRequestProto rpcRequestProto) {
            try {
                MinionGrpcClient.this.requestHandlerExecutor.execute(() -> MinionGrpcClient.this.processRpcRequest(rpcRequestProto));
            }
            catch (Throwable e) {
                LOG.error("Error while processing the RPC Request {}", (Object)rpcRequestProto, (Object)e);
            }
        }

        public void onError(Throwable throwable) {
            LOG.error("Error in RPC streaming", throwable);
            MinionGrpcClient.this.rpcStream = null;
        }

        public void onCompleted() {
            LOG.error("Closing RPC message handler");
            MinionGrpcClient.this.rpcStream = null;
        }
    }
}

