/*
 * Decompiled with CFR 0.152.
 */
package org.opennms.karaf.extender;

import com.google.common.collect.Lists;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Dictionary;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.karaf.features.FeaturesService;
import org.apache.karaf.kar.KarService;
import org.opennms.karaf.extender.Feature;
import org.opennms.karaf.extender.KarDependencyHandler;
import org.opennms.karaf.extender.Repository;
import org.ops4j.pax.url.mvn.MavenResolver;
import org.osgi.service.cm.Configuration;
import org.osgi.service.cm.ConfigurationAdmin;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KarafExtender {
    private static final Logger LOG = LoggerFactory.getLogger(KarafExtender.class);
    private static final String PAX_MVN_PID = "org.ops4j.pax.url.mvn";
    private static final String PAX_MVN_REPOSITORIES = "org.ops4j.pax.url.mvn.repositories";
    public static final String WAIT_FOR_KAR_ATTRIBUTE = "wait-for-kar";
    public static final String FEATURES_URI = "features.uri";
    public static final String FEATURES_BOOT = "features.boot";
    private static final String COMMENT_REGEX = "^\\s*(#.*)?$";
    private static final Pattern FEATURE_VERSION_PATTERN = Pattern.compile("(?<name>.+?)(/(?<version>.*))?(\\s+(?<attributes>.*))?");
    private final Path m_karafHome = Paths.get(System.getProperty("karaf.home"), new String[0]);
    private final Path m_repositories = this.m_karafHome.resolve("repositories");
    private final Path m_featuresBootDotD = this.m_karafHome.resolve(Paths.get("etc", "featuresBoot.d"));
    private ConfigurationAdmin m_configurationAdmin;
    private MavenResolver m_mavenResolver;
    private FeaturesService m_featuresService;
    private KarService m_karService;
    private Thread m_installThread;
    private Thread m_karDependencyInstallThread;

    public void init() throws InterruptedException {
        List<Repository> repositories;
        Objects.requireNonNull(this.m_configurationAdmin, "configurationAdmin");
        Objects.requireNonNull(this.m_mavenResolver, "mavenResolver");
        Objects.requireNonNull(this.m_featuresService, "featuresService");
        Objects.requireNonNull(this.m_karService, "karService");
        try {
            repositories = this.getRepositories();
        }
        catch (IOException e) {
            LOG.error("Failed to retrieve the list of repositories. Aborting.", (Throwable)e);
            return;
        }
        List<Feature> featuresBoot = repositories.stream().flatMap(r -> r.getFeaturesBoot().stream()).collect(Collectors.toList());
        try {
            featuresBoot.addAll(this.getFeaturesBoot());
        }
        catch (IOException e) {
            LOG.error("Failed to retrieve the list of features to boot. Aborting.", (Throwable)e);
            return;
        }
        this.filterFeatures(featuresBoot);
        if (!repositories.isEmpty()) {
            StringBuilder mavenReposSb = new StringBuilder();
            for (Repository repository : repositories) {
                if (mavenReposSb.length() != 0) {
                    mavenReposSb.append(",");
                }
                mavenReposSb.append(repository.toMavenUri());
            }
            String mavenRepos = mavenReposSb.toString();
            LOG.info("Updating Maven repositories to include: {}", (Object)mavenRepos);
            try {
                Configuration config = this.m_configurationAdmin.getConfiguration(PAX_MVN_PID);
                if (config == null) {
                    throw new IOException("The OSGi configuration (admin) registry was found for pid org.ops4j.pax.url.mvn, but a configuration could not be located/generated.  This shouldn't happen.");
                }
                Dictionary props = config.getProperties();
                if (!mavenRepos.equals(props.get(PAX_MVN_REPOSITORIES))) {
                    props.put(PAX_MVN_REPOSITORIES, mavenRepos);
                    config.update(props);
                }
            }
            catch (IOException e) {
                LOG.error("Failed to update the list of Maven repositories to '{}'. Aborting.", (Object)mavenRepos, (Object)e);
                return;
            }
            LOG.info("Waiting up-to 30 seconds for the Maven repositories to be updated...");
            Thread.sleep(2000L);
            for (int i = 28; i > 0 && !this.canResolveAllFeatureUris(repositories); --i) {
                Thread.sleep(1000L);
            }
            for (Repository repository : repositories) {
                for (URI featureUri : repository.getFeatureUris()) {
                    try {
                        LOG.info("Adding feature repository: {}", (Object)featureUri);
                        this.m_featuresService.addRepository(featureUri);
                    }
                    catch (Exception e) {
                        LOG.error("Failed to add feature repository '{}'. Skipping.", (Object)featureUri, (Object)e);
                    }
                }
            }
        } else {
            LOG.debug("No repositories to install.");
        }
        Set featuresToInstall = featuresBoot.stream().filter(f -> f.getKarDependency() == null).map(Feature::toInstallString).collect(Collectors.toCollection(LinkedHashSet::new));
        if (!featuresToInstall.isEmpty()) {
            this.m_installThread = new Thread(() -> {
                try {
                    LOG.info("Installing features: {}", (Object)featuresToInstall);
                    this.m_featuresService.installFeatures(featuresToInstall, EnumSet.noneOf(FeaturesService.Option.class));
                }
                catch (Exception e) {
                    LOG.error("Failed to install one or more features.", (Throwable)e);
                }
            });
            this.m_installThread.setName("Karaf-Extender-Feature-Install");
            this.m_installThread.start();
        } else {
            LOG.debug("No features to install.");
        }
        List<Feature> featuresWithKarDependencies = featuresBoot.stream().filter(f -> f.getKarDependency() != null).collect(Collectors.toList());
        if (!featuresWithKarDependencies.isEmpty()) {
            KarDependencyHandler karDependencyHandler = new KarDependencyHandler(featuresWithKarDependencies, this.m_karService, this.m_featuresService);
            this.m_karDependencyInstallThread = new Thread(karDependencyHandler);
            this.m_karDependencyInstallThread.setName("Karaf-Extender-Feature-Install-For-Kars");
            this.m_karDependencyInstallThread.start();
        } else {
            LOG.debug("No features with dependencies on .kar files to install.");
        }
    }

    public void destroy() {
        if (this.m_installThread != null) {
            this.m_installThread.interrupt();
        }
        if (this.m_karDependencyInstallThread != null) {
            this.m_karDependencyInstallThread.interrupt();
        }
    }

    private boolean canResolveAllFeatureUris(List<Repository> repositories) {
        for (Repository repository : repositories) {
            for (URI featureUri : repository.getFeatureUris()) {
                try {
                    if (this.m_mavenResolver.resolve(featureUri.toString()) != null) continue;
                    return false;
                }
                catch (Exception e) {
                    return false;
                }
            }
        }
        return true;
    }

    public List<Repository> getRepositories() throws IOException {
        List<Path> repositoryPaths = KarafExtender.getRepositoryFolders(this.m_repositories);
        LinkedList repositories = Lists.newLinkedList();
        for (Path repositoryPath : repositoryPaths) {
            try {
                Path featuresBootPath;
                LinkedList featureUris = Lists.newLinkedList();
                Path featuresUriPath = repositoryPath.resolve(FEATURES_URI);
                if (featuresUriPath.toFile().isFile()) {
                    for (String line : KarafExtender.getLinesIn(featuresUriPath)) {
                        featureUris.add(new URI(line));
                    }
                }
                List<Feature> featuresBoot = (featuresBootPath = repositoryPath.resolve(FEATURES_BOOT)).toFile().isFile() ? this.getFeaturesIn(featuresBootPath) : Collections.emptyList();
                repositories.add(new Repository(repositoryPath, featureUris, featuresBoot));
            }
            catch (URISyntaxException e) {
                LOG.error("Failed to generate one or more feature URIs for repository {}. Skipping.", (Object)repositoryPath, (Object)e);
            }
        }
        return repositories;
    }

    private static Map<String, String> parseAttributes(String attributes) {
        LinkedHashMap<String, String> attributeMap = new LinkedHashMap<String, String>();
        if (attributes == null) {
            return attributeMap;
        }
        for (String attributeKvp : attributes.split("\\s")) {
            String[] tokens = attributeKvp.split("=");
            if (tokens.length != 2) continue;
            attributeMap.put(tokens[0].trim(), tokens[1].trim());
        }
        return attributeMap;
    }

    public List<Feature> getFeaturesIn(Path featuresBootFile) throws IOException {
        LinkedList features = Lists.newLinkedList();
        for (String line : KarafExtender.getLinesIn(featuresBootFile)) {
            Matcher m = FEATURE_VERSION_PATTERN.matcher(line);
            if (!m.matches()) continue;
            Map<String, String> attributes = KarafExtender.parseAttributes(m.group("attributes"));
            Feature feature = Feature.builder().withName(m.group("name")).withVersion(m.group("version")).withKarDependency(attributes.get(WAIT_FOR_KAR_ATTRIBUTE)).build();
            features.add(feature);
        }
        return features;
    }

    public List<Feature> getFeaturesBoot() throws IOException {
        LinkedList features = Lists.newLinkedList();
        for (Path featuresBootFile : KarafExtender.getFilesIn(this.m_featuresBootDotD)) {
            features.addAll(this.getFeaturesIn(featuresBootFile));
        }
        return features;
    }

    public void filterFeatures(List<Feature> features) {
        HashSet<Feature> featuresToExclude = new HashSet<Feature>();
        Iterator<Feature> it = features.iterator();
        while (it.hasNext()) {
            Feature feature = it.next();
            if (!feature.getName().startsWith("!") || feature.getName().length() <= 1) continue;
            featuresToExclude.add(Feature.builder().withName(feature.getName().substring(1)).withVersion(feature.getVersion()).build());
            it.remove();
        }
        features.removeAll(featuresToExclude);
    }

    private static List<Path> getFilesIn(Path folder) throws IOException {
        ArrayList<Path> files = new ArrayList<Path>();
        try (DirectoryStream<Path> directoryStream = Files.newDirectoryStream(folder);){
            for (Path path : directoryStream) {
                if (path.toFile().isDirectory() || path.getFileName().toString().startsWith(".")) continue;
                files.add(path);
            }
        }
        Collections.sort(files);
        return files;
    }

    private static List<Path> getRepositoryFolders(Path folder) throws IOException {
        LinkedList paths = Lists.newLinkedList();
        if (!folder.toFile().exists()) {
            LOG.info("Repository folder {} does not exist. No repositories will be added.", (Object)folder);
            return paths;
        }
        try (DirectoryStream<Path> directoryStream = Files.newDirectoryStream(folder);){
            for (Path path : directoryStream) {
                if (!path.toFile().isDirectory() || path.getFileName().toString().startsWith(".")) continue;
                paths.add(path);
            }
        }
        Collections.sort(paths);
        return paths;
    }

    private static List<String> getLinesIn(Path file) throws IOException {
        return Files.readAllLines(file).stream().filter(l -> !l.matches(COMMENT_REGEX)).collect(Collectors.toList());
    }

    public void setConfigurationAdmin(ConfigurationAdmin configurationAdmin) {
        this.m_configurationAdmin = configurationAdmin;
    }

    public void setMavenResolver(MavenResolver mavenResolver) {
        this.m_mavenResolver = mavenResolver;
    }

    public void setFeaturesService(FeaturesService featuresService) {
        this.m_featuresService = featuresService;
    }

    public void setKarService(KarService karService) {
        this.m_karService = karService;
    }
}

