/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.google.gwt.maps.utility.client.snaptoroute;

import com.google.gwt.core.client.JavaScriptObject;
import com.google.gwt.maps.client.MapWidget;
import com.google.gwt.maps.client.geom.LatLng;
import com.google.gwt.maps.client.overlay.Marker;
import com.google.gwt.maps.client.overlay.Polyline;

/**
 * Snaps a marker to the closest point on a line, based on the current
 * position of the cursor.
 */
public class SnapToRoute extends JavaScriptObject {

  /**
   * Constructs a new SnapToRoute that will snap the specified marker to the
   * specified route.
   * 
   * @param map The map widget.
   * @param marker The marker to display.
   * @param polyline The polyline defining the route to snap to.
   * @return A SnapToRoute object.
   */
  public static native SnapToRoute newInstance(MapWidget map, Marker marker,
      Polyline polyline) /*-{
    return new $wnd.SnapToRoute(
      map.@com.google.gwt.maps.client.MapWidget::getPeer()(),
      @com.google.gwt.maps.utility.client.impl.MapsHelper::getOverlayPeer(Lcom/google/gwt/maps/client/overlay/Overlay;)(marker),
      @com.google.gwt.maps.utility.client.impl.MapsHelper::getOverlayPeer(Lcom/google/gwt/maps/client/overlay/Overlay;)(polyline)
    );
  }-*/;

  protected SnapToRoute() { }

  /**
   * Identifies the point on the polyline that is closest to a given point.
   * 
   * @param latlng The starting point.
   * @return The point on the polyline that is closest to the specified point.
   */
  public final native LatLng getClosestLatLng(LatLng latlng) /*-{
    return this.getClosestLatLng(latlng);
  }-*/;
  
  /**
   * Retrieves the distance in meters, along the polyline, of the closest
   * point on route to the given point.
   * 
   * @param latlng The starting point.
   * @return The distance, in meters, to the given point, along the polyline.
   */
  public final native double getDistAlongRoute(LatLng latlng) /*-{
    return this.getDistAlongRoute(latlng);
  }-*/;
  
  /**
   * Retrieves the distance in meters, along the polyline, of the closest
   * point on route to the given point.
   * 
   * @return The distance, in meters, to the given point, along the polyline.
   */
  public final native double getDistAlongRoute() /*-{
    return this.getDistAlongRoute();
  }-*/;
  
  /**
   * Updates the marker and/or polyline used by this control.
   * 
   * @param marker The marker to display.
   * @param polyline The polyline defining the route to snap to.
   */
  public final native void updateTargets(Marker marker, Polyline polyline) /*-{
    this.updateTargets(
      @com.google.gwt.maps.utility.client.impl.MapsHelper::getOverlayPeer(Lcom/google/gwt/maps/client/overlay/Overlay;)(marker),
      @com.google.gwt.maps.utility.client.impl.MapsHelper::getOverlayPeer(Lcom/google/gwt/maps/client/overlay/Overlay;)(polyline)
    );
  }-*/;
  
}