/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.google.gwt.maps.utility.client.snapshotcontrol;

import com.google.gwt.core.client.JavaScriptObject;
import com.google.gwt.maps.client.overlay.Polygon;

/**
 * Defines snapshot style attributes for Polygon.
 */
public class SnapShotPolygonStyle extends JavaScriptObject {

  /**
   * Constructs a Polygon snapshot style.
   * 
   * @return A SnapShotPolygonStyle object.
   */
  public native static SnapShotPolygonStyle newInstance() /*-{
    return {};
  }-*/;
  
  /**
   * Constructs a Polygon snapshot style.
   * 
   * @param fillColor The style's fill color.
   * @param fillOpacity The style's fill opacity.
   * @param lineColor The style's line color.
   * @param lineOpacity The style's line opacity.
   * @param lineWeight The style's line weight.
   * @return A SnapShotPolygonStyle object.
   */
  public native static SnapShotPolygonStyle newInstance(String fillColor,
      double fillOpacity, String lineColor, double lineOpacity,
      double lineWeight) /*-{
    return {
      ssFillColor: fillColor,
      ssFillOpacity: fillOpacity,
      ssColor: lineColor,
      ssOpacity: lineOpacity,
      ssWeight: lineWeight
    };
  }-*/;
  
  protected SnapShotPolygonStyle() { }
  
  /**
   * Applies the style to the specified Polygon.
   * 
   * @param polygon The Polygon to which to apply the style.
   */
  public final native void applyTo(Polygon polygon) /*-{
    var p = @com.google.gwt.maps.utility.client.impl.MapsHelper::getOverlayPeer(Lcom/google/gwt/maps/client/overlay/Overlay;)(polygon);
    if (this.ssFillColor !== undefined) p.ssFillColor = this.ssFillColor;
    if (this.ssFillOpacity !== undefined) p.ssFillOpacity = this.ssFillOpacity;
    if (this.ssColor !== undefined) p.ssColor = this.ssColor;
    if (this.ssOpacity !== undefined) p.ssOpacity = this.ssOpacity;
    if (this.ssWeight !== undefined) p.ssWeight = this.ssWeight;
  }-*/;
  
  /**
   * Retrieves the style's fill color.
   * 
   * @return The style's fill color.
   */
  public final native String getFillColor() /*-{
    return this.ssFillColor;
  }-*/;
  
  /**
   * Retrieves the style's fill opacity.
   * 
   * @return The style's fill opacity.
   */
  public final native double getFillOpacity() /*-{
    return this.ssFillOpacity || 0;
  }-*/;
  
  /**
   * Retrieves the style's line color.
   * 
   * @return The style's line color.
   */
  public final native String getLineColor() /*-{
    return this.ssColor;
  }-*/;
  
  /**
   * Retrieves the style's line opacity.
   * 
   * @return The style's line opacity.
   */
  public final native double getLineOpacity() /*-{
    return this.ssOpacity || 0;
  }-*/;
  
  /**
   * Retrieves the style's line weight.
   * 
   * @return The style's line weight.
   */
  public final native double getLineWeight() /*-{
    return this.ssWeight || 0;
  }-*/;
  
  /**
   * Sets the style's fill color.
   * 
   * @param fillColor The style's fill color.
   */
  public final native void setFillColor(String fillColor) /*-{
    this.ssFillColor = fillColor;
  }-*/;
  
  /**
   * Sets the style's fill opacity.
   * 
   * @param fillOpacity The style's fill opacity.
   */
  public final native void setFillOpacity(double fillOpacity) /*-{
    this.ssFillOpacity = fillOpacity;
  }-*/;
  
  /**
   * Sets the style's line color.
   * 
   * @param lineColor The style's line color.
   */
  public final native void setLineColor(String lineColor) /*-{
    this.ssColor = lineColor;
  }-*/;
  
  /**
   * Sets the style's line opacity.
   * 
   * @param lineOpacity The style's line opacity.
   */
  public final native void setLineOpacity(double lineOpacity) /*-{
    this.ssOpacity = lineOpacity;
  }-*/;
  
  /**
   * Sets the style's line weight.
   * 
   * @param lineWeight The style's line weight.
   */
  public final native void setLineWeight(double lineWeight) /*-{
    this.ssWeight = lineWeight;
  }-*/;
}
