/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.google.gwt.maps.utility.client.snapshotcontrol;

import com.google.gwt.maps.client.control.Control;
import com.google.gwt.maps.client.geom.LatLng;
import com.google.gwt.maps.client.geom.Size;
import com.google.gwt.maps.utility.client.impl.SnapShotControlImpl;

/**
 * Provides capabilities for generating an image "snapshot" of a given map.
 */
public class SnapShotControl extends Control {

  /**
   * Creates a SnapShotControl.
   */
  public SnapShotControl() {
    super(SnapShotControlImpl.newInstance());
  }

  /**
   * Creates a SnapShotControl with the specified options.
   * 
   * @param options The snapshot control options.
   */
  public SnapShotControl(SnapShotControlOptions options) {
    super(SnapShotControlImpl.newInstance(options));
  }

  /**
   * Generates a new snapshot URL using the current map center.
   * 
   * @return The new snapshot image URL.
   */
  public String getImage() {
    return ((SnapShotControlImpl)this.jsoPeer).getImage();
  }
  
  /**
   * Generates a new snapshot URL using the specified map center.
   * If the specified map center is null, the center of the snapshot is
   * auto-calculated based on the overlay positions.
   * 
   * @param mapCenterPos The map center to use in the snapshot.
   * @return The new snapshot image URL.
   */
  public String getImage(LatLng mapCenterPos) {
    return ((SnapShotControlImpl)this.jsoPeer).getImage(mapCenterPos);
  }
  
  /**
   * Retrieves the most recently generated image URL.
   * 
   * @return The most recently generated image URL.
   */
  public String getImageUrl() {
    return ((SnapShotControlImpl)this.jsoPeer).getImageUrl();
  }
  
  /**
   * Sets the visibility of the control to hidden.
   */
  public void hide() {
    ((SnapShotControlImpl)this.jsoPeer).hide();
  }
  
  /**
   * Retrieves whether the control is hidden.
   * 
   * @return Whether the control is hidden.
   */
  public boolean isHidden() {
    return ((SnapShotControlImpl)this.jsoPeer).isHidden();
  }
  
  /**
   * Sets the image format.
   * The options are "gif", "jpg", "jpg-baseline", "png8", "png32".
   * 
   * @param format The image format.
   */
  public void setFormat(String format) {
    ((SnapShotControlImpl)this.jsoPeer).setFormat(format);
  }
  
  /**
   * Sets the language for the snapshot's copyright text.
   * 
   * @param language The language for the snapshot's copyright text.
   */
  public void setLanguage(String language) {
    ((SnapShotControlImpl)this.jsoPeer).setLanguage(language);
  }
  
  /**
   * Sets the map size for the snapshot.
   * 
   * @param mapSize The map size for the snapshot.
   */
  public void setMapSize(Size mapSize) {
    ((SnapShotControlImpl)this.jsoPeer).setMapSize(mapSize);
  }
  
  /**
   * Sets the map type for the snapshot.
   * The options are "roadmap", "satellite", "hybrid", "terrain".
   * If specified type is an empty string, the map type will be
   * auto-computed.
   * 
   * @param mapType The map type for the snapshot.
   */
  public void setMapType(String mapType) {
    ((SnapShotControlImpl)this.jsoPeer).setMapType(mapType);
  }
  
  /**
   * Sets whether to use mobile-optimized tiles.
   * 
   * @param mobile Whether to use mobile-optimized tiles.
   */
  public void setMobile(boolean mobile) {
    ((SnapShotControlImpl)this.jsoPeer).setMobile(mobile);
  }

  /**
   * Sets the visibility of the control to visible.
   */
  public void show() {
    ((SnapShotControlImpl)this.jsoPeer).show();
  }
  
  /**
   * Generates a new snapshot URL, at the specified map center, and displays a
   * popup containing the image, along with the URL.
   */
  public void showPopup() {
    ((SnapShotControlImpl)this.jsoPeer).showPopup();
  }
  
  /**
   * Generates a new snapshot URL, at the specified map center, and displays a
   * popup containing the image, along with the URL.
   * 
   * @param mapCenterPos The map center to use for the snapshot.
   */
  public void showPopup(LatLng mapCenterPos) {
    ((SnapShotControlImpl)this.jsoPeer).showPopup(mapCenterPos);
  }
  
  /**
   * Sets whether to use polyline encoding.
   * 
   * @param useEncode Whether to use polyline encoding.
   */
  public void setUsePolylineEncode(boolean useEncode) {
    ((SnapShotControlImpl)this.jsoPeer).setUsePolylineEncode(useEncode);
  }
  
}