/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.google.gwt.maps.utility.client.popupmarker;

import com.google.gwt.maps.client.overlay.Icon;
import com.google.gwt.maps.client.overlay.MarkerOptions;

/**
 * Defines optional arguments to PopupMarker.
 */
public class PopupMarkerOptions extends MarkerOptions {

  /**
   * Constructs a PopupMarker options object.
   * 
   * @return A PopupMarkerOptions object.
   */
  public static native PopupMarkerOptions newInstance() /*-{
    return {
      bgColor: "CCCCFF",
      style: "normal",
      text: "",
      textColor: "000000"
    };
  }-*/;
  
  /**
   * Constructs a PopupMarker options object.
   * 
   * @param icon The marker icon.
   * @return A PopupMarkerOptions object.
   */
  public static PopupMarkerOptions newInstance(Icon icon) {
    PopupMarkerOptions opts = PopupMarkerOptions.newInstance();
    opts.setIcon(icon);
    return opts;
  }

  /**
   * Constructs a PopupMarker options object.
   * 
   * @param bgColor The background color.
   * @param chart The chart options.
   * @param style The popup style.
   * @param text The popup text.
   * @param textColor The popup text color.
   * @return A PopupMarkerOptions object.
   */
  public static native PopupMarkerOptions newInstance(String bgColor,
      PopupMarkerChartApiOptions chart, String style, String text,
      String textColor) /*-{
    return {
      bgColor: bgColor,
      chart: chart,
      style: style,
      text: text,
      textColor: textColor
    };
    
  }-*/;
  
  protected PopupMarkerOptions() { }
  
  /**
   * Retrieves the background color as a 6-digit hexadecimal number.
   * If the style property is not set to "chart", then this property is
   * ignored. The default value is "CCCCFF".
   * 
   * @return The background color.
   */
  public final native String getBgColor() /*-{
    return this.bgColor;
  }-*/;
  
  /**
   * Retrieves the customization options for the Charts API output.
   * If the style property is not set to "chart", then this property is
   * ignored.
   * 
   * @return The chart options.
   */
  public final native PopupMarkerChartApiOptions getChart() /*-{
    return this.chart;
  }-*/;
  
  /**
   * Retrieves the style of popup. If set to "chart", then this library shows
   * the popup using the output of the Google Chart API, and additional options
   * should be specified in the chart property.
   * The default value is "normal".
   * 
   * @return The popup style.
   */
  public final native String getStyle() /*-{
    return this.style;
  }-*/;

  /**
   * Retrieves the text shown in the popup. If the style property is set to
   * "chart", then this string must follow the format for text in the Google
   * Charts API. The default value is a blank string.
   * 
   * @return The popup text.
   */
  public final native String getText() /*-{
    return this.text;
  }-*/;
  
  /**
   * Retrieves the text color as a 6-digit hexadecimal number.
   * If the style property is not set to "chart", then this property is
   * ignored. The default value is "000000".
   * 
   * @return The popup text color.
   */
  public final native String getTextColor() /*-{
    return this.textColor;
  }-*/;
  
  /**
   * Sets the background color as a 6-digit hexadecimal number.
   * If the style property is not set to "chart", then this property is
   * ignored. The default value is "CCCCFF".
   * 
   * @param bgColor The background color.
   */
  public final native void setBgColor(String bgColor) /*-{
    this.bgColor = bgColor;
  }-*/;
  
  /**
   * Sets the customization options for the Charts API
   * output. If the style property is not set to "chart", then this property is
   * ignored.
   * 
   * @param chart The chart options.
   */
  public final native void setChart(PopupMarkerChartApiOptions chart) /*-{
    this.chart = chart;
  }-*/;
  
  /**
   * Sets the style of popup. If set to "chart", then this library shows
   * the popup using the output of the Google Chart API, and additional options
   * should be specified in the chart property.
   * The default value is  "normal".
   * 
   * @param style The popup style.
   */
  public final native void setStyle(String style) /*-{
    this.style = style;
  }-*/;

  /**
   * Sets the text shown in the popup. If the style property is set to
   * "chart", then this string must follow the format for text in the Google
   * Charts API. The default value is a blank string.
   * 
   * @param text The popup text.
   */
  public final native void setText(String text) /*-{
    this.text = text;
  }-*/;
  
  /**
   * Sets the text color as a 6-digit hexadecimal number.
   * If the style property is not set to "chart", then this property is
   * ignored. The default value is "000000".
   * 
   * @param textColor The popup text color.
   */
  public final native void setTextColor(String textColor) /*-{
    this.textColor = textColor;
  }-*/;

}