/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.google.gwt.maps.utility.client.markertracker;

import com.google.gwt.core.client.JavaScriptObject;

/**
 * Defines optional arguments to MarkerTracker.
 */
public class MarkerTrackerOptions extends JavaScriptObject {

  /**
   * Constructs a MarkerTracker options object.
   * 
   * @return A MarkerTrackerOptions object.
   */
  public static native MarkerTrackerOptions newInstance() /*-{
    return {
      iconScale: 0.6,
      padding: 25,
      color: '#FF0000',
      weight: 20,
      length: 20,
      opacity: 0.8,
      updateEvent: 'move',
      panEvent: 'click',
      quickPanEnabled: true
    }
  }-*/;
  
  protected MarkerTrackerOptions() { }

  /**
   * Retrieves the icon size scale factor, 0 = no icon.
   * The default value is 0.6.
   * 
   * @return The icon size scale factor.
   */
  public final native double getIconScale() /*-{
    return this.iconScale || 0;
  }-*/;
  
  /**
   * Retrieves the number of pixels between the arrow and the edge of the map.
   * The default value is 25.
   * 
   * @return The number of pixels between the arrow and the edge of the map.
   */
  public final native int getPadding() /*-{
    return this.padding || 0;
  }-*/;
  
  /**
   * Retrieves the color of the arrow. The default value is '#ff0000'.
   * 
   * @return The color of the arrow.
   */
  public final native String getColor() /*-{
    return this.color;
  }-*/;
  
  /**
   * Retrieves the thickness of the lines that make up the arrows.
   * The default value is 20.
   * 
   * @return The thickness of the lines that make up the arrows.
   */
  public final native int getWeight() /*-{
    return this.weight || 0;
  }-*/;
  
  /**
   * Retrieves the length of the arrow. The default value is 20.
   * 
   * @return The length of the arrow.
   */
  public final native int getLength() /*-{
    return this.length || 0;
  }-*/;
  
  /**
   * Retrieves the opacity of the arrow. The default value is 0.8.
   * 
   * @return The opacity of the arrow.
   */
  public final native double getOpacity() /*-{
    return this.opacity || 0;
  }-*/;
  
  /**
   * Retrieves the name of the Map event that triggers the arrows to update,
   * ( i.e. using 'moveend' reduces processing when using many MarkerTrackers).
   * The default value is 'move'.
   * 
   * @return The name of the Map event that triggers the arrows to update.
   */
  public final native String getUpdateEvent() /*-{
    return this.updateEvent;
  }-*/;
  
  /**
   * Retrieves the name of the Marker event that triggers a quick pan to the
   * tracked marker. The default value is 'click'.
   * 
   * @return The name of the Marker event that triggers a quick pan.
   */
  public final native String getPanEvent() /*-{
    return this.panEvent;
  }-*/;
  
  /**
   * Retrieves whether quick pan is enabled.
   * 
   * @return Whether quick pan is enabled.
   */
  public final native boolean getQuickPanEnabled() /*-{
    return this.quickPanEnabled;
  }-*/;
  
  /**
   * Sets the icon size scale factor, 0 = no icon.
   * The default value is 0.6.
   * 
   * @param iconScale The icon size scale factor.
   */
  public final native void setIconScale(double iconScale) /*-{
    this.iconScale = iconScale;
  }-*/;
  
  /**
   * Sets the number of pixels between the arrow and the edge of the map.
   * The default value is 25.
   * 
   * @param padding The number of pixels between the arrow and the edge
   * of the map.
   */
  public final native void setPadding(int padding) /*-{
    this.padding = padding;
  }-*/;
  
  /**
   * Sets the color of the arrow. The default value is '#ff0000'.
   * 
   * @param color The color of the arrow.
   */
  public final native void setColor(String color) /*-{
    this.color = color;
  }-*/;
  
  /**
   * Sets the thickness of the lines that make up the arrows.
   * The default value is 20.
   * 
   * @param weight The thickness of the lines that make up the arrows.
   */
  public final native void setWeight(int weight) /*-{
    this.weight = weight;
  }-*/;
  
  /**
   * Sets the length of the arrow. The default value is 20.
   * 
   * @param length The length of the arrow.
   */
  public final native void setLength(int length) /*-{
    this.length = length;
  }-*/;
  
  /**
   * Sets the opacity of the arrow. The default value is 0.8.
   * 
   * @param opacity The opacity of the arrow.
   */
  public final native void setOpacity(double opacity) /*-{
    this.opacity = opacity;
  }-*/;
  
  /**
   * Sets the name of the Map event that triggers the arrows to update,
   * ( i.e. using 'moveend' reduces processing when using many MarkerTrackers).
   * The default value is 'move'.
   * 
   * @param updateEvent The name of the Map event that triggers the arrows to update.
   */
  public final native void setUpdateEvent(String updateEvent) /*-{
    this.updateEvent = updateEvent;
  }-*/;
  
  /**
   * Sets the name of the Marker event that triggers a quick pan to the
   * tracked marker. The default value is 'click'.
   * 
   * @param panEvent The name of the Marker event that triggers a quick pan.
   */
  public final native void setPanEvent(String panEvent) /*-{
    this.panEvent = panEvent;
  }-*/;
  
  /**
   * Sets whether quick pan is enabled.
   * 
   * @param quickPanEnabled Whether quick pan is enabled.
   */
  public final native void setQuickPanEnabled(boolean quickPanEnabled) /*-{
    this.quickPanEnabled = quickPanEnabled;
  }-*/;
  
}