/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.google.gwt.maps.utility.client.markermanager;

import com.google.gwt.core.client.JavaScriptObject;

/**
 * Defines optional arguments to MarkerManager.
 */
public class MarkerManagerOptions extends JavaScriptObject {

  /**
   * Constructs a MarkerManager options object.
   * 
   * @return A MarkerManagerOptions object.
   */
  public static native MarkerManagerOptions newInstance() /*-{
    return {
      trackMarkers: false
    }
  }-*/;

  /**
   * Constructs a MarkerManager options object.
   * 
   * @param borderPadding The extra padding outside the current viewport.
   * @param maxZoom The maximum zoom level.
   * @param trackMarkers Whether to track markers' movements.
   * @return A MarkerManagerOptions object.
   */
  public static native MarkerManagerOptions newInstance(int borderPadding,
      int maxZoom, boolean trackMarkers) /*-{
    return {
      borderPadding: borderPadding,
      maxZoom: maxZoom,
      trackMarkers: trackMarkers
    }
  }-*/;
  
  protected MarkerManagerOptions() { }
  
  /**
   * Sets, in pixels, the extra padding outside the map's current viewport
   * monitored by a manager. Markers that fall within this padding are added
   * to the map, even if they are not fully visible.
   * 
   * @return The extra padding outside the current viewport.
   */
  public final native int getBorderPadding() /*-{
    return this.borderPadding || 0;
  }-*/;
  
  /**
   * Gets the maximum zoom level monitored by a marker manager. If not given,
   * the manager assumes the maximum map zoom level. This value is also used
   * when markers are added to the manager without the optional maxZoom
   * parameter.
   * 
   * @return The maximum zoom level.
   */
  public final native int getMaxZoom() /*-{
    return this.maxZoom || 0;
  }-*/;
  
  /**
   * Retrieves whether or not a marker manager should track markers' movements.
   * If you wish to move managed markers using the setPoint/setLatLng methods,
   * this option should be set to true. The default value is false.
   * 
   * @return Whether to track markers' movements.
   */
  public final native boolean getTrackMarkers() /*-{
    return this.trackMarkers;
  }-*/;

  /**
   * Gets, in pixels, the extra padding outside the map's current viewport
   * monitored by a manager. Markers that fall within this padding are added
   * to the map, even if they are not fully visible.
   * 
   * @param borderPadding The extra padding outside the current viewport.
   */
  public final native void setBorderPadding(int borderPadding) /*-{
    this.borderPadding = borderPadding;
  }-*/;
  
  /**
   * Sets the maximum zoom level monitored by a marker manager. If not given,
   * the manager assumes the maximum map zoom level. This value is also used
   * when markers are added to the manager without the optional maxZoom
   * parameter.
   * 
   * @param maxZoom The maximum zoom level.
   */
  public final native void setMaxZoom(int maxZoom) /*-{
    this.maxZoom = maxZoom;
  }-*/;

  /**
   * Sets whether or not a marker manager should track markers' movements.
   * If you wish to move managed markers using the setPoint/setLatLng methods,
   * this option should be set to true. The default value is false.
   * 
   * @param trackMarkers Whether to track markers' movements.
   */
  public final native void setTrackMarkers(boolean trackMarkers) /*-{
    this.trackMarkers = trackMarkers;
  }-*/;

}