/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.google.gwt.maps.utility.client.markermanager;

import com.google.gwt.core.client.JavaScriptObject;
import com.google.gwt.maps.client.MapWidget;
import com.google.gwt.maps.client.overlay.Marker;

/**
 * Marker manager is an interface between the map and the user, designed to
 * manage adding and removing many points when the viewport changes. 
 */
public class MarkerManager extends JavaScriptObject {

  /**
   * Constructs a new MarkerManager that will show/hide markers on a map.
   * 
   * @param map The map widget to manage.
   * @return A MarkerManager object.
   */
  public static native MarkerManager newInstance(MapWidget map) /*-{
    return new $wnd.MarkerManager(
      map.@com.google.gwt.maps.client.MapWidget::getPeer()()
    );
  }-*/;

  /**
   * Constructs a new MarkerManager that will show/hide markers on a map.
   * 
   * @param map The map widget to manage.
   * @param options The marker manager options.
   * @return A MarkerManager object.
   */
  public static native MarkerManager newInstance(MapWidget map,
      MarkerManagerOptions options) /*-{
    return new $wnd.MarkerManager(
      map.@com.google.gwt.maps.client.MapWidget::getPeer()(),
      options
    );
  }-*/;

  protected MarkerManager() { }

  /**
   * Adds a single marker to the map.
   * 
   * @param marker The marker to add.
   * @param minZoom The minimum zoom to which this marker applies.
   * @param maxZoom The maximum zoom to which this marker applies.
   */
  public final native void addMarker(Marker marker, int minZoom, int maxZoom) /*-{
    this.addMarker(
      @com.google.gwt.maps.utility.client.impl.MapsHelper::getOverlayPeer(Lcom/google/gwt/maps/client/overlay/Overlay;)(marker),
      minZoom, maxZoom
    );
  }-*/;
  
  /**
   * Adds many markers at once. Does not actually update the map, just the
   * internal grid.
   * 
   * @param markers The markers to add.
   * @param minZoom The minimum zoom to which this marker applies.
   */
  public final native void addMarkers(Marker[] markers, int minZoom) /*-{
    this.addMarkers(
      @com.google.gwt.maps.utility.client.impl.MapsHelper::getOverlayPeers([Lcom/google/gwt/maps/client/overlay/Overlay;)(markers),
      minZoom);
  }-*/;
  
  /**
   * Adds many markers at once. Does not actually update the map, just the
   * internal grid.
   * 
   * @param markers The markers to add.
   * @param minZoom The minimum zoom to which this marker applies.
   * @param maxZoom The maximum zoom to which this marker applies.
   */
  public final native void addMarkers(Marker[] markers, int minZoom, int maxZoom) /*-{
    this.addMarkers(
      @com.google.gwt.maps.utility.client.impl.MapsHelper::getOverlayPeers([Lcom/google/gwt/maps/client/overlay/Overlay;)(markers),
      minZoom, maxZoom);
  }-*/;
  
  /**
   * Removes all markers in the manager, and removes any visible markers from
   * the map.
   */
  public final native void clearMarkers() /*-{
    this.clearMarkers();
  }-*/;
  
  /**
   * Returns a marker given latitude, longitude and zoom. If the marker does
   * not exist, the method will return a new marker. If a new marker is
   * created, it will NOT be added to the manager.
   * 
   * @param lat The latitude of the marker to be retrieved.
   * @param lng The longitude of the marker to be retrieved.
   * @param zoom The zoom of the marker to be retrieved.
   * @return The marker corresponding to the specified latitude, longitude and
   * zoom, or a new marker if none is found.
   */
  public final native Marker getMarker(double lat, double lng, int zoom) /*-{
    var m = this.getMarker(lat, lng, zoom);
    return @com.google.gwt.maps.client.overlay.Marker::createPeer(Lcom/google/gwt/core/client/JavaScriptObject;)(m);
  }-*/;
  
  /**
   * Calculates the total number of markers potentially visible at a given zoom
   * level.
   * 
   * @param zoom The zoom level.
   * @return The number of markers at the specified zoom level.
   */
  public final native int getMarkerCount(int zoom) /*-{
    return this.getMarkerCount(zoom) || 0;
  }-*/;
  
  /**
   * Hides the manager if it's currently visible.
   */
  public final native void hide() /*-{
    this.hide();
  }-*/;
  
  /**
   * Retrieves whether the manager is hidden.
   * 
   * @return True if the manager is hidden, false otherwise.
   */
  public final native boolean isHidden() /*-{
    return this.isHidden();
  }-*/;
  
  /**
   * Retrieves the visibility setting of this layer.
   * 
   * @return True if this layer is hidden, false otherwise.
   */
  public final native boolean isVisible() /*-{
    return this.visible();
  }-*/;
  
  /**
   * Forces the marker-manager into a good state.
   * 1. If never before initialized, shows all the markers.
   * 2. If previously initialized, removes and re-adds all markers.
   */
  public final native void refresh() /*-{
    this.refresh();
  }-*/;
  
  /**
   * Removes a marker from the manager and from the map
   * (if it's currently visible).
   * 
   * @param marker The marker to remove.
   */
  public final native void removeMarker(Marker marker) /*-{
    this.removeMarker(
      @com.google.gwt.maps.utility.client.impl.MapsHelper::getOverlayPeer(Lcom/google/gwt/maps/client/overlay/Overlay;)(marker)
    );
  }-*/;
  
  /**
   * Shows the manager if it's currently hidden.
   */
  public final native void show() /*-{
    this.show();
  }-*/;
  
  /**
   * Toggles the visibility of the manager.
   */
  public final native void toggle() /*-{
    this.toggle();
  }-*/;
}