/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.google.gwt.maps.utility.client.markerclusterer;

import com.google.gwt.core.client.JavaScriptObject;
import com.google.gwt.maps.client.geom.Size;

/**
 * Defines styles arguments to MarkerClusterer.
 */
public class MarkerClustererStyleOptions extends JavaScriptObject {

  /**
   * Constructs a MarkerClusterer style options object.
   * 
   * @return A MarkerStyleOptions object.
   */
  public static native MarkerClustererStyleOptions newInstance() /*-{
    return {
      opt_textColor: 'black'
    };
  }-*/;

  /**
   * Constructs a MarkerClusterer style options object.
   * 
   * @param imageWidth The image width.
   * @param imageHeight The image height.
   * @param imageUrl The image url.
   * @return A MarkerStyleOptions object.
   */
  public static native MarkerClustererStyleOptions newInstance(int imageWidth,
      int imageHeight, String imageUrl) /*-{
    return {
      width: imageWidth,
      height: imageHeight,
      url: imageUrl,
      opt_textColor: 'black'
    };
  }-*/;

  /**
   * Constructs a MarkerClusterer style options object.
   * 
   * @param imageWidth The image width.
   * @param imageHeight The image height.
   * @param imageUrl The image url.
   * @param textAnchor The text anchor.
   * @param textColor The text color.
   * @return A MarkerStyleOptions object.
   */
  public static native MarkerClustererStyleOptions newInstance(int imageWidth, int imageHeight,
      String imageUrl, Size textAnchor, String textColor) /*-{
    return {
      width: imageWidth,
      height: imageHeight,
      url: imageUrl,
      opt_anchor: [textAnchor.width, textAnchor.height],
      opt_textColor: textColor
    };
  }-*/;
  
  protected MarkerClustererStyleOptions() { }

  /**
   * Retrieves the image height.
   * 
   * @return The image height.
   */
  public final native int getImageHeight() /*-{
    return this.height || 0;
  }-*/;

  /**
   * Retrieves the image url.
   * 
   * @return The image url.
   */
  public final native String getImageUrl() /*-{
    return this.url;
  }-*/;

  /**
   * Retrieves the image width.
   * 
   * @return The image width.
   */
  public final native int getImageWidth() /*-{
    return this.width || 0;
  }-*/;

  /**
   * Retrieves the anchor for label text.
   * If not set, the text will align center and middle.
   * 
   * @return The text anchor.
   */
  public final native Size getTextAnchor() /*-{
    if(this.opt_anchor) return new $wnd.GSize(this.opt_anchor[0], this.opt_anchor[1]);
    return null;
  }-*/;

  /**
   * Retrieves the text color. The default value is "black".
   * 
   * @return The text color.
   */
  public final native String getTextColor() /*-{
    return this.opt_textColor;
  }-*/;

  /**
   * Sets the image height.
   * 
   * @param imageHeight The image height.
   */
  public final native void setImageHeight(int imageHeight) /*-{
    this.height = imageHeight;
  }-*/;

  /**
   * Sets the image url.
   * 
   * @param imageUrl The image url.
   */
  public final native void setImageUrl(String imageUrl) /*-{
    this.url = imageUrl;
  }-*/;

  /**
   * Sets the image width.
   * 
   * @param imageWidth The image width.
   */
  public final native void setImageWidth(int imageWidth) /*-{
    this.width = imageWidth;
  }-*/;

  /**
   * Sets the anchor for label text.
   * If not set, the text will align center and middle.
   * 
   * @param textAnchor The text anchor.
   */
  public final native void setTextAnchor(Size textAnchor) /*-{
    if (textAnchor) this.opt_anchor = [textAnchor.width, textAnchor.height];
    else this.opt_anchor = null;
  }-*/;

  /**
   * Sets the text color. The default value is "black".
   * 
   * @param textColor The text color.
   */
  public final native void setTextColor(String textColor) /*-{
    this.opt_textColor = textColor;
  }-*/;
  
}