/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.google.gwt.maps.utility.client.markerclusterer;

import com.google.gwt.core.client.JavaScriptObject;

/**
 * Defines optional arguments to MarkerClusterer.
 */
public class MarkerClustererOptions extends JavaScriptObject {
  
  /**
   * Constructs a MarkerClusterer options object.
   * 
   * @return A MarkerClustererOptions object.
   */
  public static native MarkerClustererOptions newInstance() /*-{
    return {
      gridSize: 60
    };
  }-*/;
  
  /**
   * Constructs a MarkerClusterer options object.
   * 
   * @param gridSize The grid size of a cluster in pixel.
   * @param maxZoom The max zoom level monitored by a marker cluster.
   * @param styles The custom styles.
   * @return A MarkerClustererOptions object.
   */
  public static MarkerClustererOptions newInstance(int gridSize, int maxZoom,
      MarkerClustererStyleOptions[] styles) {
    MarkerClustererOptions opts = MarkerClustererOptions.newInstance();
    opts.setGridSize(gridSize);
    opts.setMaxZoom(maxZoom);
    opts.setStyles(styles);
    return opts;
  }
  
  protected MarkerClustererOptions() { }

  /**
   * Retrieves the grid size of a cluster in pixels. Each cluster will be a
   * square. If you want the algorithm to run faster, you can set this value
   * larger. The default value is 60.
   * 
   * @return The grid size.
   */
  public final native int getGridSize() /*-{
    return this.gridSize || 0;
  }-*/;

  /**
   * Retrieves the max zoom level monitored by a marker cluster. If not given,
   * the marker cluster assumes the maximum map zoom level. When maxZoom is
   * reached or exceeded all markers will be shown without cluster.
   * 
   * @return The max zoom.
   */
  public final native int getMaxZoom() /*-{
    return this.maxZoom || 0;
  }-*/;

  /**
   * Retrieves the custom styles for the cluster markers. The array should be
   * ordered according to increasing cluster size, with the style for the
   * smallest clusters first, and the style for the largest clusters last.
   * 
   * @return The custom styles.
   */
  public final native MarkerClustererStyleOptions[] getStyles() /*-{
    if (!this.styles) return null;
    return @com.google.gwt.maps.utility.client.impl.ArrayHelper::toArray(Lcom/google/gwt/core/client/JsArray;)(this.styles);
  }-*/;

  /**
   * Sets the grid size of a cluster in pixels. Each cluster will be a
   * square. If you want the algorithm to run faster, you can set this value
   * larger. The default value is 60.
   * 
   * @param gridSize The grid size.
   */
  public final native void setGridSize(int gridSize) /*-{
    this.gridSize = gridSize;
  }-*/;

  /**
   * Sets the max zoom level monitored by a marker cluster. If not given,
   * the marker cluster assumes the maximum map zoom level. When maxZoom is
   * reached or exceeded all markers will be shown without cluster.
   * 
   * @param maxZoom The max zoom.
   */
  public final native void setMaxZoom(int maxZoom) /*-{
    this.maxZoom = maxZoom;
  }-*/;
  
  /**
   * Sets the custom styles for the cluster markers. The array should be
   * ordered according to increasing cluster size, with the style for the
   * smallest clusters first, and the style for the largest clusters last.
   * 
   * @param styles The custom styles.
   */
  public final native void setStyles(MarkerClustererStyleOptions[] styles) /*-{
    if (!styles) this.styles = null;
    else this.styles = @com.google.gwt.maps.utility.client.impl.ArrayHelper::fromArray([Lcom/google/gwt/core/client/JavaScriptObject;)(styles);
  }-*/;
  
}