/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.google.gwt.maps.utility.client.markerclusterer;

import com.google.gwt.core.client.JavaScriptObject;
import com.google.gwt.maps.client.MapWidget;
import com.google.gwt.maps.client.overlay.Marker;

/**
 * Marker clusterer creates and manages per-zoom-level clusters for large
 * amounts of markers (hundreds or thousands).
 */
public class MarkerClusterer extends JavaScriptObject {

  /**
   * Constructs a new MarkerClusterer to cluster markers on the map.
   * 
   * @param map The map widget to manage.
   * @return A MarkerClusterer object.
   */
  public static native MarkerClusterer newInstance(MapWidget map) /*-{
    return new $wnd.MarkerClusterer(
      map.@com.google.gwt.maps.client.MapWidget::getPeer()()
    );
  }-*/;
  
  /**
   * Constructs a new MarkerClusterer to cluster markers on the map.
   * 
   * @param map The map widget to manage.
   * @param markers The markers to add.
   * @return A MarkerClusterer object.
   */
  public static native MarkerClusterer newInstance(MapWidget map,
      Marker[] markers) /*-{
    return new $wnd.MarkerClusterer(
      map.@com.google.gwt.maps.client.MapWidget::getPeer()(),
      @com.google.gwt.maps.utility.client.impl.MapsHelper::getOverlayPeers([Lcom/google/gwt/maps/client/overlay/Overlay;)(markers)
    );
  }-*/;
  
  /**
   * Constructs a new MarkerClusterer to cluster markers on the map.
   * 
   * @param map The map widget to manage.
   * @param markers The markers to add.
   * @param options The marker manager options
   * @return A MarkerClusterer object.
   */
  public static native MarkerClusterer newInstance(MapWidget map,
      Marker[] markers, MarkerClustererOptions options) /*-{
    return new $wnd.MarkerClusterer(
      map.@com.google.gwt.maps.client.MapWidget::getPeer()(),
      @com.google.gwt.maps.utility.client.impl.MapsHelper::getOverlayPeers([Lcom/google/gwt/maps/client/overlay/Overlay;)(markers),
      options
    );
  }-*/;

  protected MarkerClusterer() { }

  /**
   * Adds a set of markers.
   * 
   * @param markers The markers to add.
   */
  public final native void addMarkers(Marker[] markers) /*-{
    this.addMarkers(
      @com.google.gwt.maps.utility.client.impl.MapsHelper::getOverlayPeers([Lcom/google/gwt/maps/client/overlay/Overlay;)(markers)
    );
  }-*/;
  
  /**
   * Removes all markers from MarkerClusterer.
   */
  public final native void clearMarkers() /*-{
    this.clearMarkers();
  }-*/;
  
  /**
   * Retrieves the total number of clusters.
   * 
   * @return The total number of clusters.
   */
  public final native int getTotalClusters() /*-{
    return this.getTotalClusters();
  }-*/;
  
  /**
   * Retrieves the total number of markers.
   * 
   * @return The total number of markers.
   */
  public final native int getTotalMarkers() /*-{
    return this.getTotalMarkers();
  }-*/;
  
  /**
   * Removes a marker.
   * 
   * @param marker The marker to remove.
   */
  public final native void removeMarker(Marker marker) /*-{
    this.removeMarker(
      @com.google.gwt.maps.utility.client.impl.MapsHelper::getOverlayPeer(Lcom/google/gwt/maps/client/overlay/Overlay;)(marker)
    );
  }-*/;
  
  /**
   * Collects all markers of clusters in viewport and regroup them.
   */
  public final native void resetViewport() /*-{
    this.resetViewport();
  }-*/;
  
}