/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.google.gwt.maps.utility.client.labeledmarker;

import com.google.gwt.maps.client.geom.Size;
import com.google.gwt.maps.client.overlay.Icon;
import com.google.gwt.maps.client.overlay.MarkerOptions;

/**
 * Defines optional arguments to LabeledMarker.
 */
public class LabeledMarkerOptions extends MarkerOptions {
  
  /**
   * Constructs a LabeledMarker options object.
   * 
   * @return A LabeledMarkerOptions object.
   */
  public static native LabeledMarkerOptions newInstance() /*-{
    return {
      labelClass: "LabeledMarker_markerLabel",
      labelOffset: new $wnd.GSize(0, 0),
      labelText: ""
    };
  }-*/;
  
  /**
   * Constructs a LabeledMarker options object.
   * 
   * @param icon The marker icon.
   * @return A LabeledMarkerOptions object.
   */
  public static LabeledMarkerOptions newInstance(Icon icon) {
    LabeledMarkerOptions opts = LabeledMarkerOptions.newInstance();
    opts.setIcon(icon);
    return opts;
  }
  
  /**
   * Constructs a LabeledMarker options object.
   * 
   * @param labelClass The CSS class name.
   * @param labelOffset The label offset.
   * @param labelText The label text/html.
   * @return A LabeledMarkerOptions object.
   */
  public static native LabeledMarkerOptions newInstance(String labelClass,
      Size labelOffset, String labelText) /*-{
    return {
      labelClass: labelClass,
      labelOffset: labelOffset,
      labelText: labelText
    };
  }-*/;
  
  protected LabeledMarkerOptions() { }
  
  /**
   * Retrieves the CSS class to use for the overlay div.
   * The default value is "LabeledMarker_markerLabel".
   * 
   * @return The CSS class name.
   */
  public final native String getLabelClass() /*-{
    return this.labelClass;
  }-*/;
  
  /**
   * Retrieves the label offset, the x- and y-distance between the marker's
   * latlng and the upper-left corner of the text div (default (0,0)).
   * 
   * @return The label offset.
   */
  public final native Size getLabelOffset() /*-{
    return this.labelOffset;
  }-*/;
  
  /**
   * Retrieves the label text/html to place in the overlay div.
   * The default value is an empty string.
   * 
   * @return The label text/html.
   */
  public final native String getLabelText() /*-{
    return this.labelText;
  }-*/;
  
  /**
   * Sets the CSS class to use for the overlay div.
   * The default value is "LabeledMarker_markerLabel".
   * 
   * @param labelClass The CSS class name.
   */
  public final native void setLabelClass(String labelClass) /*-{
    this.labelClass = labelClass;
  }-*/;
  
  /**
   * Sets the label offset, the x- and y-distance between the marker's
   * latlng and the upper-left corner of the text div (default (0,0)).
   * 
   * @param labelOffset The label offset.
   */
  public final native void setLabelOffset(Size labelOffset) /*-{
    this.labelOffset = labelOffset;
  }-*/;

  /**
   * Sets the label text/html to place in the overlay div.
   * The default value is an empty string.
   * 
   * @param labelText The label text/html.
   */
  public final native void setLabelText(String labelText) /*-{
    this.labelText = labelText;
  }-*/;
  
}