/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.google.gwt.maps.utility.client.impl;

import com.google.gwt.core.client.JavaScriptObject;
import com.google.gwt.maps.client.geom.LatLng;
import com.google.gwt.maps.client.geom.Size;
import com.google.gwt.maps.utility.client.snapshotcontrol.SnapShotControlOptions;

/**
 * The JavaScript implementation of SnapShotControl.
 */
public class SnapShotControlImpl extends JavaScriptObject {

  /**
   * Constructs a SnapShotControl JavaScript object.
   * 
   * @return A SnapShotControl JavaScript object.
   */
  public static native SnapShotControlImpl newInstance() /*-{
    return new $wnd.SnapShotControl();
  }-*/;

  /**
   * Constructs a SnapShotControl JavaScript object with the
   * specified options.
   * 
   * @param options The snapshot control options.
   * @return A SnapShotControl JavaScript object.
   */
  public static native SnapShotControlImpl newInstance(
      SnapShotControlOptions options) /*-{
    return new $wnd.SnapShotControl(options);
  }-*/;
  
  protected SnapShotControlImpl() { }

  /**
   * Generates a new snapshot URL using the current map center.
   * 
   * @return The new snapshot image URL.
   */
  public final native String getImage() /*-{
    return this.getImage();
  }-*/;
  
  /**
   * Generates a new snapshot URL using the specified map center.
   * If the specified map center is null, the center of the snapshot is
   * auto-calculated based on the overlay positions.
   * 
   * @param mapCenterPos The map center to use in the snapshot.
   * @return The new snapshot image URL.
   */
  public final native String getImage(LatLng mapCenterPos) /*-{
    if (mapCenterPos === null) return this.getImage(false);
    else return this.getImage(mapCenterPos);
  }-*/;
  
  /**
   * Retrieves the most recently generated image URL.
   * 
   * @return The most recently generated image URL.
   */
  public final native String getImageUrl() /*-{
    return this.getImageUrl();
  }-*/;
  
  /**
   * Sets the visibility of the control to hidden.
   */
  public final native void hide() /*-{
    this.hide();
  }-*/;
  
  /**
   * Retrieves whether the control is hidden.
   * 
   * @return Whether the control is hidden.
   */
  public final native boolean isHidden() /*-{
    return this.isHidden();
  }-*/;
  
  /**
   * Sets the image format.
   * The options are "gif", "jpg", "jpg-baseline", "png8", "png32".
   * 
   * @param format The image format.
   */
  public final native void setFormat(String format) /*-{
    this.setFormat(format);
  }-*/;
  
  /**
   * Sets the language for the snapshot's copyright text.
   * 
   * @param language The language for the snapshot's copyright text.
   */
  public final native void setLanguage(String language) /*-{
    this.setLanguage(language);
  }-*/;
  
  /**
   * Sets the map size for the snapshot.
   * 
   * @param mapSize The map size for the snapshot.
   */
  public final native void setMapSize(Size mapSize) /*-{
    this.setMapSize(mapSize);
  }-*/;
  
  /**
   * Sets the map type for the snapshot.
   * The options are "roadmap", "satellite", "hybrid", "terrain".
   * If specified type is an empty string, the map type will be
   * auto-computed.
   * 
   * @param mapType The map type for the snapshot.
   */
  public final native void setMapType(String mapType) /*-{
    this.setMapType(mapType);
  }-*/;
  
  /**
   * Sets whether to use mobile-optimized tiles.
   * 
   * @param mobile Whether to use mobile-optimized tiles.
   */
  public final native void setMobile(boolean mobile) /*-{
    this.isMobile(mobile);
  }-*/;

  /**
   * Sets the visibility of the control to visible.
   */
  public final native void show() /*-{
    this.show();
  }-*/;
  
  /**
   * Generates a new snapshot URL, at the specified map center, and displays a
   * popup containing the image, along with the URL.
   */
  public final native void showPopup() /*-{
    this.showPopup();
  }-*/;
  
  /**
   * Generates a new snapshot URL, at the specified map center, and displays a
   * popup containing the image, along with the URL.
   * 
   * @param mapCenterPos The map center to use for the snapshot.
   */
  public final native void showPopup(LatLng mapCenterPos) /*-{
    this.showPopup(mapCenterPos);
  }-*/;
  
  /**
   * Sets whether to use polyline encoding.
   * 
   * @param useEncode Whether to use polyline encoding.
   */
  public final native void setUsePolylineEncode(boolean useEncode) /*-{
    this.usePolylineEncode(useEncode);
  }-*/;
}
