/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.google.gwt.maps.utility.client;

import com.google.gwt.dom.client.Element;
import com.google.gwt.user.client.DOM;
import com.google.gwt.user.client.Timer;
import com.google.gwt.user.client.ui.RootPanel;

/**
 * The Google Maps Utility module.
 */
public class GoogleMapsUtility {
  
  static String apiVersion = "1.0";
  
  /**
   * Retrieves the Google Maps Utility version.
   * 
   * @return The Google Maps Utility version.
   */
  public static String getVersion() {
    return apiVersion;
  }
  
  /**
   * Determines if the API is loaded.
   * 
   * @param packages The Utility packages to check for.
   * @return Whether the API packages are loaded and available.
   */
  public static boolean isLoaded(GoogleMapsUtilityPackage... packages) {
    boolean loaded = true;
    for (GoogleMapsUtilityPackage pack : packages) {
      if (!isPackageLoaded(pack.getIndicator())) {
        loaded = false;
        break;
      }
    }
    return loaded;
  }
  
  /**
   * Loads the API with a default polling interval of 100 milliseconds.
   * 
   * @param onLoad Callback to be invoked when the library is loaded.
   * @param packages The Google Maps Utility packages that should be loaded.
   */
  public static void loadUtilityApi(Runnable onLoad,
      GoogleMapsUtilityPackage... packages) {
    loadUtilityApi(onLoad, 100, packages);
  }

  /**
   * Loads the API.
   * 
   * @param onLoad Callback to be invoked when the library is loaded.
   * @param pollingInterval The time interval, in milliseconds, to use when
   * checking for the availability of the API.
   * @param packages The Google Maps Utility packages that should be loaded.
   */
  public static void loadUtilityApi(final Runnable onLoad, int pollingInterval,
      final GoogleMapsUtilityPackage... packages) {
    for (GoogleMapsUtilityPackage pack : packages) {
      loadPackage(pack.getSource());
    }
    Timer timer = new Timer() {
      public void run() {
        if (isLoaded(packages)) {
          this.cancel();
          onLoad.run();
        }
      }
    };
    timer.scheduleRepeating(pollingInterval);
  }
  
  /**
   * Determines whether a given package is loaded by using the
   * package's indicator.
   * 
   * @param indicator A top-level object that indicates the presence of the
   * package.
   * @return Whether the specified package is loaded amd available.
   */
  private static native boolean isPackageLoaded(String indicator) /*-{
    return $wnd[indicator] !== undefined;
  }-*/;
  
  /**
   * Loads a package by inserting the corresponding script reference.
   * 
   * @param src The script source for the package to be loaded.
   */
  private static void loadPackage(String src) {
    Element e = DOM.createElement("script");
    e.setAttribute("language", "JavaScript");
    e.setAttribute("src", src);
    RootPanel.get().getElement().appendChild(e);
  }
}
