/*_############################################################################
  _## 
  _##  SNMP4J-AgentX - AgentXSharedMOTableSupport.java  
  _## 
  _##  Copyright (C) 2005-2009  Frank Fock (SNMP4J.org)
  _##  
  _##  This program is free software; you can redistribute it and/or modify
  _##  it under the terms of the GNU General Public License version 2 as 
  _##  published by the Free Software Foundation.
  _##
  _##  This program is distributed in the hope that it will be useful,
  _##  but WITHOUT ANY WARRANTY; without even the implied warranty of
  _##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  _##  GNU General Public License for more details.
  _##
  _##  You should have received a copy of the GNU General Public License
  _##  along with this program; if not, write to the Free Software
  _##  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
  _##  MA  02110-1301  USA
  _##  
  _##########################################################################*/

package org.snmp4j.agent.agentx.subagent;

import java.io.IOException;

import org.snmp4j.agent.agentx.*;
import org.snmp4j.agent.mo.*;
import org.snmp4j.log.LogAdapter;
import org.snmp4j.log.LogFactory;
import org.snmp4j.smi.*;
import org.snmp4j.agent.agentx.subagent.index.AnyNewIndexOID;
import org.snmp4j.agent.agentx.subagent.index.NewIndexOID;

/**
 * The <code>AgentXSharedMOTableSupport</code> provides helper functions for
 * shared table implementations to register rows and indexes at a master agent.
 *
 * @author Frank Fock
 * @version 1.0
 */
public class AgentXSharedMOTableSupport implements MOTableRowListener {

  private static final LogAdapter LOGGER =
      LogFactory.getLogger(AgentXSharedMOTableSupport.class);

  public static final int INDEX_MODE_ALLOCATE = AgentXProtocol.FLAG_ALLOCATE_INDEX;
  public static final int INDEX_MODE_ANY_INDEX = AgentXProtocol.FLAG_ANY_INDEX;
  public static final int INDEX_MODE_NEW_INDEX = AgentXProtocol.FLAG_NEW_INDEX;

  private AgentX agentX;
  private AgentXSession session;
  private OctetString context;
  private byte priority = AgentXProtocol.DEFAULT_PRIORITY;
  private byte indexMode = INDEX_MODE_ALLOCATE;

  /**
   * Creates a shared table support object for a AgentX connection, session,
   * and context.
   *
   * @param agentX
   *    an AgentX connection.
   * @param session
   *    an AgentXSession session (does not need to be open at creation time).
   * @param context
   *    a context ("" by default).
   */
  public AgentXSharedMOTableSupport(AgentX agentX, AgentXSession session,
                                    OctetString context) {
    this.agentX = agentX;
    this.session = session;
    this.context = context;
  }

  /**
   * Creates a shared table support object for a AgentX connection, session,
   * and context.
   *
   * @param agentX
   *    an AgentX connection.
   * @param session
   *    an AgentXSession session (does not need to be open at creation time).
   * @param context
   *    a context ("" by default).
   * @param priority
   *    the registration priority used for this shared table support.
   * @param indexAllocationMode
   *    the index allocation mode to be used as default for this shared table.
   */
  public AgentXSharedMOTableSupport(AgentX agentX, AgentXSession session,
                                    OctetString context,
                                    byte priority, byte indexAllocationMode) {
    this(agentX, session, context);
    this.priority = priority;
    this.indexMode = indexAllocationMode;
  }

  /**
   * Process shared table row events. If index mode is
   * {@link #INDEX_MODE_ALLOCATE} this method will do nothing if the associated
   * AgentX session is closed. For other index modes, the event's veto status
   * will be set to the AgentX error {@link AgentXProtocol#AGENTX_NOT_OPEN}.
   * <p>
   * If the index OID of a created row has zero length then, depending on the
   * current index mode, a new or any new index is allocated at the master
   * agent.
   *
   * @param event
   *    a <code>MOTableRowEvent</code> indicating a row change in an AgentX
   *    shared table.
   */
  public void rowChanged(MOTableRowEvent event) {
    if ((indexMode == INDEX_MODE_ALLOCATE) && getSession().isClosed()) {
      // ignore closed session for allocation mode
      if (LOGGER.isInfoEnabled()) {
        LOGGER.info("Row event " + event +
                    " ignored, because session to master agent is closed: " +
                    getSession());
      }
      return;
    }
    switch (event.getType()) {
      case MOTableRowEvent.CREATE: {
        OID index2Allocate = event.getRow().getIndex();
        int status =
            allocateIndex(context, event.getTable().getIndexDef(),
                          (event.getRow().getIndex().size() == 0) ?
                          indexMode : 0,
                          index2Allocate);
        if (status != AgentXProtocol.AGENTX_SUCCESS) {
          event.setVetoStatus(status);
        }
        break;
      }
      case MOTableRowEvent.ADD: {
        int status = registerRow(event.getTable(), event.getRow());
        if (status != AgentXProtocol.AGENTX_SUCCESS) {
          event.setVetoStatus(status);
        }
        break;
      }
      case MOTableRowEvent.DELETE: {
        int status = unregisterRow(event.getTable(), event.getRow());
        if ((status != AgentXProtocol.AGENTX_SUCCESS) &&
            (status != AgentXProtocol.AGENTX_UNKNOWN_REGISTRATION)) {
          event.setVetoStatus(status);
        }
        else {
          OID index2Deallocate = event.getRow().getIndex();
          status = deallocateIndex(context, event.getTable().getIndexDef(),
                                   index2Deallocate);
        }
        break;
      }
      default: {
        if (LOGGER.isDebugEnabled()) {
          LOGGER.debug("Ignored AgentX shared table event "+event);
        }
      }
    }
  }

  /**
   * Allocate a new or any index at the master agent and return its value in
   * <code>allocateIndex</code>.
   *
   * @param context
   *    the context for which to allocate the index. Specify an empty
   *    <code>OctetString</code> for the default context.
   * @param indexDef
   *    the index definition with OID values for sub-index definitions.
   * @param indexAllocationMode
   *    one of {@link AgentXProtocol#FLAG_ANY_INDEX},
   *    {@link AgentXProtocol#FLAG_NEW_INDEX}, or 0 (if index value is supplied
   *    by <code>allocateIndex</code>).
   * @param allocatedIndex
   *    the index value to allocate or if <code>indexAllocationMode</code> is
   *    not zero then an (arbitrary non-null OID) which returns the allocated
   *    new index value. If <code>allocateIndex</code> is an instance of
   *    {@link AnyNewIndexOID} or {@link NewIndexOID} the index value of the
   *    row will be replaced by a globally unique index value allocated by the
   *    master agent. The caller is responsible for changing the rows index
   *    in the table model of the shared table.
   * @return
   *    {@link AgentXProtocol#AGENTX_SUCCESS} if the index could be allocated
   *     or an AgentX protocol error code if allocation failed and
   *    <code>allocateIndex</code> is not altered.
   */
  public int allocateIndex(OctetString context,
                           MOTableIndex indexDef,
                           byte indexAllocationMode,
                           OID allocatedIndex) {
    VariableBinding[] vbs = new VariableBinding[indexDef.size()];
    Variable[] indexValues;
    if (allocatedIndex instanceof AnyNewIndexOID) {
      indexAllocationMode = INDEX_MODE_ANY_INDEX;
    }
    else if (allocatedIndex instanceof NewIndexOID) {
      indexAllocationMode = INDEX_MODE_NEW_INDEX;
    }
    if (indexAllocationMode == 0) {
      indexValues = indexDef.getIndexValues(allocatedIndex);
    }
    else {
      indexValues = new Variable[indexDef.size()];
      for (int i=0; i<indexDef.size(); i++) {
        MOTableSubIndex subIndex = indexDef.getIndex(i);
        indexValues[i] =
            AbstractVariable.createFromSyntax(subIndex.getSmiSyntax());
      }
    }
    for (int i=0; i<indexDef.size(); i++) {
      MOTableSubIndex subIndex = indexDef.getIndex(i);
      OID oid = subIndex.getOid();
      if (oid == null) {
        throw new IllegalArgumentException("Sub-index "+i+" has no OID");
      }
      vbs[i] = new VariableBinding();
      vbs[i].setOid(oid);
      vbs[i].setVariable(indexValues[i]);
    }
    AgentXIndexAllocatePDU pdu = new AgentXIndexAllocatePDU(context, vbs);
    if (indexAllocationMode != 0) {
      pdu.addFlag(indexAllocationMode);
    }
    pdu.setSessionAttributes(session);
    try {
      AgentXResponseEvent response =
          agentX.send(pdu, session.createAgentXTarget(),
                      session.getPeer().getTransport());
      if (response.getResponse() != null) {
        AgentXResponsePDU resp = response.getResponse();
        if (resp.getErrorStatus() == AgentXProtocol.AGENTX_SUCCESS) {
          OID index =
              indexDef.getIndexOID(getVariables(resp.getVariableBindings()));
          allocatedIndex.setValue(index.getValue());
          if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Allocated index "+allocatedIndex+" for context "+
                         context+" and index definition "+indexDef);
          }
        }
        else {
          if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Index allocation failed for context "+
                         context+" and index definition "+indexDef+
                         " with value "+allocatedIndex);
          }
        }
        return resp.getErrorStatus();
      }
      else {
        return AgentXProtocol.AGENTX_TIMEOUT;
      }
    }
    catch (IOException ex) {
      LOGGER.error("Failed to allocate index "+indexDef+" at "+session, ex);
    }
    return AgentXProtocol.AGENTX_DISCONNECT;
  }

  /**
   * Deallocate an index at the master agent.
   *
   * @param context
   *    the context for which to allocate the index. Specify an empty
   *    <code>OctetString</code> for the default context.
   * @param indexDef
   *    the index definition with OID values for sub-index definitions.
   * @param allocatedIndex
   *    the index value of the previously allocated index.
   * @return
   *    {@link AgentXProtocol#AGENTX_SUCCESS} if the index could be deallocated
   *    or an AgentX protocol error code if deallocation failed.
   */
  public int deallocateIndex(OctetString context,
                             MOTableIndex indexDef,
                             OID allocatedIndex) {
    VariableBinding[] vbs = new VariableBinding[indexDef.size()];
    Variable[] indexValues = indexDef.getIndexValues(allocatedIndex);
    for (int i=0; i<indexDef.size(); i++) {
      vbs[i] = new VariableBinding();
      MOTableSubIndex subIndex = indexDef.getIndex(i);
      OID oid = subIndex.getOid();
      if (oid == null) {
        throw new IllegalArgumentException("Sub-index "+i+" has no OID");
      }
      vbs[i].setOid(oid);
      vbs[i].setVariable(indexValues[i]);
    }
    AgentXIndexDeallocatePDU pdu = new AgentXIndexDeallocatePDU(context, vbs);
    pdu.setSessionAttributes(session);
    try {
      AgentXResponseEvent response =
          agentX.send(pdu, session.createAgentXTarget(),
                      session.getPeer().getTransport());
      if (response.getResponse() != null) {
        AgentXResponsePDU resp = response.getResponse();
        return resp.getErrorStatus();
      }
      else {
        return AgentXProtocol.AGENTX_TIMEOUT;
      }
    }
    catch (IOException ex) {
      LOGGER.error("Failed to deallocate index "+indexDef+" at "+session, ex);
    }
    return AgentXProtocol.AGENTX_DISCONNECT;
  }


  public int registerRow(MOTable table, MOTableRow row2Register) {
    OID subtree = new OID(table.getOID());
    subtree.append(table.getColumn(0).getColumnID());
    subtree.append(row2Register.getIndex());
    AgentXRegisterPDU pdu =
        new AgentXRegisterPDU(context, subtree, priority,
                              (byte)(table.getOID().size()+1),
                              table.getColumn(table.getColumnCount()-1).
                              getColumnID());
    if (table.getColumnCount() == 1) {
      pdu.addFlag(AgentXProtocol.FLAG_INSTANCE_REGISTRATION);
    }
    pdu.setSessionAttributes(session);
    try {
      AgentXResponseEvent resp =
          agentX.send(pdu, session.createAgentXTarget(),
                      session.getPeer().getTransport());
      if (resp.getResponse() == null) {
        return AgentXProtocol.AGENTX_TIMEOUT;
      }
      else if (resp.getResponse().getErrorStatus() !=
               AgentXProtocol.AGENTX_SUCCESS) {
        return resp.getResponse().getErrorStatus();
      }
    }
    catch (IOException ex) {
      LOGGER.error("Failed to send AgentXRegister pdu "+pdu+" to "+session+
                   " because: "+ex.getMessage(), ex);
      return AgentXProtocol.AGENTX_DISCONNECT;
    }
    return AgentXProtocol.AGENTX_SUCCESS;
  }

  public int unregisterRow(MOTable table, MOTableRow row2Unregister) {
    OID subtree = new OID(table.getOID());
    subtree.append(table.getColumn(0).getColumnID());
    subtree.append(row2Unregister.getIndex());
    AgentXUnregisterPDU pdu =
        new AgentXUnregisterPDU(context, subtree, priority,
                                (byte)(table.getOID().size()+1),
                                table.getColumn(table.getColumnCount()-1).
                                getColumnID());
    pdu.setSessionAttributes(session);
    try {
      AgentXResponseEvent resp =
          agentX.send(pdu, session.createAgentXTarget(),
                      session.getPeer().getTransport());
      if (resp.getResponse() == null) {
        return AgentXProtocol.AGENTX_TIMEOUT;
      }
      else if (resp.getResponse().getErrorStatus() !=
               AgentXProtocol.AGENTX_SUCCESS) {
        return resp.getResponse().getErrorStatus();
      }
    }
    catch (IOException ex) {
      LOGGER.error("Failed to send AgentXRegister pdu "+pdu+" to "+session+
                   " because: "+ex.getMessage(), ex);
      return AgentXProtocol.AGENTX_DISCONNECT;
    }
    return AgentXProtocol.AGENTX_SUCCESS;
  }

  private static Variable[] getVariables(VariableBinding[] vbs) {
    Variable[] variables = new Variable[vbs.length];
    for (int i=0; i<vbs.length; i++) {
      variables[i] = vbs[i].getVariable();
    }
    return variables;
  }

  public void setPriority(byte priority) {
    this.priority = priority;
  }

  /**
   * Sets the AgentX session to be used for this shared table support.
   * @param session
   *   an <code>AgentXSession</code> instance.
   */
  public void setSession(AgentXSession session) {
    this.session = session;
  }

  /**
   * Sets the index mode to be used by this shared table support object.
   * {@link #INDEX_MODE_ALLOCATE} simply allocates index values at the master
   * agent, whereas {@link #INDEX_MODE_ANY_INDEX} fetches any currently
   * unique index value from the master agent for a new row and
   * {@link #INDEX_MODE_NEW_INDEX} fetches a new index (never used before by
   * the master).
   * @param indexMode
   *    an index mode to be used for shared tables supported by this object.
   */
  public void setIndexMode(byte indexMode) {
    this.indexMode = indexMode;
  }

  public void setContext(OctetString context) {
    this.context = context;
  }

  public byte getPriority() {
    return priority;
  }

  /**
   * Gets the AgentX session used by this shared table support object.
   * @return
   *   an <code>AgentXSession</code> instance or <code>null</code> if there
   *   is no connection/session established with the master agent.
   */
  public AgentXSession getSession() {
    return session;
  }

  public byte getIndexMode() {
    return indexMode;
  }

  public OctetString getContext() {
    return context;
  }

  public AgentX getAgentX() {
    return agentX;
  }

}
